/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;

/**
 * A Balloon is an Area containing a Text and a Polygon.
 * There are no ports.  The balloon "points" to another JGoObject,
 * given by the Anchor property.  This object should implement the
 * Partner property so that geometry changes to the anchor will
 * cause this balloon's polygon to be reshaped to point to the
 * anchor's new position.  You can implement the Partner property
 * by adding the following lines to your node or link class:
 * <pre>
 *   private JGoObject myPartner = null;
 *   public JGoObject getPartner() { return myPartner; }
 *   public void setPartner(JGoObject obj) { myPartner = obj; }
 *   public JGoObject copyObject(JGoCopyEnvironment env) {
 *     JGoObject copy = super.copyObject(env);
 *     if (myPartner != null) env.delay(this);
 *     return copy;
 *   }
 *   public void copyObjectDelayed(JGoCopyEnvironment env, JGoObject newobj) {
 *     super.copyObjectDelayed(env, newobj);
 *     YOUROBJECTCLASS n = (YOUROBJECTCLASS)newobj;
 *     n.myPartner = (JGoObject)env.get(myPartner);
 *   }
 * </pre>
 * <p>
 * If the text is changed, the polygon is automatically resized to
 * "hold" the text.
 * <p>
 * By default the text is not editable by the user, but it can be
 * made editable by calling setEditable(true).
 */
public class Balloon extends JGoArea implements JGoIdentifiablePart
{
  /** Create an empty Balloon.  Call initialize(String) before using it. */
  public Balloon()
  {
    super();
  }

  /** Create a Balloon displaying the given String. */
  public Balloon(String s)
  {
    super();
    initialize(s);
  }

  public void initialize(String s)
  {
    setResizable(false);

    myPoly = new JGoPolygon();
    myPoly.setSelectable(false);
    myPoly.setPen(JGoPen.lightGray);
    myPoly.setBrush(JGoBrush.makeStockBrush(new Color(0xFF, 0xFF, 0xCC)));

    myLabel = new JGoText(s);
    myLabel.setMultiline(true);
    myLabel.setSelectable(false);
    myLabel.setResizable(false);
    myLabel.setDraggable(false);
    myLabel.setEditable(false);
    myLabel.setEditOnSingleClick(true);  // in case it becomes editable
    myLabel.setTransparent(true);

    addObjectAtHead(myPoly);
    addObjectAtTail(myLabel);
  }

  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    Balloon newobj = (Balloon)newarea;
    env.delay(this);
    newobj.myPartID = myPartID;

    super.copyChildren(newarea, env);

    newobj.myPoly = (JGoPolygon)env.get(myPoly);
    newobj.myLabel = (JGoText)env.get(myLabel);
  }

  public void copyObjectDelayed(JGoCopyEnvironment env, JGoObject newobj) {
    Balloon b = (Balloon)newobj;
    b.myAnchor = (JGoObject)env.get(myAnchor);
    b.updatePolygon();
  }
  
  /**
   * When an object is removed, make sure there are no more references from fields.
   */
  public JGoObject removeObjectAtPos(JGoListPosition pos)
  {
    JGoObject child = super.removeObjectAtPos(pos);
    if (child == myPoly)
      myPoly = null;
    else if (child == myLabel)
      myLabel = null;
    return child;
  }

  protected void moveChildren(Rectangle prevRect) {
    super.moveChildren(prevRect);
    updatePolygon();
  }

  /**
   * Position all the parts of the area relative to the text label.
   * Leave room for the JGoPolygon decorations.
   */
  public void layoutChildren(JGoObject childchanged) {
    if (isInitializing()) return;
    JGoText label = getLabel();
    if (label == null) return;
    JGoPolygon back = getPolygon();
    if (back != null) {
      setInitializing(true);

      Insets insets = getInsets();
      // get the rectangle surrounding the text
      Rectangle rect = new Rectangle(label.getLeft() - insets.left,
                                     label.getTop() - insets.top,
                                     label.getWidth() + insets.left + insets.right,
                                     label.getHeight() + insets.top + insets.bottom);
      Dimension corner = new Dimension(4, 4);
      int cw = corner.width;
      if (cw > rect.width/2)
        cw = rect.width/2;
      int ch = corner.height;
      if (ch > rect.height/2)
        ch = rect.height/2;
      int x = rect.x;
      int y = rect.y;
      int xa = x + cw;
      int ya = y + ch;
      int x2 = x + rect.width/2;
      int y2 = y + rect.height/2;
      int xb = x + rect.width - cw;
      int yb = y + rect.height - ch;
      int xw = x + rect.width;
      int yh = y + rect.height;

      back.foredate(JGoPolygon.ChangedAllPoints);
      // avoid any intermediate update()'s
      back.setSuspendUpdates(true);
      back.removeAllPoints();

      int basewidth = Math.min(rect.width-cw, getBaseWidth());
      int baseheight = Math.min(rect.height-ch, getBaseWidth());

      int labelx = label.getLeft();
      int labely = label.getTop();
      int labelr = labelx+label.getWidth();
      int labelb = labely+label.getHeight();
      Point p1 = label.getSpotLocation(Center);
      Point r2 = computeAnchorPoint();
      Point r1 = new Point(0, 0);
      label.getNearestIntersectionPoint(r2.x, r2.y, p1.x, p1.y, r1);

      if (r1.y <= labely && r1.x < x2) {
        back.addPoint(x, y);
        back.addPoint(r2);
        back.addPoint(x+basewidth, y);
      } else {
        back.addPoint(xa, y);
      }
      if (r1.y <= labely && r1.x >= x2) {
        back.addPoint(xw-basewidth, y);
        back.addPoint(r2);
        back.addPoint(xw, y);
      } else {
        back.addPoint(xb, y);
      }
      if (r1.x >= labelr & r1.y < y2) {
        back.addPoint(xw, y);
        back.addPoint(r2);
        back.addPoint(xw, y+baseheight);
      } else {
        back.addPoint(xw, ya);
      }
      if (r1.x >= labelr & r1.y >= y2) {
        back.addPoint(xw, yh-baseheight);
        back.addPoint(r2);
        back.addPoint(xw, yh);
      } else {
        back.addPoint(xw, yb);
      }
      if (r1.y >= labelb && r1.x >= x2) {
        back.addPoint(xw, yh);
        back.addPoint(r2);
        back.addPoint(xw-basewidth, yh);
      } else {
        back.addPoint(xb, yh);
      }
      if (r1.y >= labelb && r1.x < x2) {
        back.addPoint(x+basewidth, yh);
        back.addPoint(r2);
        back.addPoint(x, yh);
      } else {
        back.addPoint(xa, yh);
      }
      if (r1.x <= labelx && r1.y >= y2) {
        back.addPoint(x, yh);
        back.addPoint(r2);
        back.addPoint(x, yh-baseheight);
      } else {
        back.addPoint(x, yb);
      }
      if (r1.x <= labelx && r1.y < y2) {
        back.addPoint(x, y+baseheight);
        back.addPoint(r2);
        back.addPoint(x, y);
      } else {
        back.addPoint(x, ya);
      }

      back.setSuspendUpdates(false);
      back.update(JGoPolygon.ChangedAllPoints, 0, null);

      setInitializing(false);
    }
  }

  public Point computeAnchorPoint() {
    Point p1;
    if (getLabel() != null)
      p1 = getLabel().getSpotLocation(Center);
    else
      p1 = getSpotLocation(Center);
    Point p2;
    if (getAnchor() != null) {
      p2 = getAnchor().getSpotLocation(Center);
      Point r2 = new Point(0, 0);
      if (getAnchor().getNearestIntersectionPoint(p1.x, p1.y, p2.x, p2.y, r2))
        return r2;
      else
        return p2;
    } else {
      if (getLabel() != null)
        p2 = new Point(getLabel().getLeft()-30, getLabel().getTop()-30);
      else
        p2 = new Point(p1.x-30, p1.y-30);
      return p2;
    }
  }

  public Rectangle computeBoundingRect() {
    JGoText label = getLabel();
    if (label == null) return super.computeBoundingRect();
    Insets insets = getInsets();
    // get the rectangle surrounding the text
    Rectangle rect = new Rectangle(label.getLeft() - insets.left,
                                   label.getTop() - insets.top,
                                   label.getWidth() + insets.left + insets.right,
                                   label.getHeight() + insets.top + insets.bottom);
    return rect;
  }

  public void expandRectByPenWidth(Rectangle rect) {
    JGoObject back = getPolygon();
    if (back != null) {
      rect.add(back.getBoundingRect());
      back.expandRectByPenWidth(rect);
    }
  }

  public void updatePolygon() {
    layoutChildren(null);
  }

  protected void ownerChange(JGoObjectCollection oldOwner, JGoObjectCollection newOwner, JGoObject mainObject) {
    if (newOwner == null && getAnchor() != null) {
      setAnchor(null);
    }
  }

  protected void partnerUpdate(JGoObject partner, int hint, int prevInt, Object prevVal) {
    super.partnerUpdate(partner, hint, prevInt, prevVal);
    if (partner == getAnchor() && hint == JGoObject.ChangedGeometry) {
      updatePolygon();
    }
  }

  public Insets getInsets() { return myInsets; }
  private Insets myInsets = new Insets(2, 4, 8, 8);  // top, left, bottom, right

  public int getBaseWidth() { return myBaseWidth; }
  public void setBaseWidth(int b) {
    int old = myBaseWidth;
    if (old != b) {
      myBaseWidth = b;
      update(ChangedBaseWidth, old, null);
      updatePolygon();
    }
  }

  public JGoObject getAnchor() { return myAnchor; }

  public void setAnchor(JGoObject obj) {
    JGoObject old = myAnchor;
    if (old != obj) {
      if (old != null && old.getPartner() == this) {
        old.setPartner(null);
        old.setUpdatePartner(false);
      }
      myAnchor = obj;
      if (obj != null) {
        obj.setPartner(this);
        obj.setUpdatePartner(true);
      }
      update(ChangedAnchor, 0, old);
      updatePolygon();
    }
  }


  public int getPartID() {
    return myPartID;
  }

  public void setPartID(int id) {
    int old = myPartID;
    if (old != id) {
      myPartID = id;
      update(ChangedPartID, old, null);
    }
  }

  // get access to the parts of the node
  public JGoText getLabel() { return myLabel; }
  public JGoPolygon getPolygon() { return myPoly; }

  // These are convenience methods

  public String getText() { return getLabel().getText(); }
  public void setText(String s) { getLabel().setText(s); }

  public boolean isEditable() { return getLabel().isEditable(); }
  public void setEditable(boolean e) { getLabel().setEditable(e); }


  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("polyobj")) {
      myPoly = (JGoPolygon)referencedObject;
    } else if (attr.equals("label")) {
      myLabel = (JGoText)referencedObject;
    } else if (attr.equals("anchor")) {
      myAnchor = (JGoObject)referencedObject;
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add Comment element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jBalloon = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.Balloon", jGoElementGroup);
      jBalloon.setAttribute("partid", Integer.toString(myPartID));
      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myPoly != null) {
        svgDoc.registerReferencingNode(jBalloon, "polyobj", myPoly);
      }
      if (myLabel != null) {
        svgDoc.registerReferencingNode(jBalloon, "label", myLabel);
      }
      if (myAnchor != null) {
        svgDoc.registerReferencingNode(jBalloon, "anchor", myAnchor);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a Comment element
      String partid = jGoChildElement.getAttribute("partid");
      if (partid.length() > 0)
        myPartID = Integer.parseInt(partid);
      String rectobj = jGoChildElement.getAttribute("polyobj");
      svgDoc.registerReferencingObject(this, "polyobj", rectobj);
      String label = jGoChildElement.getAttribute("label");
      svgDoc.registerReferencingObject(this, "label", label);
      String anchor = jGoChildElement.getAttribute("anchor");
      svgDoc.registerReferencingObject(this, "anchor", anchor);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }


  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getFlags()) {
      case ChangedPartID:
        e.setNewValueInt(getPartID());
        return;
      case ChangedAnchor:
        e.setNewValue(getAnchor());
        return;
      case ChangedBaseWidth:
        e.setNewValueInt(getBaseWidth());
        return;
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getFlags()) {
      case ChangedPartID:
        setPartID(e.getValueInt(undo));
        return;
      case ChangedAnchor:
        setAnchor((JGoObject)e.getValue(undo));
        return;
      case ChangedBaseWidth:
        setBaseWidth(e.getValueInt(undo));
        return;
      default:
        super.changeValue(e, undo);
        return;
    }
  }

  /** a CHANGED JGoDocumentEvent or JGoViewEvent hint: */
  public static final int ChangedPartID = JGoDocumentEvent.LAST+301;
  public static final int ChangedAnchor = JGoDocumentEvent.LAST+302;
  public static final int ChangedBaseWidth = JGoDocumentEvent.LAST+303;

    // State
  private int myPartID = -1;
  private JGoText myLabel = null;
  private JGoPolygon myPoly = null;
  private JGoObject myAnchor = null;
  private int myBaseWidth = 30;
}
