/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 *  This software is proprietary to and embodies the confidential
 *  technology of Northwoods Software Corporation. Possession, use, or
 *  copying of this software and media is authorized only pursuant to a
 *  valid written license from Northwoods or an authorized sublicensor.
 */

package com.nwoods.jgo.examples;

import java.awt.*;
import java.awt.geom.GeneralPath;
import com.nwoods.jgo.*;

public class Diamond extends JGoDrawable {
  public Diamond() {}

  public Diamond(Rectangle rect)
  {
    super(rect);
  }

  public Diamond(Point loc, Dimension size)
  {
    super(loc, size);
  }

  public void geometryChange(Rectangle prevRect)
  {
    super.geometryChange(prevRect);
    resetPath();
  }

  public void paint(Graphics2D g, JGoView view)
  {
    GeneralPath poly = getPath(view);
    drawPath(g, getPen(), getBrush(), poly);
  }

  public void expandRectByPenWidth(Rectangle rect)
  {
    JGoPen pen = getPen();
    if (pen != null) {
      // normal miter limit is ten times the pen width
      int penWidth = 5*pen.getWidth();
      rect.x -= penWidth;
      rect.y -= penWidth;
      rect.width += penWidth*2;
      rect.height += penWidth*2;
    }
  }

  public boolean isPointInObj(Point pnt)
  {
    // do the easy rectangular case first
    if (!super.isPointInObj(pnt))
      return false;
    GeneralPath poly = getPath(null);
    return poly.contains(pnt.x, pnt.y);
  }

  public boolean getNearestIntersectionPoint(int px, int py, int cx, int cy, Point result) {
    Rectangle r = getBoundingRect();
    int pw = getPen().getWidth();
    int x0 = r.x + r.width/2;    int y0 = r.y - pw;
    int x1 = r.x + r.width + pw; int y1 = r.y + r.height/2;
    int x2 = x0;                 int y2 = r.y + r.height + pw;
    int x3 = r.x - pw;           int y3 = y1;

    Point P = new Point(0, 0);
    double closestdist = 10e20d;
    if (JGoStroke.getNearestIntersectionOnLine(x0, y0, x1, y1, px, py, cx, cy, P)) {
      // calculate the (non-square-rooted) distance from C to P
      double dist = (P.x-px) * (P.x-px) + (P.y-py) * (P.y-py);
      if (dist < closestdist) {
        closestdist = dist;
        result.x = P.x;
        result.y = P.y;
      }
    }

    if (JGoStroke.getNearestIntersectionOnLine(x1, y1, x2, y2, px, py, cx, cy, P)) {
      double dist = (P.x-px) * (P.x-px) + (P.y-py) * (P.y-py);
      if (dist < closestdist) {
        closestdist = dist;
        result.x = P.x;
        result.y = P.y;
      }
    }

    if (JGoStroke.getNearestIntersectionOnLine(x2, y2, x3, y3, px, py, cx, cy, P)) {
      double dist = (P.x-px) * (P.x-px) + (P.y-py) * (P.y-py);
      if (dist < closestdist) {
        closestdist = dist;
        result.x = P.x;
        result.y = P.y;
      }
    }

    if (JGoStroke.getNearestIntersectionOnLine(x3, y3, x0, y0, px, py, cx, cy, P)) {
      double dist = (P.x-px) * (P.x-px) + (P.y-py) * (P.y-py);
      if (dist < closestdist) {
        closestdist = dist;
        result.x = P.x;
        result.y = P.y;
      }
    }

    return (closestdist < 10e20d);
  }

  void makePath(GeneralPath path, JGoView view)
  {
    Rectangle r = getBoundingRect();
    int x0 = r.x + r.width/2;   int y0 = r.y;
    int x1 = r.x + r.width;     int y1 = r.y + r.height/2;
    int x2 = x0;                int y2 = r.y + r.height;
    int x3 = r.x;               int y3 = y1;
    path.moveTo(x0, y0);
    path.lineTo(x1, y1);
    path.lineTo(x2, y2);
    path.lineTo(x3, y3);
    path.lineTo(x0, y0);
    path.closePath();
  }

  // VIEW may be null
  GeneralPath getPath(JGoView view)
  {
    if (myPath == null) {
      myPath = new GeneralPath(GeneralPath.WIND_NON_ZERO, 2*4);
    }
    if (myPath.getCurrentPoint() == null) {
      makePath(myPath, view);
    }
    return myPath;
  }

  void resetPath()
  {
    if (myPath != null)
      myPath.reset();
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
    super.changeValue(e, undo);
    if (e.getFlags() == JGoObject.ChangedGeometry)
      resetPath();
  }
        
 public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add Diamond element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jGoRect = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.Diamond", jGoElementGroup);
    }
    // Add SVG path element
    if (svgDoc.SVGOutputEnabled()) {
      Rectangle r = getBoundingRect();
      int x[] = new int[4];
      int y[] = new int[4];
      x[0] = r.x + r.width/2;   y[0] = r.y;
      x[1] = r.x + r.width;     y[1] = r.y + r.height/2;
      x[2] = x[0];              y[2] = r.y + r.height;
      x[3] = r.x;               y[3] = y[1];
      DomElement diamond = svgDoc.createElement("path");
      String diamondArgs = "M " + Integer.toString(x[0]) + " " + Integer.toString(y[0]);
      diamondArgs += " L " +  Integer.toString(x[1]) + " " + Integer.toString(y[1]);
      diamondArgs += " L " +  Integer.toString(x[2]) + " " + Integer.toString(y[2]);
      diamondArgs += " L " +  Integer.toString(x[3]) + " " + Integer.toString(y[3]);
      diamondArgs += " Z";
      diamond.setAttribute("d", diamondArgs);
      SVGWriteAttributes(diamond);
      jGoElementGroup.appendChild(diamond);
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a <Diamond> element
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    else {
      // This is a <path> element
      // (or perhaps some other SVG element implemented as Diamond)
    }
    return svgElement.getNextSibling();
  }

  transient private GeneralPath myPath = null;
}
