/*
*  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;
import java.util.*;

/**
 * A GeneralNode has an icon, an optional top label, an optional bottom label,
 * a variable number of input ports, and a variable number of output ports.
 * The labels are GeneralNodeLabel's, positioned at the top center and/or
 * the bottom center of the icon.
 * The ports are GeneralNodePort's, one for "input" and one for
 * "output", assuming a left-to-right "flow".
 * The ports may each have GeneralNodePortLabels associated with them.
 * <p>
 * Resizing a node only resizes the icon, not the label or the ports.
 * Furthermore, resizing the icon now maintains its original aspect ratio.
 */
public class GeneralNode extends JGoNode
{
  /** Create an empty GeneralNode.  Call initialize() before using it. */
  public GeneralNode()
  {
    super();
  }

  // The location is the top-left corner of the node;
  // the size is the dimension of the icon;
  // the toplabeltext or bottomlabeltext may be null if no
  // corresponding label is desired.
  public void initialize(Point loc, Dimension size, JGoObject icon,
                         String toplabeltext, String bottomlabeltext,
                         int numinports, int numoutports)
  {
    setInitializing(true);
    // the user can move this node around
    setDraggable(true);
    // the user cannot resize this node
    setResizable(false);
    // if it does become resizable, only show four resize handles
    set4ResizeHandles(true);

    myIcon = icon;
    if (myIcon != null) {
      myIcon.setBoundingRect(loc,size);
      myIcon.setSelectable(false);
      addObjectAtHead(myIcon);
    }

    // the label is a GeneralNodeLabel, centered above the icon
    if (toplabeltext != null) {
      myTopLabel = new GeneralNodeLabel(toplabeltext, this);
    }
    // this label is also a GeneralNodeLabel, centered underneath the icon
    if (bottomlabeltext != null) {
      myBottomLabel = new GeneralNodeLabel(bottomlabeltext, this);
    }

    // create input ports and output ports, each instances of GeneralNodePort
    // here we assume input ports go on the left, output ports on the right
    for (int i = 0; i < numinports; i++) {
      String name = Integer.toString(i);
      GeneralNodePort p = new GeneralNodePort(true, name, this);
      GeneralNodePortLabel l = new GeneralNodePortLabel(name, p);
      addLeftPort(p);
    }
    for (int i = 0; i < numoutports; i++) {
      String name = Integer.toString(i);
      GeneralNodePort p = new GeneralNodePort(false, name, this);
      GeneralNodePortLabel l = new GeneralNodePortLabel(name, p);
      addRightPort(p);
    }

    setInitializing(false);
    layoutChildren(null);
    setTopLeft(loc);
  }

  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    GeneralNode newobj = (GeneralNode)newarea;

    super.copyChildren(newarea, env);

    newobj.myIcon = (JGoObject)env.get(myIcon);
    newobj.myTopLabel = (JGoText)env.get(myTopLabel);
    newobj.myBottomLabel = (JGoText)env.get(myBottomLabel);

    for (int i = 0; i < myLeftPorts.size(); i++) {
      GeneralNodePort olditem = (GeneralNodePort)myLeftPorts.get(i);
      if (olditem != null) {
        GeneralNodePort newitem = (GeneralNodePort)env.get(olditem);
        if (newitem != null) {
          newobj.myLeftPorts.add(newitem);
          // make sure the ports know their place in the node
          newitem.setSideIndex(true, newobj.myLeftPorts.size()-1);
          GeneralNodePortLabel oldlab = olditem.getLabel();
          if (oldlab != null) {
            GeneralNodePortLabel newlab = (GeneralNodePortLabel)env.get(oldlab);
            if (newlab != null) {
              // make sure the port and its label know about each other
              newitem.setLabel(newlab);
            }
          }
        }
      }
    }
    for (int i = 0; i < myRightPorts.size(); i++) {
      GeneralNodePort olditem = (GeneralNodePort)myRightPorts.get(i);
      if (olditem != null) {
        GeneralNodePort newitem = (GeneralNodePort)env.get(olditem);
        if (newitem != null) {
          newobj.myRightPorts.add(newitem);
          // make sure the ports know their place in the node
          newitem.setSideIndex(false, newobj.myRightPorts.size()-1);
          GeneralNodePortLabel oldlab = olditem.getLabel();
          if (oldlab != null) {
            GeneralNodePortLabel newlab = (GeneralNodePortLabel)env.get(oldlab);
            if (newlab != null) {
              // make sure the port and its label know about each other
              newitem.setLabel(newlab);
            }
          }
        }
      }
    }
  }

  /**
   * When an object is removed, make sure there are no more references from fields.
   */
  public JGoObject removeObjectAtPos(JGoListPosition pos)
  {
    JGoObject child = super.removeObjectAtPos(pos);
    if (child == myTopLabel)
      myTopLabel = null;
    else if (child == myBottomLabel)
      myBottomLabel = null;
    else if (child == myIcon)
      myIcon = null;
    return child;
  }

  /**
   * Keep the parts of a GeneralNode positioned relative to each other
   * by setting their locations using some of the standard spots of
   * any JGoObject.
   */
  public void layoutChildren(JGoObject childchanged)
  {
    if (isInitializing()) return;
    setInitializing(true);

    JGoObject icon = getIcon();
    JGoObject toplabel = getTopLabel();
    JGoObject bottomlabel = getBottomLabel();

    int num = getNumLeftPorts();
    int totalh = 0;  // total height of left ports
    int maxw = 0;  // maximum width of left ports
    for (int i = 0; i < num; i++) {
      GeneralNodePort p = getLeftPort(i);
      if (!p.isVisible()) continue;
      totalh += getPortAndLabelHeight(p);
      maxw = Math.max(maxw, getPortAndLabelWidth(p));
    }

    if (icon != null) {
      Dimension minIconSize = getMinimumIconSize();
      int newW = Math.max(minIconSize.width, icon.getWidth());
      int newH = Math.max(minIconSize.height, icon.getHeight());
      icon.setBoundingRect(icon.getLeft() - (newW - icon.getWidth())/2,
                           icon.getTop() - (newH - icon.getHeight())/2,
                           newW, newH);
    }

    int rectx = 0;
    int recty = 0;
    if (icon != null) {
      rectx = icon.getLeft();
    } else {
      rectx = getLeft();
    }
    if (icon != null)
      recty = icon.getTop();
    else
      recty = getTop() + (toplabel != null ? toplabel.getHeight() : 0);

    if (icon != null && icon.getHeight() > totalh) {
      recty += (icon.getHeight()-totalh)/2;
    }

    int h = 0;  // height of visible items so far
    for (int i = 0; i < num; i++) {
      GeneralNodePort p = getLeftPort(i);
      if (!p.isVisible()) continue;
      h += getPortAndLabelHeight(p)/2;
      p.setSpotLocation(RightCenter, rectx, recty+h);
      p.layoutLabel();
      h += getPortAndLabelHeight(p)/2;
    }

    num = getNumRightPorts();
    totalh = 0;  // total height of right ports
    for (int i = 0; i < num; i++) {
      GeneralNodePort p = getRightPort(i);
      if (!p.isVisible()) continue;
      totalh += getPortAndLabelHeight(p);
    }

    if (icon != null) {
      rectx = icon.getLeft() + icon.getWidth();
    } else {
      rectx = getLeft() + getWidth();
    }
    if (icon != null)
      recty = icon.getTop();
    else
      recty = getTop() + (toplabel != null ? toplabel.getHeight() : 0);

    if (icon != null && icon.getHeight() > totalh) {
      recty += (icon.getHeight()-totalh)/2;
    }

    h = 0;  // height of visible items so far
    for (int i = 0; i < num; i++) {
      GeneralNodePort p = getRightPort(i);
      if (!p.isVisible()) continue;
      h += getPortAndLabelHeight(p)/2;
      p.setSpotLocation(LeftCenter, rectx, recty+h);
      p.layoutLabel();
      h += getPortAndLabelHeight(p)/2;
    }

    if (toplabel != null) {
      if (icon != null) {
        toplabel.setSpotLocation(BottomCenter, icon, TopCenter);
      } else {
        toplabel.setSpotLocation(TopCenter, this, TopCenter);
      }
    }
    if (bottomlabel != null) {
      if (icon != null) {
        bottomlabel.setSpotLocation(TopCenter, icon, BottomCenter);
      } else {
        bottomlabel.setSpotLocation(BottomCenter, this, BottomCenter);
      }
    }

    setInitializing(false);
  }

  /**
   * If this object is resized, do the part positioning lay out again.
   * The ports and the text labels do not get resized; only the icon
   * changes size, while keeping its old aspect ratio.
   */
  public void rescaleChildren(Rectangle prevRect)
  {
    // only change size of icon; need to calculate its new size while
    // keeping its old aspect ratio
    if (getIcon() != null && getIcon().isVisible()) {
      int oldw = getIcon().getWidth();
      int oldh = getIcon().getHeight();
      if (oldw <= 0) oldw = 1;
      double ratio = oldh/((double)oldw);
      // figure out how much space is left in the area after accounting
      // for any ports and labels
      Dimension minSize = getMinimumSizeWithIconSize(0, 0);
      int iconw = getWidth() - minSize.width;
      int iconh = getHeight();
      if (getTopLabel() != null && getTopLabel().isVisible())
        iconh -= getTopLabel().getHeight();
      if (getBottomLabel() != null && getBottomLabel().isVisible())
        iconh -= getBottomLabel().getHeight();
      Dimension minIconSize = getMinimumIconSize();
      iconw = Math.max(iconw, minIconSize.width);
      iconh = Math.max(iconh, minIconSize.height);
      // now we have the maximum bounds for the icon, figure out the
      // right width and height that fit while maintaining the aspect ratio
      double maxratio = iconh/((double)iconw);
      if (ratio < maxratio)
        iconh = (int)Math.rint(ratio*iconw);
      else
        iconw = (int)Math.rint(iconh/ratio);
      getIcon().setSize(iconw, iconh);
    }
  }

  public int getPortAndLabelWidth(GeneralNodePort port)
  {
    if (!port.isVisible()) return 0;
    GeneralNodePortLabel label = port.getLabel();
    if (label != null && label.isVisible())
      return port.getWidth() + port.getLabelSpacing() + label.getWidth();
    else
      return port.getWidth();
  }

  public int getPortAndLabelHeight(GeneralNodePort port)
  {
    if (!port.isVisible()) return 0;
    GeneralNodePortLabel label = port.getLabel();
    if (label != null && label.isVisible())
      return Math.max(port.getHeight(), label.getHeight());
    else
      return port.getHeight();
  }

  // compute the minimum size of this node assuming the icon is of size WxH
  public Dimension getMinimumSizeWithIconSize(int w, int h)
  {
    // account for any ports and labels on the left side
    int num = getNumLeftPorts();
    int portandlabelwidth = 0;  // max width
    int portandlabelheight = 0; // total height
    for (int i = 0; i < num; i++) {
      GeneralNodePort p = getLeftPort(i);
      portandlabelwidth = Math.max(portandlabelwidth, getPortAndLabelWidth(p));
      portandlabelheight += getPortAndLabelHeight(p);
    }
    w += portandlabelwidth;
    h = Math.max(h, portandlabelheight);

    // now consider the ports on the right side
    num = getNumRightPorts();
    portandlabelwidth = 0;  // max width
    portandlabelheight = 0; // total height
    for (int i = 0; i < num; i++) {
      GeneralNodePort p = getRightPort(i);
      portandlabelwidth = Math.max(portandlabelwidth, getPortAndLabelWidth(p));
      portandlabelheight += getPortAndLabelHeight(p);
    }
    w += portandlabelwidth;
    h = Math.max(h, portandlabelheight);

    // consider top and bottom labels
    int labelh = 0;
    JGoObject lab = getTopLabel();
    if (lab != null && lab.isVisible()) {
      w = Math.max(w, lab.getWidth());
      h += lab.getHeight();
    }
    lab = getBottomLabel();
    if (lab != null && lab.isVisible()) {
      w = Math.max(w, lab.getWidth());
      h += lab.getHeight();
    }

    return new Dimension(w, h);
  }

  public Dimension getMinimumIconSize()
  {
    return new Dimension(20, 20);
  }

  public Dimension getMinimumSize()
  {
    // account for any minimum desired icon size
    Dimension minIconSize = getMinimumIconSize();
    int w = minIconSize.width;
    int h = minIconSize.height;

    return getMinimumSizeWithIconSize(w, h);
  }


  // constrain to the minimum width and height
  public void setBoundingRect(int left, int top, int width, int height)
  {
    Dimension minSize = getMinimumSize();
    super.setBoundingRect(left, top,
                          Math.max(width, minSize.width),
                          Math.max(height, minSize.height));
  }

  // limit the minimum width and height for resizing
  protected Rectangle handleResize(Graphics2D g, JGoView view, Rectangle prevRect,
                                   Point newPoint, int whichHandle, int event,
                                   int minWidth, int minHeight)
  {
    Dimension minSize = getMinimumSize();
    Rectangle newRect = super.handleResize(g, view, prevRect, newPoint, whichHandle, event,
                                           Math.max(minWidth, minSize.width), Math.max(minHeight, minSize.height));
    // resize continuously (default only does setBoundingRect on MouseUp)
    if (event == JGoView.EventMouseMove)
      setBoundingRect(newRect);
    return null;
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("toplabel")) {
      myTopLabel = (JGoText)referencedObject;
    }
    else if (attr.equals("bottomlabel")) {
      myBottomLabel = (JGoText)referencedObject;
    }
    else if (attr.equals("icon")) {
      myIcon = (JGoObject)referencedObject;
    }
    else if (attr.equals("leftports")) {
      myLeftPorts.add(referencedObject);
    }
    else if (attr.equals("rightports")) {
      myRightPorts.add(referencedObject);
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add GeneralNode element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jGeneralNode = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.GeneralNode", jGoElementGroup);
      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myTopLabel != null) {
        svgDoc.registerReferencingNode(jGeneralNode, "toplabel", myTopLabel);
      }
      if (myBottomLabel != null) {
        svgDoc.registerReferencingNode(jGeneralNode, "bottomlabel",
                                       myBottomLabel);
      }
      if (myIcon != null) {
        svgDoc.registerReferencingNode(jGeneralNode, "icon", myIcon);
      }
      for (int i = 0; i < myLeftPorts.size(); i++) {
        svgDoc.registerReferencingNode(jGeneralNode, "leftports",
                                       (GeneralNodePort) myLeftPorts.get(i));
      }
      for (int i = 0; i < myRightPorts.size(); i++) {
        svgDoc.registerReferencingNode(jGeneralNode, "rightports",
                                       (GeneralNodePort) myRightPorts.get(i));
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a JGoBasicNode element
      String toplabel = jGoChildElement.getAttribute("toplabel");
      svgDoc.registerReferencingObject(this, "toplabel", toplabel);
      String bottomlabel = jGoChildElement.getAttribute("bottomlabel");
      svgDoc.registerReferencingObject(this, "bottomlabel", bottomlabel);
      String icon = jGoChildElement.getAttribute("icon");
      svgDoc.registerReferencingObject(this, "icon", icon);

      String sLeftPorts = jGoChildElement.getAttribute("leftports");
      while (sLeftPorts.length() > 0) {
        int nEnd = sLeftPorts.indexOf(" ");
        if (nEnd == -1)
          nEnd = sLeftPorts.length();
        String sPort = sLeftPorts.substring(0, nEnd);
        if (nEnd >= sLeftPorts.length())
          sLeftPorts = "";
        else
          sLeftPorts = sLeftPorts.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "leftports", sPort);
      }
      String sRightPorts = jGoChildElement.getAttribute("rightports");
      while (sRightPorts.length() > 0) {
        int nEnd = sRightPorts.indexOf(" ");
        if (nEnd == -1)
          nEnd = sRightPorts.length();
        String sPort = sRightPorts.substring(0, nEnd);
        if (nEnd >= sRightPorts.length())
          sRightPorts = "";
        else
          sRightPorts = sRightPorts.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "rightports", sPort);
      }

      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }


  // get the basic parts of a General Node

  public JGoText getTopLabel() { return myTopLabel; }

  public JGoText getBottomLabel() { return myBottomLabel; }

  public JGoObject getIcon() { return myIcon; }


  // get the number of ports on each side

  public int getNumLeftPorts() { return myLeftPorts.size(); }

  public int getNumRightPorts() { return myRightPorts.size(); }


  // get a port, indexed from zero, on each side

  public GeneralNodePort getLeftPort(int i)
  {
    if (i < 0 || i >= myLeftPorts.size())
      return null;
    else
      return (GeneralNodePort)myLeftPorts.get(i);
  }

  public GeneralNodePort getRightPort(int i)
  {
    if (i < 0 || i >= myRightPorts.size())
      return null;
    else
      return (GeneralNodePort)myRightPorts.get(i);
  }


  // internal method
  void initializePort(GeneralNodePort p)
  {
    if (p == null) return;
    p.setSelectable(false);
    p.setDraggable(false);
    p.setResizable(false);
    if (p.getParent() == null) {
      addObjectAtTail(p);
      if (p.getLabel() != null)
        addObjectAtTail(p.getLabel());
    }
  }

  final public void addLeftPort(GeneralNodePort p)
  {
    insertLeftPort(getNumLeftPorts(), p);
  }

  final public void addRightPort(GeneralNodePort p)
  {
    insertRightPort(getNumRightPorts(), p);
  }

  public void insertLeftPort(int i, GeneralNodePort p)
  {
    if (p == null) return;
    if (i < 0) return;
    if (i < myLeftPorts.size()) {
      myLeftPorts.add(i, p);
      p.setSideIndex(true, i);
    } else {
      myLeftPorts.add(p);
      p.setSideIndex(true, myLeftPorts.size()-1);
    }
    initializePort(p);
    layoutChildren(p);
    update(PortInsertedChanged, -(i+1), p);
  }

  public void insertRightPort(int i, GeneralNodePort p)
  {
    if (p == null) return;
    if (i < 0) return;
    if (i < myRightPorts.size()) {
      myRightPorts.add(i, p);
      p.setSideIndex(false, i);
    } else {
      myRightPorts.add(p);
      p.setSideIndex(false, myRightPorts.size()-1);
    }
    initializePort(p);
    layoutChildren(p);
    update(PortInsertedChanged, i, p);
  }


  // internal method
  void deletePort(GeneralNodePort p)
  {
    if (p != null) {
      if (p.getLabel() != null)
        removeObject(p.getLabel());
      removeObject(p);
      p.setSideIndex(p.isOnLeftSide(), -1);
    }
  }

  public void removeLeftPort(int i)
  {
    if (i < 0 || i >= myLeftPorts.size()) return;
    GeneralNodePort oldp = (GeneralNodePort)myLeftPorts.remove(i);
    deletePort(oldp);
    layoutChildren(oldp);
    update(PortRemovedChanged, -(i+1), oldp);
  }

  public void removeRightPort(int i)
  {
    if (i < 0 || i >= myRightPorts.size()) return;
    GeneralNodePort oldp = (GeneralNodePort)myRightPorts.remove(i);
    deletePort(oldp);
    layoutChildren(oldp);
    update(PortRemovedChanged, i, oldp);
  }


  public void setLeftPort(int i, GeneralNodePort p)
  {
    GeneralNodePort oldp = getLeftPort(i);
    if (oldp != p) {
      if (oldp != null) {
        if (p != null)
          p.setBoundingRect(oldp.getBoundingRect());
        removeObject(oldp);
      }
      myLeftPorts.set(i, p);
      p.setSideIndex(true, i);
      initializePort(p);
      update(PortSetChanged, -(i+1), oldp);
    }
  }

  public void setRightPort(int i, GeneralNodePort p)
  {
    GeneralNodePort oldp = getRightPort(i);
    if (oldp != p) {
      if (oldp != null) {
        if (p != null)
          p.setBoundingRect(oldp.getBoundingRect());
        removeObject(oldp);
      }
      myRightPorts.set(i, p);
      p.setSideIndex(false, i);
      initializePort(p);
      update(PortSetChanged, i, oldp);
    }
  }


  // undo/redo support

  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getFlags()) {
      case PortInsertedChanged:
        // the old value information indicates where it had been inserted
        return;
      case PortRemovedChanged:
        // the old value information indicates where and what had been removed
        return;
      case PortSetChanged: {
        // use the old value integer to figure out which port had been replaced,
        // and then remember the new port so that Redo will work
        int i = e.getOldValueInt();
        if (i < 0) {
          e.setNewValue(getLeftPort(-i-1));
        } else {
          e.setNewValue(getRightPort(i));
        }
        return; }
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getFlags()) {
      case PortInsertedChanged: {
        // the old value information indicates where it had been inserted
        int i = e.getOldValueInt();
        if (i < 0) {
          i = -i - 1;
          if (undo) {
            removeLeftPort(i);
          } else {
            insertLeftPort(i, (GeneralNodePort)e.getOldValue());
          }
        } else {
          if (undo) {
            removeRightPort(i);
          } else {
            insertRightPort(i, (GeneralNodePort)e.getOldValue());
          }
        }
        return; }
      case PortRemovedChanged: {
        // the old value information indicates where and what had been removed
        int i = e.getOldValueInt();
        if (i < 0) {
          i = -i - 1;
          if (undo) {
            insertLeftPort(i, (GeneralNodePort)e.getOldValue());
          } else {
            removeLeftPort(i);
          }
        } else {
          if (undo) {
            insertRightPort(i, (GeneralNodePort)e.getOldValue());
          } else {
            removeRightPort(i);
          }
        }
        return; }
      case PortSetChanged: {
        // use the old value integer to figure out which port needs to be updated
        int i = e.getOldValueInt();
        if (i < 0) {
          i = -i - 1;
          setLeftPort(i, (GeneralNodePort)e.getValue(undo));
        } else {
          setRightPort(i, (GeneralNodePort)e.getValue(undo));
        }
        return; }
      default:
        super.changeValue(e, undo);
        return;
    }
  }


  // Event hints
  public static final int PortInsertedChanged = JGoDocumentEvent.LAST + 10010;
  public static final int PortRemovedChanged = JGoDocumentEvent.LAST + 10011;
  public static final int PortSetChanged = JGoDocumentEvent.LAST + 10012;


    // State
  protected JGoText myTopLabel = null;
  protected JGoText myBottomLabel = null;
  protected JGoObject myIcon = null;
  protected ArrayList myLeftPorts = new ArrayList();
  protected ArrayList myRightPorts = new ArrayList();

  /*
   * A real application will have some other data associated with
   * the node, holding state and methods to be called according to
   * the needs of the application.
   */
}
