/*
 *  Copyright (c) Northwoods Software Corporation, 1999-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;

/**
 * GeneralNodePort has been implemented as a port of StyleTriangle,
 * assuming ports are only on the left and right sides of a GeneralNode.
 * <p>
 * GeneralNodePort also has some smarts about the kinds of links
 * that can be made to this kind of port.
 * <p>
 * You can easily change the appearance of the port by calling
 * setPortStyle(JGoPort.StyleObject) and then setPortObject to
 * change the object used to represent the port.
 * For example you may want to use an instance of JGoImage.
 * These objects can be shared by more than one JGoPort.
 * <p>
 * Each GeneralNodePort can have a GeneralNodePortLabel associated
 * with it, positioned alongside the port on the outside, and each
 * one knows its index in the parent GeneralNode's vector of
 * input or output ports.  Each GeneralNodePort also has a name;
 * setting the name will change the label's text.
 */
public class GeneralNodePort extends JGoPort
{
  /** This creates a StyleEllipse port.  Call initialize() before using. */
  public GeneralNodePort()
  {
    super();
  }

  /** This creates a light gray StyleTriangle port of the appropriate direction. */
  public GeneralNodePort(boolean input, String name, JGoArea parent)
  {
    super(TriangleRect());
    initialize(input, name, parent);
  }

  public void initialize(boolean input, String name, JGoArea parent)
  {
    setSelectable(false);
    setDraggable(false);
    setResizable(false);
    setVisible(true);
    // assume an outlined light gray triangle
    setStyle(StyleTriangle);
    setPen(JGoPen.darkGray);
    setBrush(JGoBrush.lightGray);
    // assume inputs are on the left, outputs are on the right
    if (input) {
      myLeftSide = true;
      setValidSource(false);
      setValidDestination(true);
      setToSpot(LeftCenter);
    } else {
      myLeftSide = false;
      setValidSource(true);
      setValidDestination(false);
      setFromSpot(RightCenter);
    }
    // put in the GeneralNode area
    setTopLeft(parent.getLeft(), parent.getTop());
    parent.addObjectAtTail(this);
    // each port has a name
    setName(name);
  }

  // ports remember whether they are inputs or outputs
  public final boolean isInput()
  {
    return isValidDestination();
  }

  public final boolean isOutput()
  {
    return isValidSource();
  }

  // Only allow links from output ports to input ports,
  // only between different nodes, and
  // only if there isn't already a link to "to".
  public boolean validLink(JGoPort to)
  {
    if (to.getParent() == null)
      return false;
    if (to instanceof GeneralNodePort)
      return (super.validLink(to) &&
              getParent() != to.getParent() &&
              isOutput() &&
              (to instanceof GeneralNodePort) &&
              ((GeneralNodePort)to).isInput() &&
              !alreadyLinked(to));
    else
      return (super.validLink(to) &&
              getParent() != to.getParent() &&
              isOutput());
  }

  // return true if there already is a link from this port to
  // the given port
  public boolean alreadyLinked(JGoPort dst)
  {
    JGoListPosition pos = getFirstLinkPos();
    while (pos != null) {
      JGoLink link = getLinkAtPos(pos);
      pos = getNextLinkPos(pos);

      if (link.getFromPort() == this && link.getToPort() == dst)
        return true;
    }
    return false;
  }


  // A port on the left side doesn't have to be an input port, and
  // one on the right side doesn't have to be an output port.
  // Keeping this as a separate bit of state may allow for
  // ports that are input and output, on either side of the node.
  public boolean isOnLeftSide() { return myLeftSide; }

  // This method is a convenient way of distinguishing between
  // ports on the same GeneralNode
  public int getIndex() {  return myIndex; }

  // this is just used internally by GeneralNode when
  // adding/inserting/removing ports
  void setSideIndex(boolean b, int x)
  {
    myLeftSide = b;
    myIndex = x;
  }


  // access to the port label

  public GeneralNodePortLabel getLabel() { return myPortLabel; }

  public void setLabel(GeneralNodePortLabel lab)
  {
    GeneralNodePortLabel oldlabel = myPortLabel;
    if (oldlabel != lab) {
      if (oldlabel != null) {
        if (getParent() != null)
          getParent().removeObject(oldlabel);
        oldlabel.setPartner(null);
      }

      myPortLabel = lab;

      if (lab != null) {
        lab.setPartner(this);
        if (isOnLeftSide())
          lab.setAlignment(JGoText.ALIGN_RIGHT);
        else
          lab.setAlignment(JGoText.ALIGN_LEFT);
        if (getParent() != null)
          getParent().addObjectAtTail(lab);
      }
      layoutLabel();

      update(LabelChanged, 0, oldlabel);
    }
  }

  // position the port label to the appropriate side of the port,
  // so that it doesn't overlap the icon
  public void layoutLabel()
  {
    if (getLabel() != null) {
      if (isOnLeftSide()) {
        getLabel().setSpotLocation(CenterRight, this, CenterLeft);
        getLabel().setLeft(getLabel().getLeft() - getLabelSpacing());
      } else {
        getLabel().setSpotLocation(CenterLeft, this, CenterRight);
        getLabel().setLeft(getLabel().getLeft() + getLabelSpacing());
      }
    }
  }

  // return the desired distance between the port label and the port itself
  public int getLabelSpacing()
  {
    return 2;
  }


  // be smart about the point where links should go, to take into
  // account the size of any port label
  public Point getLinkPoint(int spot, Point result)
  {
    switch (spot) {
      default:
        return super.getLinkPoint(spot, result);
      case RightCenter:
      {
        Rectangle rect = getBoundingRect();
        if (result == null)
          result = new Point(0, 0);

        result.x = rect.x + rect.width;
        result.y = rect.y + rect.height/2;

        GeneralNodePortLabel lab = getLabel();
        if (lab != null && lab.isVisible()) {
          result.x += lab.getWidth() + getLabelSpacing();
        }
        break;
      }
      case LeftCenter:
      {
        Rectangle rect = getBoundingRect();
        if (result == null)
          result = new Point(0, 0);

        result.x = rect.x;
        result.y = rect.y + rect.height/2;

        GeneralNodePortLabel lab = getLabel();
        if (lab != null && lab.isVisible()) {
          result.x -= lab.getWidth() + getLabelSpacing();
        }
        break;
      }
    }

    return result;
  }

  public Point getLinkPointFromPoint(int x, int y, Point result) {
    if (isOnLeftSide()) {
      return getLinkPoint(LeftCenter, result);
    } else {
      return getLinkPoint(RightCenter, result);
    }
  }

  // the name property
  public String getName() { return myName; }

  public void setName(String n)
  {
    String oldname = myName;
    if (!oldname.equals(n)) {
      myName = n;

      if (getLabel() != null)
        getLabel().setText(n);

      update(NameChanged, 0, oldname);
    }
  }

  /**
   * A convenience method for returning the parent as a GeneralNode.
   */
  public GeneralNode getNode() { return (GeneralNode)getParent(); }


  // show the name of the port, in case the JGoText label is not present
  public String getToolTipText()
  {
    return getName();
  }


  // undo/redo support

  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getFlags()) {
      case NameChanged:
        e.setNewValue(getName());
        return;
      case LabelChanged:
        e.setNewValue(getLabel());
        return;
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getFlags()) {
      case NameChanged:
        setName((String)e.getValue(undo));
        return;
      case LabelChanged:
        setLabel((GeneralNodePortLabel)e.getValue(undo));
        return;
      default:
        super.changeValue(e, undo);
        return;
    }
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("portlabel")) {
      myPortLabel = (GeneralNodePortLabel)referencedObject;
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add GeneralNodePort element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jGeneralNodePort = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.GeneralNodePort", jGoElementGroup);
      jGeneralNodePort.setAttribute("index", Integer.toString(myIndex));
      jGeneralNodePort.setAttribute("leftside", myLeftSide ? "true" : "false");
      jGeneralNodePort.setAttribute("name", myName);
      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myPortLabel != null) {
        svgDoc.registerReferencingNode(jGeneralNodePort, "portlabel",
                                       myPortLabel);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a GeneralNodePort element
      myIndex = Integer.parseInt(jGoChildElement.getAttribute("index"));
      myLeftSide = jGoChildElement.getAttribute("leftside").equals("true");
      myName = jGoChildElement.getAttribute("name");
      String portlabel = jGoChildElement.getAttribute("portlabel");
      svgDoc.registerReferencingObject(this, "portlabel", portlabel);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }

  // Event hints
  public static final int NameChanged = JGoDocumentEvent.LAST + 10015;
  public static final int LabelChanged = JGoDocumentEvent.LAST + 10016;


  // return the bounding rectangle for the triangle-shaped port
  static public Rectangle TriangleRect()
  {
    return myTriangleRect;
  }

  static private Rectangle myTriangleRect = new Rectangle(0, 0, 8, 8);

  // instance state
  private boolean myLeftSide = true;
  private int myIndex = 0;
  private GeneralNodePortLabel myPortLabel = null;
  private String myName = "";
}

