/*
 *  Copyright (c) Northwoods Software Corporation, 1999-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;

/**
 * GeneralNodePortLabel is just a JGoText that treats a single click as a command
 * to start editing the text of the label, and that knows about the
 * GeneralNodePort that it is associated with.
 */
public class GeneralNodePortLabel extends JGoText
{
  /** Create an empty label for a GeneralNode.  Call initialize() before using. */
  public GeneralNodePortLabel()
  {
    super();
  }

  /** Create a label containing the given text for a GeneralNode */
  public GeneralNodePortLabel(String text, GeneralNodePort port)
  {
    super(text);
    initialize(text, port);
  }

  // initialize the right state, add to the parent area, and associate this label with the port
  public void initialize(String text, GeneralNodePort port)
  {
    setSelectable(false);
    setDraggable(false);
    setResizable(false);
    setVisible(true);
    setEditable(true);
    setEditOnSingleClick(true);
    setTransparent(true);
    setFontSize(getFontSize()-1);
    // put in the parent area, which should be a GeneralNode
    JGoArea parent = port.getParent();
    setTopLeft(parent.getLeft(), parent.getTop());
    parent.addObjectAtTail(this);
    // and associate with the GeneralNodePort
    port.setLabel(this);
  }


  // when the label's text changes, update the port's name too
  public void setText(String str)
  {
    if (!getText().equals(str)) {
      super.setText(str);
      if (getPort() != null)
        getPort().setName(getText());
    }
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("port")) {
      myPort = (GeneralNodePort)referencedObject;
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add GeneralNodePortLabel element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jGeneralNodePortLabel = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.GeneralNodePortLabel", jGoElementGroup);
      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myPort != null) {
        svgDoc.registerReferencingNode(jGeneralNodePortLabel, "port", myPort);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a GeneralNodePortLabel element
      String port = jGoChildElement.getAttribute("port");
      svgDoc.registerReferencingObject(this, "port", port);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }

  // access to the port corresponding to this label
  public GeneralNodePort getPort() { return myPort; }

  // to associate a port with a label, call GeneralNodePort.setLabel instead
  void setPort(GeneralNodePort p) { myPort = p; }

  // State
  private GeneralNodePort myPort = null;
}
