/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import java.applet.*;
import java.awt.*;
import java.awt.dnd.*;
import java.awt.event.*;
import javax.swing.*;
import com.nwoods.jgo.*;

/**
 * This is a Simple JGo applet/application that demonstrates the use
 * of JGoIconicNodes, adding objects from a palette without using Java drag-and-drop,
 * selecting the object under the mouse without clicking, and performing some
 * action with any single mouse click both on objects and in the background.
 * For nodes, the action brings up a dialog to modify the label.
 * For both nodes and links, a right-mouse-click deletes the object, after a
 * confirming dialog.
 * <p>
 * Users can just click at spots in the view in order to create new IconicNodes
 * at that location.  The node's icon is determined by the currently selected image
 * in the palette.  Unlike most examples, drag-and-drop is disabled from the palette.
 * When no image is selected in the palette, no JGoIconicNode is created when the user
 * clicks in the background.  The palette is also different than most because the
 * contents of the palette are just JGoImages, rather than whole nodes such as
 * JGoIconicNodes.
 * <p>
 * When the "Insert Mode" check box is turned off, clicking in the background
 * does not insert any new JGoIconicNodes.
 * <p>
 * As the mouse moves around the view, any object underneath the mouse is selected.
 * The status bar is updated with a description of the primary selection.
 * <p>
 * However, when the "Point to Select" mode is turned off, passing over an object
 * does not change the selection.  A single click selects an object, and a double-click
 * performs the action on the object.
 * <p>
 * Various keyboard commands are also defined--see the implementation of
 * handleKeyPressed for details.
 */
public class IconicApp extends JApplet implements JGoViewListener {
  public IconicApp() {
    myView = new JGoView() {
      public void doUncapturedMouseMove(int i, Point point, Point point1) {
        if (myPointToSelectModeCheckBox.isSelected())
          handleSelectionUnderMouse(point);
        super.doUncapturedMouseMove(i, point, point1);
      }
    };
    myView.setBorder(new javax.swing.border.EtchedBorder());
    myView.setHidingDisabledScrollbars(true);
    myView.getDocument().setDocumentSize(400, 300);
    myPalette = new JGoPalette();
    myPanel = new JPanel();
    myPanel.setLayout(new FlowLayout());
    myInsertModeCheckBox = new JCheckBox("Insert Mode");
    myInsertModeCheckBox.setSelected(true);
    myPanel.add(myInsertModeCheckBox);
    myPointToSelectModeCheckBox = new JCheckBox("Point to Select");
    myPointToSelectModeCheckBox.setSelected(true);
    myPanel.add(myPointToSelectModeCheckBox);
    myLinkModeCheckBox = new JCheckBox("Linking Mode");
    myLinkModeCheckBox.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent actionevent) {
        setLinkMode(myLinkModeCheckBox.isSelected());
      }
    });
    myPanel.add(myLinkModeCheckBox);
    myStatusBarLabel = new JLabel();
    myStatusBarLabel.setText(" ");
    myView.addViewListener(this);
    myView.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent evt) {
        handleKeyPressed(evt);
      }
    });
    getContentPane().add(myView, BorderLayout.CENTER);
    getContentPane().add(myPalette, BorderLayout.WEST);
    getContentPane().add(myPanel, BorderLayout.NORTH);
    getContentPane().add(myStatusBarLabel, BorderLayout.SOUTH);
  }

  public static void main(String args[]) {  // run applet as an application
    try {
      JFrame mainFrame = new JFrame();
      // close the application when the main window closes
      mainFrame.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent event) {
          System.exit(0);
        }
      });
      mainFrame.setTitle("Point to Select, Click for Action or Insertion");
      mainFrame.setSize(500, 400);

      IconicApp app = new IconicApp();
      Container contentPane = mainFrame.getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(app, BorderLayout.CENTER);
      contentPane.validate();

      mainFrame.setVisible(true);

      app.init();
      app.start();
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }
  }

  public void init() {
    // disable starting a drag-and-drop in the view
    // disable dragging from the palette and dropping in the view
    myView.setDragDropEnabled(false);  // ??
    myView.setIncludingNegativeCoords(true);
    myView.setHidingDisabledScrollbars(true);
    myView.getDocument().setUndoManager(new JGoUndoManager());
    myPalette.setInternalMouseActions(DnDConstants.ACTION_NONE);
    myPalette.setDragDropEnabled(false);

    // in this example, the palette just holds images, not whole nodes
    JGoImage img = new JGoImage();
    img.setResizable(false);
    img.loadImage((IconicApp.class).getResource("demo1/star.gif"), true);
    img.setSize(32, 32);
    myPalette.getDocument().addObjectAtTail(img);
    img = new JGoImage();
    img.setResizable(false);
    img.loadImage((IconicApp.class).getResource("demo1/doc.gif"), true);
    img.setSize(32, 32);
    myPalette.getDocument().addObjectAtTail(img);
    myPalette.selectObject(img);
  }

  // As the mouse moves over objects, we select them, thereby deselecting any
  // other object.  As the mouse moves over the background, we clear the selection,
  // unless there's already a multiple selection specified by the user.
  void handleSelectionUnderMouse(Point point) {
    int num = myView.getSelection().getNumObjects();
    JGoObject obj = myView.pickDocObject(point, true);
    if (obj != null && num <= 1)
      myView.selectObject(obj);
    else if (obj == null && num == 1)
      myView.getSelection().clearSelection();
  }

  // cut/copy/paste only works when the clipboard is available--
  // i.e. not normally when running as an applet
  void handleKeyPressed(KeyEvent evt) {
    int code = evt.getKeyCode();
    if (code == KeyEvent.VK_DELETE) {  // delete the currently selected objects
      myView.deleteSelection();
    } else if (code == KeyEvent.VK_INSERT) {   // insert a new node in the middle of the window
      Rectangle rectangle = myView.getViewRect();
      insertNode(new Point(rectangle.x + rectangle.width / 2, rectangle.y + rectangle.height / 2));
    } else if (code == KeyEvent.VK_HOME) {  // scroll so that the top-left-most object is at the top-left corner
      Rectangle docbounds = myView.getDocument().computeBounds();
      myView.setViewPosition(docbounds.x, docbounds.y);
    } else if (code == KeyEvent.VK_END) {  // scroll so that the bottom-right-most object is at the bottom-right corner
      Rectangle docbounds = myView.getDocument().computeBounds();
      Dimension viewsize = myView.getExtentSize();
      myView.setViewPosition(Math.max(docbounds.x, docbounds.x+docbounds.width-viewsize.width),
                             Math.max(docbounds.y, docbounds.y+docbounds.height-viewsize.height));
    } else if (evt.isControlDown() && code == KeyEvent.VK_A) {  // select all
      myView.selectAll();
    } else if (evt.isControlDown() && code == KeyEvent.VK_X) {  // cut
      myView.cut();
    } else if (evt.isControlDown() && code == KeyEvent.VK_C) {  // copy
      myView.copy();
    } else if (evt.isControlDown() && code == KeyEvent.VK_V) {  // paste
      myView.paste();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Z) {  // undo
      myView.getDocument().undo();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Y) {  // redo
      myView.getDocument().redo();
    }
  }

  // When a link is drawn by the user, make it wide and blue with an arrowhead.
  // A single click on a node starts editing the node--you can substitute other operations
  // if you desire.  A single right mouse click causes a different action, in this demo,
  // a prompted deletion.
  // When PointToSelect mode is false, these actions occur on a double-click instead.
  // A click in the background inserts a new node at that point.
  // We also handle selection gained and selection lost view events to maintain
  // the status bar's description of the current object, the primary selection.
  public void viewChanged(JGoViewEvent evt) {
    if (evt.getHint() == JGoViewEvent.LINK_CREATED && (evt.getJGoObject() instanceof JGoLink)) {
      JGoLink link = (JGoLink)evt.getJGoObject();
      link.setPen(JGoPen.make(JGoPen.SOLID, 2, Color.blue));
      link.setArrowHeads(false, true);
    } else if (evt.getHint() == JGoViewEvent.CLICKED) {
      if (myPointToSelectModeCheckBox.isSelected()) {
        if ((evt.getModifiers() & InputEvent.BUTTON3_MASK) != 0)
          deleteObject(evt.getJGoObject());
        else
          editObject(evt.getJGoObject());
      }
    } else if (evt.getHint() == JGoViewEvent.DOUBLE_CLICKED) {
      if (!myPointToSelectModeCheckBox.isSelected()) {
        if ((evt.getModifiers() & InputEvent.BUTTON3_MASK) != 0)
          deleteObject(evt.getJGoObject());
        else
          editObject(evt.getJGoObject());
      }
    } else if (evt.getHint() == JGoViewEvent.BACKGROUND_CLICKED) {
      if (myInsertModeCheckBox.isSelected() &&
          myPalette.getSelection().getPrimarySelection() != null &&
          myView.getSelection().getNumObjects() == 0 &&
          (evt.getModifiers() & InputEvent.BUTTON3_MASK) == 0) {  // not right-mouse-button
        insertNode(evt.getPointDocCoords());
      }
    } else if (evt.getHint() == JGoViewEvent.SELECTION_GAINED ||
               evt.getHint() == JGoViewEvent.SELECTION_LOST) {
      describeCurrentObject();
    }
  }

  // Create an JGoIconicNode at a given point with an icon whose image is selected in the palette.
  // The node's Port is Visible when linking mode is true so that users can draw new links.
  // When changing a document, we do so within a transaction so that undo/redo works normally.
  void insertNode(Point point) {
    JGoDocument doc = myView.getDocument();
    doc.startTransaction();
    JGoObject obj = myPalette.getSelection().getPrimarySelection();
    JGoIconicNode node = new JGoIconicNode(Integer.toString(++myNodeCounter));
    // copy the JGoImage
    JGoImage img = (JGoImage)obj.copyObject(new JGoCopyMap());
    // initialize the new node's Icon and its other properties
    node.setIcon(img);
    node.setLocation(point);
    node.getLabel().setEditable(false);
    node.getPort().setBoundingRect(node.getIcon().getBoundingRect());
    node.getPort().setVisible(myLinkModeCheckBox.isSelected());
    doc.addObjectAtTail(node);
    doc.endTransaction("inserted node");
  }

  // For this example, editing a node just lets the user change the label.
  // You can substitute much more descriptive dialogs for the information appropriate
  // to your application, or perform other actions.
  void editObject(JGoObject obj) {
    if (obj == null) return;
    obj = obj.getParentNode();
    if (obj instanceof JGoIconicNode) {
      JGoIconicNode node = (JGoIconicNode)obj;
      String str = JOptionPane.showInputDialog(this, "Enter a new label:");
      if (str != null && !str.equals(node.getLabel().getText())) {
        JGoDocument doc = myView.getDocument();
        doc.startTransaction();
        node.getLabel().setText(str);
        doc.endTransaction("changed label");
      }
    }  // but don't do anything for a link
  }

  // An alternative action, for right-mouse-button events.
  // You might substitute bringing up a popup menu with different commands.
  void deleteObject(JGoObject obj) {
    if (obj == null) return;
    if (JOptionPane.showConfirmDialog(this, "Delete selection?", "", JOptionPane.OK_CANCEL_OPTION) == JOptionPane.OK_OPTION) {
      myView.deleteSelection();
    }
  }

  // Look at the primary selection and update the status bar correspondingly.
  void describeCurrentObject() {
    JGoObject obj = myView.getSelection().getPrimarySelection();
    if (obj == null) {
      myStatusBarLabel.setText(" ");
    } else {
      if (obj instanceof JGoIconicNode) {
        JGoIconicNode node = (JGoIconicNode)obj;
        myStatusBarLabel.setText("node " + node.getLabel().getText());
      } else if (obj instanceof JGoLink) {
        JGoLink link = (JGoLink)obj;
        JGoIconicNode from = (JGoIconicNode)link.getFromPort().getParent();
        JGoIconicNode to = (JGoIconicNode)link.getToPort().getParent();
        String msg = "link from " + from.getLabel().getText() + " to " + to.getLabel().getText();
        myStatusBarLabel.setText(msg);
      }
    }
  }

  // Modify all of the IconicNodes so that their ports are Visible if users need to be able to
  // draw links between them; otherwise the ports are not Visible, so that a mouse down on the
  // icon does not start drawing a new link.
  // Note that we cause all these document changes to avoid being recorded by the UndoManager.
  void setLinkMode(boolean flag) {
    JGoDocument doc = myView.getDocument();
    doc.setSkipsUndoManager(true);
    JGoListPosition pos = doc.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = doc.getObjectAtPos(pos);
      pos = doc.getNextObjectPosAtTop(pos);
      if (obj instanceof JGoIconicNode) {
        JGoIconicNode node = (JGoIconicNode)obj;
        node.getPort().setVisible(flag);
      }
    }
    doc.setSkipsUndoManager(false);
  }

  private JGoView myView;
  private JGoPalette myPalette;
  private JPanel myPanel;
  private JCheckBox myPointToSelectModeCheckBox;
  private JCheckBox myInsertModeCheckBox;
  private JCheckBox myLinkModeCheckBox;
  private JLabel myStatusBarLabel;
  private int myNodeCounter = 0;
}
