/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;
import java.util.ArrayList;
import com.nwoods.jgo.examples.ListAreaRect;


public class ListArea extends JGoArea
{
  public ListArea()
  {
    super();
  }

  public void initialize()
  {
    // the rectangle is the selected, resizable object
    myRect = new ListAreaRect();
    myRect.setBrush(JGoBrush.lightGray);  //JGoBrush.makeStockBrush(new Color(230, 230, 230))
    myRect.setSelectable(true);
    addObjectAtHead(myRect);

    myBar = new JGoScrollBar();
    myBar.setVertical(isVertical());
    myBar.setSelectable(false);
    addObjectAtTail(myBar);
  }


  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    ListArea newobj = (ListArea)newarea;

    newobj.myVertical = myVertical;
    newobj.myScrollBarOnRight = myScrollBarOnRight;
    newobj.myAlignment = myAlignment;
    newobj.myIconAlignment = myIconAlignment;
    newobj.myFirstItem = myFirstItem;
    newobj.myLastItem = myLastItem;
    // let maximum item size be recalculated (when == -1)
    newobj.myInsets.top = myInsets.top;
    newobj.myInsets.left = myInsets.left;
    newobj.myInsets.bottom = myInsets.bottom;
    newobj.myInsets.right = myInsets.right;
    newobj.mySpacing = mySpacing;
    newobj.myLinePen = myLinePen;
    newobj.myIconSpacing = myIconSpacing;

    super.copyChildren(newarea, env);

    newobj.myRect = (JGoRectangle)env.get(myRect);
    for (int i = 0; i < myVector.size(); i++) {
      JGoObject olditem = (JGoObject)myVector.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myVector.add(newitem);
    }
    for (int i = 0; i < myIcons.size(); i++) {
      JGoObject olditem = (JGoObject)myIcons.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myIcons.add(newitem);
    }
    for (int i = 0; i < myLeftPorts.size(); i++) {
      JGoObject olditem = (JGoObject)myLeftPorts.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myLeftPorts.add(newitem);
    }
    for (int i = 0; i < myRightPorts.size(); i++) {
      JGoObject olditem = (JGoObject)myRightPorts.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myRightPorts.add(newitem);
    }
    newobj.myBar = (JGoScrollBar)env.get(myBar);
  }


  // the parts, other than the items

  // the background rectangle
  public JGoRectangle getRect()
  {
    return myRect;
  }


  // the scroll bar (always present, but may be invisible if not needed)
  public JGoScrollBar getScrollBar()
  {
    return myBar;
  }


  // If true, the scroll bar is on the side, to scroll up and down vertically.
  // Otherwise the scroll bar is on the bottom, to scroll sideways.
  public boolean isVertical()
  {
    return myVertical;
  }

  public void setVertical(boolean v)
  {
    boolean oldVert = myVertical;
    if (oldVert != v) {
      myVertical = v;
      if (getScrollBar() != null) {
        getScrollBar().setVertical(isVertical());
      }
      Insets insets = getInsets();
      if (v) {
        if (isScrollBarOnRight()) {
          insets.right += myBarSize;
          insets.bottom -= myBarSize;
        } else {
          insets.left += myBarSize;
          insets.top -= myBarSize;
        }
      } else {
        if (isScrollBarOnRight()) {
          insets.right -= myBarSize;
          insets.bottom += myBarSize;
        } else {
          insets.left -= myBarSize;
          insets.top += myBarSize;
        }
      }
      layoutChildren(null);
      update(VerticalChanged, (oldVert ? 1 : 0), null);
    }
  }

  // determine where the scroll bar goes--on the right (the default) or
  // on the left.  If the orientation is horizontal instead of vertical,
  // "right" means "bottom", and "left" means "top".
  public boolean isScrollBarOnRight()
  {
    return myScrollBarOnRight;
  }

  public void setScrollBarOnRight(boolean rb)
  {
    boolean oldOnRight = myScrollBarOnRight;
    if (oldOnRight != rb) {
      myScrollBarOnRight = rb;
      Insets insets = getInsets();
      if (rb) {
        if (isVertical()) {
          insets.right += myBarSize;
          insets.left -= myBarSize;
        } else {
          insets.bottom += myBarSize;
          insets.top -= myBarSize;
        }
      } else {
        if (isVertical()) {
          insets.right -= myBarSize;
          insets.left += myBarSize;
        } else {
          insets.bottom -= myBarSize;
          insets.top += myBarSize;
        }
      }
      layoutChildren(null);
      update(ScrollBarOnRightChanged, (oldOnRight ? 1 : 0), null);
    }
  }

  public JGoPen getLinePen()
  {
    return myLinePen;
  }

  public void setLinePen(JGoPen pen)
  {
    JGoPen oldPen = myLinePen;
    if (oldPen != pen) {
      myLinePen = pen;
      layoutChildren(null);
      update(LinePenChanged, 0, oldPen);
    }
  }


  // extra space around the edges
  // the space should include room for the scroll bar
  public Insets getInsets()
  {
    return myInsets;
  }

  public void setInsets(Insets x)
  {
    Insets s = myInsets;
    if (!s.equals(x)) {
      Insets oldInsets = new Insets(s.top, s.left, s.bottom, s.right);
      myInsets.top = x.top;
      myInsets.left = x.left;
      myInsets.bottom = x.bottom;
      myInsets.right = x.right;
      layoutChildren(null);
      update(InsetsChanged, 0, oldInsets);
    }
  }

  // extra space between items
  public int getSpacing()
  {
    return mySpacing;
  }

  public void setSpacing(int newspace)
  {
    int oldSpacing = mySpacing;
    if (oldSpacing != newspace) {
      mySpacing = newspace;
      layoutChildren(null);
      update(SpacingChanged, oldSpacing, null);
    }
  }


  /**
   * Return how items are aligned in the list.
   */
  public int getAlignment()
  {
    return myAlignment;
  }

  /**
   * Set how items are aligned. The following
   * are possible alignment values:
   * <ul>
   * <li><b>ALIGN_CENTER</b> - center each item
   * <li><b>ALIGN_LEFT</b> - left justify each item
   * <li><b>ALIGN_RIGHT</b> - right justify each item
   * </ul>
   *
   * @param align one of the above alignment types
   */
  public void setAlignment(int align)
  {
    int oldAlignment = myAlignment;
    if (oldAlignment != align) {
      myAlignment = align;
      layoutChildren(null);
      update(AlignmentChanged, oldAlignment, null);
    }
  }


  /**
   * Return how icons are aligned in the list.
   */
  public int getIconAlignment()
  {
    return myIconAlignment;
  }

  /**
   * Set how icon items are aligned. The following
   * are possible alignment values:
   * <ul>
   * <li><b>ALIGN_CENTER</b> - center each item
   * <li><b>ALIGN_LEFT</b> - left justify each item
   * <li><b>ALIGN_RIGHT</b> - right justify each item
   * </ul>
   * If both the item and the icon are ALIGN_CENTERed,
   * The icon will appear just to the left or top of the item.
   *
   * @param align one of the above alignment types
   */
  public void setIconAlignment(int align)
  {
    int oldAlignment = myIconAlignment;
    if (oldAlignment != align) {
      myIconAlignment = align;
      layoutChildren(null);
      update(IconAlignmentChanged, oldAlignment, null);
    }
  }


  // extra space between the icon and the item
  public int getIconSpacing()
  {
    return myIconSpacing;
  }

  public void setIconSpacing(int newspace)
  {
    int oldSpacing = myIconSpacing;
    if (oldSpacing != newspace) {
      myIconSpacing = newspace;
      layoutChildren(null);
      update(IconSpacingChanged, oldSpacing, null);
    }
  }


  // what's the index of the first item to be shown? (if there's room)
  public int getFirstVisibleIndex()
  {
    return myFirstItem;
  }

  // start displaying at the I'th index item
  public void setFirstVisibleIndex(int i)
  {
    int oldIndex = myFirstItem;
    if (i >= 0 &&
        i <= getNumItems() &&
        oldIndex != i) {
      myFirstItem = i;
      layoutChildren(null);
      update(FirstVisibleIndexChanged, oldIndex, null);
    }
  }


  // once the items have been laid out, what was the index of the
  // last item shown?
  public int getLastVisibleIndex()
  {
    return myLastItem;
  }


  // how many items are in the list (both invisible and visible)
  public int getNumItems()
  {
    return myVector.size();
  }


  public JGoObject getItem(int i)
  {
    if (i < 0 || i >= myVector.size())
      return null;
    else
      return (JGoObject)myVector.get(i);
  }

  // this is a no-op if there was no item at index I to begin with
  public void setItem(int i, JGoObject obj)
  {
    if (i < 0 || i >= getNumItems()) return;
    JGoObject oldobj = getItem(i);
    if (oldobj == null) return;
    if (oldobj != obj) {
      removeObject(oldobj);
      myVector.set(i, obj);
      adjustMaxItemSize(obj, oldobj.getWidth(), oldobj.getHeight(), getIcon(i));

      if (obj != null) {
        addObjectAtTail(obj);

        obj.setResizable(false);

        JGoRectangle lrect = getRect();
        if (i < getFirstVisibleIndex()) {
          obj.setVisible(false);
          obj.setTopLeft(lrect.getLeft(), lrect.getTop());
        } else if (i <= getLastVisibleIndex()) {
          layoutChildren(obj);
        } else {
          obj.setVisible(false);
          if (isVertical())
            obj.setTopLeft(lrect.getLeft(),
                           lrect.getTop() + lrect.getHeight() - obj.getHeight());
          else
            obj.setTopLeft(lrect.getLeft() + lrect.getWidth() - obj.getWidth(),
                           lrect.getTop());
        }
      }
      update(ItemChanged, i, oldobj);
    }
  }

  // return index of item object, or else -1
  public int findItem(JGoObject obj)
  {
    for (int i = 0; i < getNumItems(); i++) {
      JGoObject item = getItem(i);
      if (item == obj) {
        return i;
      }
    }
    return -1;
  }


  // note that for generality, the objects associated with
  // items in the list are not necessarily JGoPorts or JGoImages,
  // but could be any object

  public JGoObject getIcon(int i)
  {
    if (i < 0 || i >= myIcons.size())
      return null;
    else
      return (JGoObject)myIcons.get(i);
  }

  // this is a no-op if there is nothing at index I
  public void setIcon(int i, JGoObject obj)
  {
    JGoObject oldobj = getIcon(i);
    if (oldobj != obj) {
      if (oldobj != null) {
        if (obj != null)
          obj.setBoundingRect(oldobj.getBoundingRect());
        removeObject(oldobj);
      }
      myIcons.set(i, obj);
      if (obj != null) {
        obj.setResizable(false);
        addObjectAtTail(obj);
        if (oldobj != null) {
          adjustMaxItemSize(obj, oldobj.getWidth(), oldobj.getHeight(), getItem(i));
        }
      }
      update(IconChanged, i, oldobj);
    }
  }


  public JGoObject getLeftPort(int i)
  {
    if (i < 0 || i >= myLeftPorts.size())
      return null;
    else
      return (JGoObject)myLeftPorts.get(i);
  }

  public void setLeftPort(int i, JGoObject obj)
  {
    JGoObject oldobj = getLeftPort(i);
    if (oldobj != obj) {
      if (oldobj != null) {
        if (obj != null)
          obj.setBoundingRect(oldobj.getBoundingRect());
        removeObject(oldobj);
      }
      myLeftPorts.set(i, obj);
      if (obj != null) {
        obj.setSelectable(false);
        obj.setDraggable(false);
        obj.setResizable(false);
        addObjectAtTail(obj);
      }
      update(LeftPortChanged, i, oldobj);
    }
  }

  public JGoObject getRightPort(int i)
  {
    if (i < 0 || i >= myRightPorts.size())
      return null;
    else
      return (JGoObject)myRightPorts.get(i);
  }

  public void setRightPort(int i, JGoObject obj)
  {
    JGoObject oldobj = getRightPort(i);
    if (oldobj != obj) {
      if (oldobj != null) {
        if (obj != null)
          obj.setBoundingRect(oldobj.getBoundingRect());
        removeObject(oldobj);
      }
      myRightPorts.set(i, obj);
      if (obj != null) {
        obj.setSelectable(false);
        obj.setDraggable(false);
        obj.setResizable(false);
        addObjectAtTail(obj);
      }
      update(RightPortChanged, i, oldobj);
    }
  }


  final public void addItem(JGoObject item, JGoObject leftport, JGoObject rightport, JGoObject icon)
  {
    insertItem(getNumItems(), item, leftport, rightport, icon);
  }

  public void insertItem(int i, JGoObject item, JGoObject leftport, JGoObject rightport, JGoObject icon)
  {
    if (i < 0 || i > getNumItems())  // == getNumItems() is OK--means add at end
      return;

    JGoRectangle lrect = getRect();

    myVector.add(i, item);
    if (item != null) {
      item.setTopLeft(lrect.getLeft(), lrect.getTop());
      item.setResizable(false);
      adjustMaxItemSize(item, -1, -1, icon);
      addObjectAtTail(item);
    }

    myIcons.add(i, icon);
    if (icon != null) {
      icon.setTopLeft(lrect.getLeft(), lrect.getTop());
      icon.setResizable(false);
      addObjectAtTail(icon);
    }

    myLeftPorts.add(i, leftport);
    if (leftport != null) {
      leftport.setTopLeft(lrect.getLeft(), lrect.getTop());
      leftport.setSelectable(false);
      leftport.setDraggable(false);
      leftport.setResizable(false);
      addObjectAtTail(leftport);
    }
    myRightPorts.add(i, rightport);
    if (rightport != null) {
      rightport.setTopLeft(lrect.getLeft(), lrect.getTop());
      rightport.setSelectable(false);
      rightport.setDraggable(false);
      rightport.setResizable(false);
      addObjectAtTail(rightport);
    }

    if (i < getFirstVisibleIndex()) {
      if (item != null) {
        item.setVisible(false);
      }
      // top-left already set to lrect.getLeft(), lrect.getTop()
      updateScrollBar();
    } else if (i <= getLastVisibleIndex()) {
      layoutChildren(null);
    } else {
      if (item != null) {
        item.setVisible(false);
        if (isVertical())
          item.setTopLeft(lrect.getLeft(),
                         lrect.getTop() + lrect.getHeight() - item.getHeight());
        else
          item.setTopLeft(lrect.getLeft() + lrect.getWidth() - item.getWidth(),
                         lrect.getTop());
      }
      updateScrollBar();
    }

    JGoObject[] newvals = new JGoObject[4];
    newvals[0] = item;
    newvals[1] = leftport;
    newvals[2] = rightport;
    newvals[3] = icon;
    update(ItemInsertedChanged, i, newvals);
  }

  public void removeItem(int i)
  {
    if (i < 0 || i >= getNumItems())
      return;

    JGoObject leftport = getLeftPort(i);
    myLeftPorts.remove(i);
    if (leftport != null)
      super.removeObject(leftport);

    JGoObject rightport = getRightPort(i);
    myRightPorts.remove(i);
    if (rightport != null)
      super.removeObject(rightport);

    JGoObject icon = getIcon(i);
    myIcons.remove(i);
    if (icon != null)
      super.removeObject(icon);

    JGoObject item = getItem(i);
    if (item != null) {
      myVector.remove(i);
      super.removeObject(item);
      adjustMaxItemSize(null, item.getWidth(), item.getHeight(), null);
    }
    if (i <= getLastVisibleIndex())
      layoutChildren(null);
    else
      updateScrollBar();

    JGoObject[] oldvals = new JGoObject[4];
    oldvals[0] = item;
    oldvals[1] = leftport;
    oldvals[2] = rightport;
    oldvals[3] = icon;
    update(ItemRemovedChanged, i, oldvals);
  }

  // this overrides JGoArea.removeObject(JGoObject)
  public void removeObject(JGoObject obj)
  {
    boolean found = false;
    for (int i = 0; i < getNumItems(); i++) {
      JGoObject item = getItem(i);
      if (item == obj) {
        removeItem(i);
        found = true;
        break;
      }
    }
    if (!found)
      super.removeObject(obj);
  }


  // don't have a way of removing a "port" object, unless
  // you call set[Left/Right]Port(i, null);

  // and of course, you can't have a port unless it's associated with an item in the list


  // this assumes that the getRect() value is the basis for all item
  // layout decisions
  public void layoutChildren(JGoObject childchanged)
  {
    if (isInitializing()) return;
    setInitializing(true);

    Insets insets = getInsets();

    JGoObject lrect = getRect();
    if (lrect == null) return;  // not yet initialized

    sendObjectToBack(lrect);

    myLastItem = getFirstVisibleIndex();  // remember last visible item index

    int left = lrect.getLeft() + insets.left;
    int top = lrect.getTop() + insets.top;
    int width = lrect.getWidth() - insets.left - insets.right;
    int height = lrect.getHeight() - insets.top - insets.bottom;

    int penwidth = 0;
    if (getLinePen() != null)
      penwidth = getLinePen().getWidth();

    if (isVertical()) {  // vertical is the normal case
      int h = 0;  // height of visible items so far
      for (int i = 0; i < getNumItems(); i++) {
        JGoObject item = getItem(i);
        JGoObject icon = getIcon(i);
        JGoObject lport = getLeftPort(i);
        JGoObject rport = getRightPort(i);

        if (i < getFirstVisibleIndex()) {
          if (item != null) {
            item.setVisible(false);
            item.setTopLeft(left, top);
          }
          if (icon != null) {
            icon.setVisible(false);
            icon.setTopLeft(left, top);
          }
          if (lport != null) {
            lport.setVisible(false);
            lport.setTopLeft(left, top);
          }
          if (rport != null) {
            rport.setVisible(false);
            rport.setTopLeft(left + width - rport.getWidth(), top);
          }
        } else {
          int size = getItemSize(i);
          if (h + size <= height) {
            layoutItem(i, left, top + h, width, size);
            h += size;
            h += Math.max(penwidth, getSpacing());
            myLastItem = i;  // remember last visible item index
          } else {
            if (item != null) {
              item.setVisible(false);
              item.setTopLeft(left, top + height - size);
            }
            if (icon != null) {
              icon.setVisible(false);
              icon.setTopLeft(left, top + height - size);
            }
            if (lport != null) {
              lport.setVisible(false);
              lport.setTopLeft(left, top + height - lport.getHeight());
            }
            if (rport != null) {
              rport.setVisible(false);
              rport.setTopLeft(left + width - rport.getWidth(), top + height - rport.getHeight());
            }
            h = height+1;  // don't position anything else visibly
          }
        }
      }
      JGoScrollBar sbar = getScrollBar();
      if (sbar != null) {
        if (isScrollBarOnRight()) {
          sbar.setBoundingRect(lrect.getLeft() + lrect.getWidth() - myBarSize,
                               lrect.getTop(),
                               myBarSize, lrect.getHeight());
        } else {
          sbar.setBoundingRect(lrect.getLeft(),
                               lrect.getTop(),
                               myBarSize, lrect.getHeight());
        }
        updateScrollBar();
      }
    } else {  // items arranged horizontally; scrollbar is horizontal too
      int w = 0;  // width of visible items so far
      for (int i = 0; i < getNumItems(); i++) {
        JGoObject item = getItem(i);
        JGoObject icon = getIcon(i);
        JGoObject lport = getLeftPort(i);
        JGoObject rport = getRightPort(i);

        if (i < getFirstVisibleIndex()) {
          if (item != null) {
            item.setVisible(false);
            item.setTopLeft(left, top);
          }
          if (icon != null) {
            icon.setVisible(false);
            icon.setTopLeft(left, top);
          }
          if (lport != null) {
            lport.setVisible(false);
            lport.setTopLeft(left, top);
          }
          if (rport != null) {
            rport.setVisible(false);
            rport.setTopLeft(left, top + height - rport.getHeight());
          }
        } else {
          int size = getItemSize(i);
          if (w + size <= width) {
            layoutItem(i, left + w, top, size, height);
            w += size;
            w += Math.max(penwidth, getSpacing());
            myLastItem = i;  // remember last visible item index
          } else {
            if (item != null) {
              item.setVisible(false);
              item.setTopLeft(left + width - size, top);
            }
            if (icon != null) {
              icon.setVisible(false);
              icon.setTopLeft(left + width - size, top);
            }
            if (lport != null) {
              lport.setVisible(false);
              lport.setTopLeft(left + width - lport.getWidth(), top);
            }
            if (rport != null) {
              rport.setVisible(false);
              rport.setTopLeft(left + width - rport.getWidth(), top + height - rport.getHeight());
            }
            w = width+1;  // don't position anything else visibly
          }
        }
      }
      JGoScrollBar sbar = getScrollBar();
      if (sbar != null) {
        if (isScrollBarOnRight()) {
          sbar.setBoundingRect(lrect.getLeft(),
                               lrect.getTop() + lrect.getHeight() - myBarSize,
                               lrect.getWidth(), myBarSize);
        } else {
          sbar.setBoundingRect(lrect.getLeft(),
                               lrect.getTop(),
                               lrect.getWidth(), myBarSize);
        }
        updateScrollBar();
      }
    }
    setInitializing(false);
  }

  // return the height (or width if vertical) needed for item I
  protected int getItemSize(int i)
  {
    int size = 0;

    JGoObject item = getItem(i);
    if (item != null) {
      if (isVertical())
        size = item.getHeight();
      else
        size = item.getWidth();
    }

    JGoObject icon = getIcon(i);
    if (icon != null) {
      if (isVertical())
        size = Math.max(size, icon.getHeight());
      else
        size = Math.max(size, icon.getWidth());
    }

    JGoObject lport = getLeftPort(i);
    if (lport != null) {
      if (isVertical())
        size = Math.max(size, lport.getHeight());
      else
        size = Math.max(size, lport.getWidth());
    }

    JGoObject rport = getRightPort(i);
    if (rport != null) {
      if (isVertical())
        size = Math.max(size, rport.getHeight());
      else
        size = Math.max(size, rport.getWidth());
    }

    return size;
  }

  // X, Y, WIDTH, and HEIGHT specify the rectangle where we should
  // position the I'th item and icon.
  // The ports should be positioned outside of the ListArea's Rect.
  protected void layoutItem(int i, int x, int y, int width, int height)
  {
    JGoObject item = getItem(i);
    JGoObject icon = getIcon(i);

    // determine the positions of the object and the icon
    // relative to the given rectangle
    int itemsz = 0;
    int iconsz = 0;
    if (item != null) {
      if (isVertical())
        itemsz = item.getWidth();
      else
        itemsz = item.getHeight();
    }
    if (icon != null) {
      if (isVertical())
        iconsz = icon.getWidth();
      else
        iconsz = icon.getHeight();
    }
    int limit = 0;
    if (isVertical())
      limit = width;
    else
      limit = height;

    int align = 0;
    int iconalign = 0;
    int iconsep = getIconSpacing();
    switch (getAlignment()) {
      default:
      case ALIGN_LEFT: {
        align = 0;
        switch (getIconAlignment()) {
          default:
          case ALIGN_LEFT:
            iconalign = 0;
            align = iconsz + iconsep;
            break;
          case ALIGN_CENTER:
            iconalign = limit/2 - iconsz/2;
            iconalign = Math.max(iconalign, itemsz + iconsep);  // don't overlap
            break;
          case ALIGN_RIGHT:
            iconalign = limit - iconsz;
            break;
        }
        break; }
      case ALIGN_CENTER: {
        align = limit/2 - itemsz/2;
        switch (getIconAlignment()) {
          default:
          case ALIGN_LEFT:
            iconalign = 0;
            align = Math.max(align + iconsep, iconsz);  // don't overlap
            break;
          case ALIGN_CENTER:
            iconalign = limit/2 - (iconsz+itemsz+iconsep)/2;
            align = iconalign + iconsz + iconsep;
            break;
          case ALIGN_RIGHT:
            iconalign = limit - iconsz;
            align = Math.min(align, limit - iconsz - itemsz - iconsep);  // don't overlap
            break;
        }
        break; }
      case ALIGN_RIGHT: {
        align = limit - itemsz;
        switch (getIconAlignment()) {
          default:
          case ALIGN_LEFT:
            iconalign = 0;
            break;
          case ALIGN_CENTER:
            iconalign = limit/2 - iconsz/2;
            iconalign = Math.min(iconalign, limit - itemsz - iconsz - iconsep);  // don't overlap
            break;
          case ALIGN_RIGHT:
            iconalign = limit - iconsz;
            align = iconalign - itemsz - iconsep;
            break;
        }
        break; }
    }

    if (item != null) {
      item.setVisible(true);
      if (isVertical())
        item.setTopLeft(x + align, y + height/2 - item.getHeight()/2);
      else
        item.setTopLeft(x + width/2 - item.getWidth()/2, y + align);
    }
    if (icon != null) {
      icon.setVisible(true);
      if (isVertical())
        icon.setTopLeft(x + iconalign, y + height/2 - icon.getHeight()/2);
      else
        icon.setTopLeft(x + width/2 - icon.getWidth()/2, y + iconalign);
    }

    JGoObject lrect = getRect();
    if (lrect == null) return;  // not yet initialized

    int rectleft = lrect.getLeft();
    int recttop = lrect.getTop();
    int rectwidth = lrect.getWidth();
    int rectheight = lrect.getHeight();

    JGoObject lport = getLeftPort(i);
    if (lport != null) {
      lport.setVisible(true);
      if (isVertical()) {
        int xpos = rectleft - lport.getWidth();
        int ypos = y + height/2 - lport.getHeight()/2;
        lport.setTopLeft(xpos, ypos);
      } else {
        int xpos = x + width/2 - lport.getWidth()/2;
        int ypos = recttop - lport.getHeight();
        lport.setTopLeft(xpos, ypos);
      }
    }

    JGoObject rport = getRightPort(i);
    if (rport != null) {
      rport.setVisible(true);
      if (isVertical()) {
        int xpos = rectleft + rectwidth;
        int ypos = y + height/2 - rport.getHeight()/2;
        rport.setTopLeft(xpos, ypos);
      } else {
        int xpos = x + width/2 - rport.getWidth()/2;
        int ypos = recttop + rectheight;
        rport.setTopLeft(xpos, ypos);
      }
    }
  }


  public void paint(Graphics2D g, JGoView view)
  {
    super.paint(g, view);

    int penwidth = 0;
    if (getLinePen() != null)
      penwidth = getLinePen().getWidth();
    if (penwidth == 0)
      return;

    JGoObject lrect = getRect();
    if (lrect == null) return;  // not yet initialized

    Insets insets = getInsets();

    int rectleft = lrect.getLeft();
    int recttop = lrect.getTop();
    int rectwidth = lrect.getWidth();
    int rectheight = lrect.getHeight();

    int left = rectleft + insets.left;
    int top = recttop + insets.top;
    int width = rectwidth - insets.left - insets.right;
    int height = rectheight - insets.top - insets.bottom;

    int limit = 0;
    if (isVertical())
      limit = height;
    else
      limit = width;

    int s = 0;  // height/width of visible items so far
    for (int i = getFirstVisibleIndex(); i < getLastVisibleIndex(); i++) {
      int size = getItemSize(i);
      if (s + size <= limit) {
        s += size;
        int sep = Math.max(penwidth, getSpacing());
        if (s + sep <= limit) {
          if (isVertical())
            JGoDrawable.drawLine(g, getLinePen(), rectleft,                top + s + sep/2,
                                                  rectleft + rectwidth, top + s + sep/2);
          else
            JGoDrawable.drawLine(g, getLinePen(), left + s + sep/2, recttop,
                                                  left + s + sep/2, recttop + rectheight);
        }
        s += sep;
      }
    }
  }


  public void updateScrollBar()
  {
    JGoScrollBar bar = getScrollBar();
    if (bar != null) {
      bar.setVertical(isVertical());
      if (getFirstVisibleIndex() == 0 &&
          getLastVisibleIndex() == getNumItems()-1) {
        bar.setVisible(false);
      } else {
        bar.setVisible(true);
        boolean oldinit = isInitializing();
        setInitializing(true);
        bar.setValues(getFirstVisibleIndex(),
                      getLastVisibleIndex()-getFirstVisibleIndex()+1,
                      0, getNumItems(), 1,
                      Math.max(getLastVisibleIndex()-getFirstVisibleIndex(), 1));
        setInitializing(oldinit);
      }
    }
  }

  // this gets the notification from the JGoScrollBar when the scroll bar
  // value, representing the first visible index, has changed
  public void update(int hint, int prevInt, Object prevVal)
  {
    if (hint == JGoScrollBar.ChangedScrollBarValue) {
      if (getScrollBar() != null && !isInitializing()) {
        setFirstVisibleIndex(getScrollBar().getValue());
      }
    } else {
      super.update(hint, prevInt, prevVal);
    }
  }


  // since setVisible() doesn't automatically call setVisible() on all the
  // children, we need to do this manually to handle the scroll bar
  public void setVisible(boolean bFlag)
  {
    if (getScrollBar() != null)
      getScrollBar().setVisible(bFlag);
    super.setVisible(bFlag);
  }


  // the JGoRectangle will act as the selected object, not this area
  public JGoObject redirectSelection()
  {
    return getRect();
  }


  // calculate the minimum size for the ListAreaRect, as determined
  // by the maximum item size plus the insets
  public Dimension getMinimumRectSize()
  {
    // figure out the smallest rectangle that will hold any item
    Dimension maxItem = getMaxItemSize();
    // now account for insets on all sides, including the scroll bar
    Insets insets = getInsets();
    int minw = maxItem.width + insets.left + insets.right;
    int minh = maxItem.height + insets.top + insets.bottom;
    return new Dimension(minw, minh);
  }

  public Dimension getMinimumSize()
  {
    // first account for the minimum ListAreaRect size
    Dimension minRect = getMinimumRectSize();
    // now account for any ports
    int leftwidth = getMaxPortWidth(true);
    int rightwidth = getMaxPortWidth(false);
    if (isVertical())
      return new Dimension(minRect.width + leftwidth + rightwidth, minRect.height);
    else
      return new Dimension(minRect.width, minRect.height + leftwidth + rightwidth);
  }

  // constrain to the minimum width and height
  public void setBoundingRect(int left, int top, int width, int height)
  {
    Dimension minSize = getMinimumSize();
    super.setBoundingRect(left, top,
                          Math.max(width, minSize.width),
                          Math.max(height, minSize.height));
  }

  protected void rescaleChildren(Rectangle prevRect)
  {
    // set the new geometry for the Rect
    if (getRect() != null) {
      Rectangle thisRect = getBoundingRect();
      // now account for any ports
      int leftwidth = getMaxPortWidth(true);
      int rightwidth = getMaxPortWidth(false);
      if (isVertical())
        getRect().setBoundingRect(thisRect.x + leftwidth, thisRect.y,
                                  thisRect.width - leftwidth - rightwidth, thisRect.height);
      else
        getRect().setBoundingRect(thisRect.x, thisRect.y + leftwidth,
                                  thisRect.width, thisRect.height - leftwidth - rightwidth);
    }
  }

  // ListArea assumes each "port" object is smaller than every item
  protected boolean geometryChangeChild(JGoObject child, Rectangle prevRect)
  {
    if (isInitializing()) return false;
    int idx = -1;
    if (child != getRect() &&
        (child.getWidth() != prevRect.width || child.getHeight() != prevRect.height) &&
        (idx = findItem(child)) >= 0) {
      // handle an item that changed size (ignore the scroll bar and ports)
      adjustMaxItemSize(child, prevRect.width, prevRect.height, getIcon(idx));
// if you want the width (or height if horizontal) to automatically shrink as
// the widest (or tallest) child becomes smaller than the current rectangle's width,
// uncomment out the following IF statement:
/*
      if (getRect() != null) {
        Dimension maxitem = getMaxItemSize();
        if (isVertical())
          getRect().setWidth(maxitem.width);
        else
          getRect().setHeight(maxitem.height);
      }
*/
      // this will call layoutChildren:
      return super.geometryChangeChild(child, prevRect);
    } else if (child == getRect()) {
      return super.geometryChangeChild(child, prevRect);
    } else {
      return false;
    }
  }

  public void adjustMaxItemSize(JGoObject obj, int oldw, int oldh, JGoObject other)
  {
    int neww = -1;
    int newh = -1;
    if (obj != null) {
      neww = obj.getWidth();
      newh = obj.getHeight();
    }
    int oldtw = oldw;
    int oldth = oldh;
    int newtw = neww;
    int newth = newh;
    if (other != null) {
      oldtw += other.getWidth();
      oldth += other.getHeight();
      newtw += other.getWidth();
      newth += other.getHeight();
      int space = getIconSpacing();
      if (isVertical()) {
        oldtw += space;
        newtw += space;
      } else {
        oldth += space;
        newth += space;
      }
    }
    Insets insets = getInsets();
    if (myMaxItemSize.width > -1) {  // valid max width
      if (neww > oldw &&                 // getting wider
          newtw > myMaxItemSize.width) {  // wider than old max width
        myMaxItemSize.width = newtw;      // new max width
        // make sure there's room for the largest item
        int newrectwidth = newtw + insets.left + insets.right;
        if (getRect().getWidth() < newrectwidth)
          getRect().setWidth(newrectwidth);
      } else if (neww < oldw &&                  // getting narrower
                 oldtw == myMaxItemSize.width) {  // used to be same as max
        myMaxItemSize.width = -1;                // recalculate
      }
    }
    if (myMaxItemSize.height > -1) {  // valid max height
      if (newh > oldh &&                  // getting taller
          newth > myMaxItemSize.height) {  // taller than old max height
        myMaxItemSize.height = newth;      // new max height
        // make sure there's room for the largest item
        int newrectheight = newth + insets.top + insets.bottom;
        if (getRect().getHeight() < newrectheight)
          getRect().setHeight(newrectheight);
      } else if (newh < oldh &&                   // getting shorter
                 oldth == myMaxItemSize.height) {  // used to be same as max
        myMaxItemSize.height = -1;                // recalculate
      }
    }
  }

  // Return the largest width of all items, and the largest height of
  // all items.  There might not be any item with both the largest
  // width and height.
  public Dimension getMaxItemSize()
  {
    if (myMaxItemSize.width < 0 ||
        myMaxItemSize.height < 0) {
      myMaxItemSize.width = -1;
      myMaxItemSize.height = -1;
      for (int i = 0; i < myVector.size(); i++) {
        JGoObject item = (JGoObject)myVector.get(i);

        int itemw = 0;
        int itemh = 0;
        if (item != null) {
          itemw = item.getWidth();
          itemh = item.getHeight();
          if (itemw > myMaxItemSize.width)
            myMaxItemSize.width = itemw;
          if (itemh > myMaxItemSize.height)
            myMaxItemSize.height = itemh;
        }

        JGoObject icon = null;
        if (i < myIcons.size())
          icon = (JGoObject)myIcons.get(i);

        int iconw = 0;
        int iconh = 0;
        if (icon != null) {
          iconw = icon.getWidth();
          iconh = icon.getHeight();
          if (iconw > myMaxItemSize.width)
            myMaxItemSize.width = iconw;
          if (iconh > myMaxItemSize.height)
            myMaxItemSize.height = iconh;
        }
        if (item != null && icon != null) {
          if (isVertical()) {
            int sum = itemw + iconw + getIconSpacing();
            if (sum > myMaxItemSize.width)
              myMaxItemSize.width = sum;
          } else {
            int sum = itemh + iconh + getIconSpacing();
            if (sum > myMaxItemSize.height)
              myMaxItemSize.height = sum;
          }
        }
      }
    }
    return myMaxItemSize;
  }

  // Return the maximum width for all the ports on a given side
  // (when the orientation is horizontal, this returns the maximum height)
  public int getMaxPortWidth(boolean left)
  {
    int width = 0;
    ArrayList ports = (left ? myLeftPorts : myRightPorts);
    for (int i = 0; i < ports.size(); i++) {
      JGoObject p = (JGoObject)ports.get(i);
      if (p == null) continue;
      if (isVertical()) {
        width = Math.max(width, p.getWidth());
      } else {
        width = Math.max(width, p.getHeight());
      }
    }
    return width;
  }


  // undo/redo support

  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getFlags()) {
      case VerticalChanged:
        e.setNewValueBoolean(isVertical());
        return;
      case ScrollBarOnRightChanged:
        e.setNewValueBoolean(isScrollBarOnRight());
        return;
      case LinePenChanged:
        e.setNewValue(getLinePen());
        return;
      case InsetsChanged: {
        // make sure value gets copied so it doesn't get clobbered later
        Insets s = getInsets();
        e.setNewValue(new Insets(s.top, s.left, s.bottom, s.right));
        return; }
      case SpacingChanged:
        e.setNewValueInt(getSpacing());
        return;
      case AlignmentChanged:
        e.setNewValueInt(getAlignment());
        return;
      case FirstVisibleIndexChanged:
        e.setNewValueInt(getFirstVisibleIndex());
        return;
      case ItemChanged:
        e.setNewValue(getItem(e.getOldValueInt()));
        return;
      case LeftPortChanged:
        e.setNewValue(getLeftPort(e.getOldValueInt()));
        return;
      case RightPortChanged:
        e.setNewValue(getRightPort(e.getOldValueInt()));
        return;
      case ItemInsertedChanged:
        // the old value information indicates where it had been inserted
        return;
      case ItemRemovedChanged:
        // the old value information indicates where and what had been removed
        return;
      case IconChanged:
        e.setNewValue(getIcon(e.getOldValueInt()));
        return;
      case IconAlignmentChanged:
        e.setNewValueInt(getIconAlignment());
        return;
      case IconSpacingChanged:
        e.setNewValueInt(getIconSpacing());
        return;
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getFlags()) {
      case VerticalChanged:
        myVertical = e.getValueBoolean(undo);
        return;
      case ScrollBarOnRightChanged:
        myScrollBarOnRight = e.getValueBoolean(undo);
        return;
      case LinePenChanged:
        myLinePen = (JGoPen)e.getValue(undo);
        update();
        return;
      case InsetsChanged:
        Insets x = (Insets)e.getValue(undo);
        myInsets.top = x.top;
        myInsets.left = x.left;
        myInsets.bottom = x.bottom;
        myInsets.right = x.right;
        return;
      case SpacingChanged:
        mySpacing = e.getValueInt(undo);
        return;
      case AlignmentChanged:
        myAlignment = e.getValueInt(undo);
        return;
      case FirstVisibleIndexChanged:
        myFirstItem = e.getValueInt(undo);
        return;
      case ItemChanged:
        setItem(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case LeftPortChanged:
        setLeftPort(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case RightPortChanged:
        setRightPort(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case ItemInsertedChanged:
        if (undo) {
          removeItem(e.getOldValueInt());
        } else {
          JGoObject[] newvals = (JGoObject[])e.getOldValue();
          insertItem(e.getOldValueInt(), newvals[0], newvals[1], newvals[2], newvals[3]);
        }
        return;
      case ItemRemovedChanged:
        if (undo) {
          JGoObject[] newvals = (JGoObject[])e.getOldValue();
          insertItem(e.getOldValueInt(), newvals[0], newvals[1], newvals[2], newvals[3]);
        } else {
          removeItem(e.getOldValueInt());
        }
        return;
      case IconChanged:
        setIcon(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case IconAlignmentChanged:
        myIconAlignment = e.getValueInt(undo);
        return;
      case IconSpacingChanged:
        myIconSpacing = e.getValueInt(undo);
        return;
      default:
        super.changeValue(e, undo);
        return;
    }
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("bar")) {
      myBar = (JGoScrollBar)referencedObject;
    }
    else if (attr.equals("rect")) {
      myRect = (JGoRectangle)referencedObject;
    }
    else if (attr.equals("vectorelements")) {
      myVector.add(referencedObject);
    }
    else if (attr.equals("icons")) {
      myIcons.add(referencedObject);
    }
    else if (attr.equals("leftports")) {
      myLeftPorts.add(referencedObject);
    }
    else if (attr.equals("rightports")) {
      myRightPorts.add(referencedObject);
    }
    else if (attr.equals("linepen")) {
      myLinePen = (JGoPen)referencedObject;
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add ListArea element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jListArea = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.ListArea", jGoElementGroup);
      jListArea.setAttribute("alignment", Integer.toString(myAlignment));
      jListArea.setAttribute("firstitem", Integer.toString(myFirstItem));
      jListArea.setAttribute("iconalignment", Integer.toString(myIconAlignment));
      jListArea.setAttribute("iconspacing", Integer.toString(myIconSpacing));
      jListArea.setAttribute("lastitem", Integer.toString(myLastItem));
      jListArea.setAttribute("scrollbaronright",
                             myScrollBarOnRight ? "true" : "false");
      jListArea.setAttribute("spacing", Integer.toString(mySpacing));
      jListArea.setAttribute("vertical", myVertical ? "true" : "false");
      jListArea.setAttribute("insetleft", Integer.toString(myInsets.left));
      jListArea.setAttribute("insetright", Integer.toString(myInsets.right));
      jListArea.setAttribute("insettop", Integer.toString(myInsets.top));
      jListArea.setAttribute("insetbottom", Integer.toString(myInsets.bottom));

      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myBar != null) {
        svgDoc.registerReferencingNode(jListArea, "bar", myBar);
      }
      if (myRect != null) {
        svgDoc.registerReferencingNode(jListArea, "rect", myRect);
      }
      for (int i = 0; i < myVector.size(); i++) {
        svgDoc.registerReferencingNode(jListArea, "vectorelements",
                                       (JGoObject) myVector.get(i));
      }
      for (int i = 0; i < myIcons.size(); i++) {
        svgDoc.registerReferencingNode(jListArea, "icons",
                                       (JGoObject) myIcons.get(i));
      }
      for (int i = 0; i < myLeftPorts.size(); i++) {
        svgDoc.registerReferencingNode(jListArea, "leftports",
                                       (JGoObject) myLeftPorts.get(i));
      }
      for (int i = 0; i < myRightPorts.size(); i++) {
        svgDoc.registerReferencingNode(jListArea, "rightports",
                                       (JGoObject) myRightPorts.get(i));
      }
      // The following objects are NOT children of this area and must be
      // explicitly written out
      if (myLinePen != null) {
        if (!svgDoc.isRegisteredReference(myLinePen)) {
          jListArea.setAttribute("embeddedlinepen", "true");
          DomElement subGroup = svgDoc.createElement("g");
          jListArea.appendChild(subGroup);
          myLinePen.SVGWriteObject(svgDoc, subGroup);
        }
        svgDoc.registerReferencingNode(jListArea, "linepen", myLinePen);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a ListArea element
      myAlignment = Integer.parseInt(jGoChildElement.getAttribute("alignment"));
      myFirstItem = Integer.parseInt(jGoChildElement.getAttribute("firstitem"));
      myIconAlignment = Integer.parseInt(jGoChildElement.getAttribute("iconalignment"));
      myIconSpacing = Integer.parseInt(jGoChildElement.getAttribute("iconspacing"));
      myLastItem = Integer.parseInt(jGoChildElement.getAttribute("lastitem"));
      myScrollBarOnRight = jGoChildElement.getAttribute("scrollbaronright").equals("true");
      mySpacing = Integer.parseInt(jGoChildElement.getAttribute("spacing"));
      myVertical = jGoChildElement.getAttribute("vertical").equals("true");
      myInsets.left = Integer.parseInt(jGoChildElement.getAttribute("insetleft"));
      myInsets.right = Integer.parseInt(jGoChildElement.getAttribute("insetright"));
      myInsets.top = Integer.parseInt(jGoChildElement.getAttribute("insettop"));
      myInsets.bottom = Integer.parseInt(jGoChildElement.getAttribute("insetbottom"));

      String bar = jGoChildElement.getAttribute("bar");
      svgDoc.registerReferencingObject(this, "bar", bar);
      String rect = jGoChildElement.getAttribute("rect");
      svgDoc.registerReferencingObject(this, "rect", rect);
      String sVectorElements = jGoChildElement.getAttribute("vectorelements");
      while (sVectorElements.length() > 0) {
        int nEnd = sVectorElements.indexOf(" ");
        if (nEnd == -1)
          nEnd = sVectorElements.length();
        String sVectorElement = sVectorElements.substring(0, nEnd);
        if (nEnd >= sVectorElements.length())
          sVectorElements = "";
        else
          sVectorElements = sVectorElements.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "vectorelements", sVectorElement);
      }
      String sIcons = jGoChildElement.getAttribute("icons");
      while (sIcons.length() > 0) {
        int nEnd = sIcons.indexOf(" ");
        if (nEnd == -1)
          nEnd = sIcons.length();
        String sIcon = sIcons.substring(0, nEnd);
        if (nEnd >= sIcons.length())
          sIcons = "";
        else
          sIcons = sIcons.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "icons", sIcon);
      }
      String sLeftPorts = jGoChildElement.getAttribute("leftports");
      while (sLeftPorts.length() > 0) {
        int nEnd = sLeftPorts.indexOf(" ");
        if (nEnd == -1)
          nEnd = sLeftPorts.length();
        String sPort = sLeftPorts.substring(0, nEnd);
        if (nEnd >= sLeftPorts.length())
          sLeftPorts = "";
        else
          sLeftPorts = sLeftPorts.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "leftports", sPort);
      }
      String sRightPorts = jGoChildElement.getAttribute("rightports");
      while (sRightPorts.length() > 0) {
        int nEnd = sRightPorts.indexOf(" ");
        if (nEnd == -1)
          nEnd = sRightPorts.length();
        String sPort = sRightPorts.substring(0, nEnd);
        if (nEnd >= sRightPorts.length())
          sRightPorts = "";
        else
          sRightPorts = sRightPorts.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "rightports", sPort);
      }
      if (jGoChildElement.getAttribute("embeddedlinepen").equals("true")) {
        svgDoc.SVGTraverseChildren(jGoDoc, jGoChildElement, null, false);
      }
      String pen = jGoChildElement.getAttribute("linepen");
      svgDoc.registerReferencingObject(this, "linepen", pen);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }


  // Event hints
  public static final int VerticalChanged = JGoDocumentEvent.LAST + 10030;
  public static final int ScrollBarOnRightChanged = JGoDocumentEvent.LAST + 10031;
  public static final int LinePenChanged = JGoDocumentEvent.LAST + 10032;
  public static final int InsetsChanged = JGoDocumentEvent.LAST + 10033;
  public static final int SpacingChanged = JGoDocumentEvent.LAST + 10034;
  public static final int AlignmentChanged = JGoDocumentEvent.LAST + 10035;
  public static final int FirstVisibleIndexChanged = JGoDocumentEvent.LAST + 10036;
  public static final int ItemChanged = JGoDocumentEvent.LAST + 10037;
  public static final int LeftPortChanged = JGoDocumentEvent.LAST + 10038;
  public static final int RightPortChanged = JGoDocumentEvent.LAST + 10039;
  public static final int ItemInsertedChanged = JGoDocumentEvent.LAST + 10040;
  public static final int ItemRemovedChanged = JGoDocumentEvent.LAST + 10041;
  public static final int IconChanged = JGoDocumentEvent.LAST + 10042;
  public static final int IconAlignmentChanged = JGoDocumentEvent.LAST + 10043;
  public static final int IconSpacingChanged = JGoDocumentEvent.LAST + 10044;


  /** Align the items at the left side */
  public static final int ALIGN_LEFT = 0;
  /** Align the items at the center */
  public static final int ALIGN_CENTER = 1;
  /** Align the items at the right side. */
  public static final int ALIGN_RIGHT = 2;


  private static int myBarSize = 14;


  // State

  // array of JGoObject items
  private ArrayList myVector = new ArrayList();
  // array of JGoObject icons
  private ArrayList myIcons = new ArrayList();
  // arrays of JGoObjects, typically JGoPorts
  private ArrayList myLeftPorts = new ArrayList();
  private ArrayList myRightPorts = new ArrayList();

  // the background rectangle that will serve as the selectable resizable object
  private JGoRectangle myRect = null;
  // the scroll bar
  private JGoScrollBar myBar = null;

  private boolean myVertical = true;
  private boolean myScrollBarOnRight = true;
  private int myAlignment = ALIGN_LEFT;
  private int myIconAlignment = ALIGN_LEFT;
  private int myFirstItem = 0;
  private int myLastItem = -1;
  // assume scroll bar is for vertically arranged items, on the right side
  private Insets myInsets = new Insets(1, 4, 1, myBarSize + 4);
  private int mySpacing = 0;
  private JGoPen myLinePen = null;
  private int myIconSpacing = 1;

  private transient Dimension myMaxItemSize = new Dimension(-1, -1);
}
