/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;
import java.util.ArrayList;

public class MultiTextNode extends JGoNode
{
  public MultiTextNode()
  {
    super();
  }

  public void initialize()
  {
    setInitializing(true);
    setResizable(false);

    myBack = createBackground();
    addObjectAtHead(myBack);
    myTopPort = createEndPort(true);
    addObjectAtTail(myTopPort);
    myBottomPort = createEndPort(false);
    addObjectAtTail(myBottomPort);

    setInitializing(false);
    layoutChildren(null);
  }

  public JGoDrawable createBackground()
  {
    JGoRectangle r = new JGoRectangle();
    r.setBrush(null);
    r.setSelectable(false);
    r.setDraggable(false);
    r.setResizable(false);
    return r;
  }

  public JGoObject createEndPort(boolean top)
  {
    JGoPort p = new JGoPort();
    p.setSize(5, 3);
    p.setStyle(JGoPort.StyleHidden);
    if (top) {
      p.setFromSpot(TopCenter);
      p.setToSpot(TopCenter);
    } else {
      p.setFromSpot(BottomCenter);
      p.setToSpot(BottomCenter);
    }
    return p;
  }

  public JGoObject createItem(int i, String s)
  {
    JGoText t = new JGoText(s);
    t.setTransparent(true);
    t.setMultiline(true);
    t.setClipping(true);
    t.setWrapping(true);
    t.setWrappingWidth(getItemWidth());
    return t;
  }

  public JGoObject createPort(int i, boolean left)
  {
    JGoPort p = new JGoPort();
    p.setSize(3, 5);
    p.setStyle(JGoPort.StyleHidden);
    if (left) {
      p.setFromSpot(LeftCenter);
      p.setToSpot(LeftCenter);
    } else {
      p.setFromSpot(RightCenter);
      p.setToSpot(RightCenter);
    }
    return p;
  }


  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    MultiTextNode newobj = (MultiTextNode)newarea;

    // let maximum item size be recalculated (when == -1)
    newobj.myInsets.top = myInsets.top;
    newobj.myInsets.left = myInsets.left;
    newobj.myInsets.bottom = myInsets.bottom;
    newobj.myInsets.right = myInsets.right;
    newobj.mySpacing = mySpacing;
    newobj.myLinePen = myLinePen;
    newobj.myItemWidth = myItemWidth;

    super.copyChildren(newarea, env);

    newobj.myBack = (JGoDrawable)env.get(myBack);
    newobj.myTopPort = (JGoObject)env.get(myTopPort);
    newobj.myBottomPort = (JGoObject)env.get(myBottomPort);

    for (int i = 0; i < myVector.size(); i++) {
      JGoObject olditem = (JGoObject)myVector.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myVector.add(newitem);
    }
    for (int i = 0; i < myLeftPorts.size(); i++) {
      JGoObject olditem = (JGoObject)myLeftPorts.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myLeftPorts.add(newitem);
    }
    for (int i = 0; i < myRightPorts.size(); i++) {
      JGoObject olditem = (JGoObject)myRightPorts.get(i);
      JGoObject newitem = (JGoObject)env.get(olditem);
      newobj.myRightPorts.add(newitem);
    }
  }


  // the parts, other than the items

  // the background rectangle
  public JGoDrawable getBackground()
  {
    return myBack;
  }

  public void setBackground(JGoDrawable b)
  {
    JGoDrawable old = myBack;
    if (old != b) {
      if (old != null) {
        if (b != null)
          b.setBoundingRect(old.getBoundingRect());
        removeObject(old);
      }
      myBack = b;
      if (b != null) {
        b.setSelectable(false);
        b.setDraggable(false);
        b.setResizable(false);
        addObjectAtHead(b);
      }
      update(BackgroundChanged, 0, old);
    }
  }


  public JGoPen getLinePen()
  {
    return myLinePen;
  }

  public void setLinePen(JGoPen pen)
  {
    JGoPen oldPen = myLinePen;
    if (oldPen != pen) {
      myLinePen = pen;
      layoutChildren(null);
      update(LinePenChanged, 0, oldPen);
    }
  }


  // extra space around the edges
  public Insets getInsets()
  {
    return myInsets;
  }

  public void setInsets(Insets x)
  {
    Insets s = myInsets;
    if (!s.equals(x)) {
      Insets oldInsets = new Insets(s.top, s.left, s.bottom, s.right);
      myInsets.top = x.top;
      myInsets.left = x.left;
      myInsets.bottom = x.bottom;
      myInsets.right = x.right;
      layoutChildren(null);
      update(InsetsChanged, 0, oldInsets);
    }
  }

  // extra space between items
  public int getSpacing()
  {
    return mySpacing;
  }

  public void setSpacing(int newspace)
  {
    int oldSpacing = mySpacing;
    if (oldSpacing != newspace) {
      mySpacing = newspace;
      layoutChildren(null);
      update(SpacingChanged, oldSpacing, null);
    }
  }


  // convenience methods
  public JGoPen getPen()
  {
    return getBackground().getPen();
  }

  public void setPen(JGoPen p)
  {
    getBackground().setPen(p);
  }

  public JGoBrush getBrush()
  {
    return getBackground().getBrush();
  }

  public void setBrush(JGoBrush b)
  {
    getBackground().setBrush(b);
  }

  public JGoObject getTopPort()
  {
    return myTopPort;
  }

  public void setTopPort(JGoObject p)
  {
    JGoObject oldPort = myTopPort;
    if (oldPort != p) {
      if (oldPort != null) {
        removeObject(oldPort);
      }
      myTopPort = p;
      if (p != null) {
        addObjectAtTail(p);
      }
      update(TopPortChanged, 0, oldPort);
    }
  }

  public JGoObject getBottomPort()
  {
    return myBottomPort;
  }

  public void setBottomPort(JGoObject p)
  {
    JGoObject oldPort = myBottomPort;
    if (oldPort != p) {
      if (oldPort != null) {
        removeObject(oldPort);
      }
      myBottomPort = p;
      if (p != null) {
        addObjectAtTail(p);
      }
      update(BottomPortChanged, 0, oldPort);
    }
  }



  // how many items are in the list (both invisible and visible)
  public int getNumItems()
  {
    return myVector.size();
  }


  public JGoObject getItem(int i)
  {
    if (i < 0 || i >= myVector.size())
      return null;
    else
      return (JGoObject)myVector.get(i);
  }

  // this is a no-op if there was no item at index I to begin with
  public void setItem(int i, JGoObject obj)
  {
    if (i < 0 || i >= getNumItems()) return;
    JGoObject oldobj = getItem(i);
    if (oldobj == null) return;
    if (oldobj != obj) {
      removeObject(oldobj);
      myVector.set(i, obj);

      if (obj != null) {
        addObjectAtTail(obj);

        obj.setSelectable(false);
        obj.setResizable(false);
        obj.setDraggable(false);

        layoutChildren(obj);
      }
      update(ItemChanged, i, oldobj);
    }
  }


  public JGoObject getLeftPort(int i)
  {
    if (i < 0 || i >= myLeftPorts.size())
      return null;
    else
      return (JGoObject)myLeftPorts.get(i);
  }

  public void setLeftPort(int i, JGoObject obj)
  {
    JGoObject oldobj = getLeftPort(i);
    if (oldobj != obj) {
      if (oldobj != null) {
        if (obj != null)
          obj.setBoundingRect(oldobj.getBoundingRect());
        removeObject(oldobj);
      }
      myLeftPorts.set(i, obj);
      if (obj != null) {
        obj.setSelectable(false);
        obj.setDraggable(false);
        obj.setResizable(false);
        addObjectAtTail(obj);
      }
      update(LeftPortChanged, i, oldobj);
    }
  }

  public JGoObject getRightPort(int i)
  {
    if (i < 0 || i >= myRightPorts.size())
      return null;
    else
      return (JGoObject)myRightPorts.get(i);
  }

  public void setRightPort(int i, JGoObject obj)
  {
    JGoObject oldobj = getRightPort(i);
    if (oldobj != obj) {
      if (oldobj != null) {
        if (obj != null)
          obj.setBoundingRect(oldobj.getBoundingRect());
        removeObject(oldobj);
      }
      myRightPorts.set(i, obj);
      if (obj != null) {
        obj.setSelectable(false);
        obj.setDraggable(false);
        obj.setResizable(false);
        addObjectAtTail(obj);
      }
      update(RightPortChanged, i, oldobj);
    }
  }


  public String getString(int i)
  {
    JGoObject obj = getItem(i);
    if (obj != null && obj instanceof JGoText) {
      JGoText t = (JGoText)obj;
      return t.getText();
    }
    return "";
  }

  public void setString(int i, String s)
  {
    JGoObject obj = getItem(i);
    if (obj != null && obj instanceof JGoText) {
      JGoText t = (JGoText)obj;
      t.setText(s);
    }
  }

  public JGoObject addString(String s)
  {
    int i = getNumItems();
    insertItem(i, createItem(i, s), createPort(i, true), createPort(i, false));
    return getItem(i);
  }

  final public void addItem(JGoObject item, JGoObject leftport, JGoObject rightport)
  {
    insertItem(getNumItems(), item, leftport, rightport);
  }

  public void insertItem(int i, JGoObject item, JGoObject leftport, JGoObject rightport)
  {
    if (i < 0 || i > getNumItems())  // == getNumItems() is OK--means add at end
      return;

    JGoDrawable r = getBackground();

    myVector.add(i, item);
    if (item != null) {
      item.setTopLeft(r.getLeft(), r.getTop());
      item.setSelectable(false);
      item.setDraggable(false);
      item.setResizable(false);
      addObjectAtTail(item);
    }

    myLeftPorts.add(i, leftport);
    if (leftport != null) {
      leftport.setTopLeft(r.getLeft(), r.getTop());
      leftport.setSelectable(false);
      leftport.setDraggable(false);
      leftport.setResizable(false);
      addObjectAtTail(leftport);
    }
    myRightPorts.add(i, rightport);
    if (rightport != null) {
      rightport.setTopLeft(r.getLeft(), r.getTop());
      rightport.setSelectable(false);
      rightport.setDraggable(false);
      rightport.setResizable(false);
      addObjectAtTail(rightport);
    }

    layoutChildren(null);

    JGoObject[] newvals = new JGoObject[3];
    newvals[0] = item;
    newvals[1] = leftport;
    newvals[2] = rightport;
    update(ItemInsertedChanged, i, newvals);
  }

  public void removeItem(int i)
  {
    if (i < 0 || i >= getNumItems())
      return;

    JGoObject leftport = getLeftPort(i);
    myLeftPorts.remove(i);
    if (leftport != null)
      super.removeObject(leftport);

    JGoObject rightport = getRightPort(i);
    myRightPorts.remove(i);
    if (rightport != null)
      super.removeObject(rightport);

    JGoObject item = getItem(i);
    if (item != null) {
      myVector.remove(i);
      super.removeObject(item);
    }

    layoutChildren(null);

    JGoObject[] oldvals = new JGoObject[3];
    oldvals[0] = item;
    oldvals[1] = leftport;
    oldvals[2] = rightport;
    update(ItemRemovedChanged, i, oldvals);
  }

  // this overrides JGoArea.removeObject(JGoObject)
  public void removeObject(JGoObject obj)
  {
    boolean found = false;
    int num = getNumItems();
    for (int i = 0; i < num; i++) {
      JGoObject item = getItem(i);
      if (item == obj) {
        removeItem(i);
        found = true;
        break;
      }
    }
    if (!found)
      super.removeObject(obj);
  }


  // don't have a way of removing a "port" object, unless
  // you call set[Left/Right]Port(i, null);

  // and of course, you can't have a port unless it's associated with an item in the list


  // this assumes that the getBackground() top-left position is the basis for all item
  // layout decisions; might someday want to generalize to any spot, like the Middle.
  public void layoutChildren(JGoObject childchanged)
  {
    if (isInitializing()) return;
    Insets insets = getInsets();

    JGoObject r = getBackground();
    if (r == null) return;  // not yet initialized
    setInitializing(true);

    sendObjectToBack(r);

    int topheight = 0;
    if (getTopPort() != null)
      topheight = getTopPort().getHeight();
    int bottomheight = 0;
    if (getBottomPort() != null)
      bottomheight = getBottomPort().getHeight();

    int leftwidth = getMaxPortWidth(true);
    int rightwidth = getMaxPortWidth(false);
    int itemwidth = getItemWidth();

    int left = r.getLeft() + insets.left;
    int top = r.getTop() + insets.top;
    int width = itemwidth + insets.left + insets.right;

    int penwidth = 0;
    if (getLinePen() != null)
      penwidth = getLinePen().getWidth();
    int sep = Math.max(penwidth, getSpacing());

    layoutEndPorts();

    int h = 0;  // height of visible items so far
    int num = getNumItems();
    for (int i = 0; i < num; i++) {
      int size = getItemHeight(i);
      layoutItem(i, left, top + h, width, size, itemwidth);
      h += size;
      h += sep;
    }
    if (num > 0)
      h -= sep;
    r.setSize(width, h + insets.top + insets.bottom);

    setInitializing(false);
  }

  // Return the standard width for all of the item objects, excluding the ports
  public int getItemWidth()
  {
    return myItemWidth;
  }

  public void setItemWidth(int w)
  {
    int old = myItemWidth;
    if (old != w) {
      myItemWidth = w;
      for (int i = 0; i < getNumItems(); i++) {
        JGoObject obj = getItem(i);
        if (obj != null && obj instanceof JGoText) {
          JGoText t = (JGoText)obj;
          t.setWrappingWidth(w);
        }
      }
      layoutChildren(null);
      update(ItemWidthChanged, old, null);
    }
  }

  // Return the maximum width for all the ports on a given side
  public int getMaxPortWidth(boolean left)
  {
    int width = 0;
    ArrayList ports = (left ? myLeftPorts : myRightPorts);
    for (int i = 0; i < ports.size(); i++) {
      JGoObject p = (JGoObject)ports.get(i);
      if (p == null) continue;
      width = Math.max(width, p.getWidth());
    }
    return width;
  }

  // return the height needed for item I, including any ports
  protected int getItemHeight(int i)
  {
    int size = 0;

    JGoObject item = getItem(i);
    if (item != null) {
      size = item.getHeight();
    }

    JGoObject lport = getLeftPort(i);
    if (lport != null) {
      size = Math.max(size, lport.getHeight());
    }

    JGoObject rport = getRightPort(i);
    if (rport != null) {
      size = Math.max(size, rport.getHeight());
    }

    return size;
  }

  protected void layoutEndPorts()
  {
    JGoObject r = getBackground();
    if (r == null) return;

    JGoObject obj = getTopPort();
    if (obj != null) {
      obj.setSpotLocation(BottomCenter, r, TopCenter);
    }

    obj = getBottomPort();
    if (obj != null) {
      obj.setSpotLocation(TopCenter, r, BottomCenter);
    }
  }

  // X, Y, WIDTH, and HEIGHT specify the rectangle where we should
  // position the I'th item.
  // The ports should be positioned outside of the MultiTextNode's Rect.
  protected void layoutItem(int i, int x, int y, int width, int height, int itemwidth)
  {
    JGoObject item = getItem(i);

    // determine the positions of the object
    // relative to the given rectangle
    if (item != null) {
      item.setTopLeft(x, y + height/2 - item.getHeight()/2);
      item.setWidth(itemwidth);
    }

    JGoObject r = getBackground();
    if (r == null) return;  // not yet initialized

    int rectleft = r.getLeft();
    int recttop = r.getTop();
    int rectwidth = r.getWidth();

    JGoObject lport = getLeftPort(i);
    if (lport != null) {
      lport.setVisible(true);
      int xpos = rectleft - lport.getWidth();
      int ypos = y + height/2 - lport.getHeight()/2;
      lport.setTopLeft(xpos, ypos);
    }

    JGoObject rport = getRightPort(i);
    if (rport != null) {
      rport.setVisible(true);
      int xpos = rectleft + rectwidth;
      int ypos = y + height/2 - rport.getHeight()/2;
      rport.setTopLeft(xpos, ypos);
    }
  }


  public void paint(Graphics2D g, JGoView view)
  {
    super.paint(g, view);

    // draw the lines between the items, if line pen has non-zero width
    int penwidth = 0;
    if (getLinePen() != null)
      penwidth = getLinePen().getWidth();
    if (penwidth <= 0)
      return;

    JGoObject r = getBackground();
    if (r == null) return;  // not yet initialized

    int rectleft = r.getLeft();
    int recttop = r.getTop();
    int rectwidth = r.getWidth();
    int rectheight = r.getHeight();

    Insets insets = getInsets();
    int left = rectleft + insets.left;
    int top = recttop + insets.top;
    int width = rectwidth - insets.left - insets.right;

    int s = 0;  // height of visible items so far
    int sep = Math.max(penwidth, getSpacing());
    int num = getNumItems()-1;  // don't draw after last item
    for (int i = 0; i < num; i++) {
      int size = getItemHeight(i);
      s += size;
      JGoDrawable.drawLine(g, getLinePen(), rectleft,             top + s + sep/2,
                                            rectleft + rectwidth, top + s + sep/2);
      s += sep;
    }
  }


  // undo/redo support

  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getFlags()) {
      case LinePenChanged:
        e.setNewValue(getLinePen());
        return;
      case InsetsChanged: {
        // make sure value gets copied so it doesn't get clobbered later
        Insets s = getInsets();
        e.setNewValue(new Insets(s.top, s.left, s.bottom, s.right));
        return; }
      case SpacingChanged:
        e.setNewValueInt(getSpacing());
        return;
      case ItemChanged:
        e.setNewValue(getItem(e.getOldValueInt()));
        return;
      case ItemWidthChanged:
        e.setNewValueInt(getItemWidth());
        return;
      case BackgroundChanged:
        e.setNewValue(getBackground());
        return;
      case TopPortChanged:
        e.setNewValue(getTopPort());
        return;
      case BottomPortChanged:
        e.setNewValue(getBottomPort());
        return;
      case LeftPortChanged:
        e.setNewValue(getLeftPort(e.getOldValueInt()));
        return;
      case RightPortChanged:
        e.setNewValue(getRightPort(e.getOldValueInt()));
        return;
      case ItemInsertedChanged:
        // the old value information indicates where it had been inserted
        return;
      case ItemRemovedChanged:
        // the old value information indicates where and what had been removed
        return;
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getFlags()) {
      case LinePenChanged:
        myLinePen = (JGoPen)e.getValue(undo);
        update();
        return;
      case InsetsChanged:
        Insets x = (Insets)e.getValue(undo);
        myInsets.top = x.top;
        myInsets.left = x.left;
        myInsets.bottom = x.bottom;
        myInsets.right = x.right;
        return;
      case SpacingChanged:
        mySpacing = e.getValueInt(undo);
        return;
      case ItemChanged:
        setItem(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case ItemWidthChanged:
        myItemWidth = e.getValueInt(undo);
        return;
      case BackgroundChanged:
        setBackground((JGoDrawable)e.getValue(undo));
        return;
      case TopPortChanged:
        setTopPort((JGoPort)e.getValue(undo));
        return;
      case BottomPortChanged:
        setBottomPort((JGoPort)e.getValue(undo));
        return;
      case LeftPortChanged:
        setLeftPort(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case RightPortChanged:
        setRightPort(e.getOldValueInt(), (JGoObject)e.getValue(undo));
        return;
      case ItemInsertedChanged:
        if (undo) {
          removeItem(e.getOldValueInt());
        } else {
          JGoObject[] newvals = (JGoObject[])e.getOldValue();
          insertItem(e.getOldValueInt(), newvals[0], newvals[1], newvals[2]);
        }
        return;
      case ItemRemovedChanged:
        if (undo) {
          JGoObject[] newvals = (JGoObject[])e.getOldValue();
          insertItem(e.getOldValueInt(), newvals[0], newvals[1], newvals[2]);
        } else {
          removeItem(e.getOldValueInt());
        }
        return;
      default:
        super.changeValue(e, undo);
        return;
    }
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("backdraw")) {
      myBack = (JGoDrawable)referencedObject;
      layoutChildren(null);
    }
    else if (attr.equals("topport")) {
      myTopPort = (JGoObject)referencedObject;
    }
    else if (attr.equals("bottomport")) {
      myBottomPort = (JGoObject)referencedObject;
    }
    else if (attr.equals("vectorelements")) {
      myVector.add(referencedObject);
      layoutChildren(null);
    }
    else if (attr.equals("leftports")) {
      myLeftPorts.add(referencedObject);
    }
    else if (attr.equals("rightports")) {
      myRightPorts.add(referencedObject);
    }
    else if (attr.equals("linepen")) {
      myLinePen = (JGoPen)referencedObject;
      layoutChildren(null);
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add MultiTextNode element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jMultiTextNode = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.MultiTextNode", jGoElementGroup);
      jMultiTextNode.setAttribute("spacing", Integer.toString(mySpacing));
      jMultiTextNode.setAttribute("itemwidth", Integer.toString(myItemWidth));
      jMultiTextNode.setAttribute("insetleft", Integer.toString(myInsets.left));
      jMultiTextNode.setAttribute("insetright", Integer.toString(myInsets.right));
      jMultiTextNode.setAttribute("insettop", Integer.toString(myInsets.top));
      jMultiTextNode.setAttribute("insetbottom",
                                  Integer.toString(myInsets.bottom));

      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myBack != null) {
        svgDoc.registerReferencingNode(jMultiTextNode, "backdraw", myBack);
      }
      if (myTopPort != null) {
        svgDoc.registerReferencingNode(jMultiTextNode, "topport", myTopPort);
      }
      if (myBottomPort != null) {
        svgDoc.registerReferencingNode(jMultiTextNode, "bottomport",
                                       myBottomPort);
      }
      for (int i = 0; i < myVector.size(); i++) {
        svgDoc.registerReferencingNode(jMultiTextNode, "vectorelements",
                                       (JGoObject) myVector.get(i));
      }
      for (int i = 0; i < myLeftPorts.size(); i++) {
        svgDoc.registerReferencingNode(jMultiTextNode, "leftports",
                                       (JGoObject) myLeftPorts.get(i));
      }
      for (int i = 0; i < myRightPorts.size(); i++) {
        svgDoc.registerReferencingNode(jMultiTextNode, "rightports",
                                       (JGoObject) myRightPorts.get(i));
      }
      // The following objects are NOT children of this area and must be
      // explicitly written out
      if (myLinePen != null) {
        if (!svgDoc.isRegisteredReference(myLinePen)) {
          jMultiTextNode.setAttribute("embeddedlinepen", "true");
          DomElement subGroup = svgDoc.createElement("g");
          jMultiTextNode.appendChild(subGroup);
          myLinePen.SVGWriteObject(svgDoc, subGroup);
        }
        svgDoc.registerReferencingNode(jMultiTextNode, "linepen", myLinePen);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a MultiTextNode element
      mySpacing = Integer.parseInt(jGoChildElement.getAttribute("spacing"));
      myItemWidth = Integer.parseInt(jGoChildElement.getAttribute("itemwidth"));
      myInsets.left = Integer.parseInt(jGoChildElement.getAttribute("insetleft"));
      myInsets.right = Integer.parseInt(jGoChildElement.getAttribute("insetright"));
      myInsets.top = Integer.parseInt(jGoChildElement.getAttribute("insettop"));
      myInsets.bottom = Integer.parseInt(jGoChildElement.getAttribute("insetbottom"));

      String backdraw = jGoChildElement.getAttribute("backdraw");
      svgDoc.registerReferencingObject(this, "backdraw", backdraw);
      String topport = jGoChildElement.getAttribute("topport");
      svgDoc.registerReferencingObject(this, "topport", topport);
      String bottomport = jGoChildElement.getAttribute("bottomport");
      svgDoc.registerReferencingObject(this, "bottomport", bottomport);
      String sVectorElements = jGoChildElement.getAttribute("vectorelements");
      while (sVectorElements.length() > 0) {
        int nEnd = sVectorElements.indexOf(" ");
        if (nEnd == -1)
          nEnd = sVectorElements.length();
        String sVectorElement = sVectorElements.substring(0, nEnd);
        if (nEnd >= sVectorElements.length())
          sVectorElements = "";
        else
          sVectorElements = sVectorElements.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "vectorelements", sVectorElement);
      }
      String sLeftPorts = jGoChildElement.getAttribute("leftports");
      while (sLeftPorts.length() > 0) {
        int nEnd = sLeftPorts.indexOf(" ");
        if (nEnd == -1)
          nEnd = sLeftPorts.length();
        String sPort = sLeftPorts.substring(0, nEnd);
        if (nEnd >= sLeftPorts.length())
          sLeftPorts = "";
        else
          sLeftPorts = sLeftPorts.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "leftports", sPort);
      }
      String sRightPorts = jGoChildElement.getAttribute("rightports");
      while (sRightPorts.length() > 0) {
        int nEnd = sRightPorts.indexOf(" ");
        if (nEnd == -1)
          nEnd = sRightPorts.length();
        String sPort = sRightPorts.substring(0, nEnd);
        if (nEnd >= sRightPorts.length())
          sRightPorts = "";
        else
          sRightPorts = sRightPorts.substring(nEnd + 1);
        svgDoc.registerReferencingObject(this, "rightports", sPort);
      }
      if (jGoChildElement.getAttribute("embeddedlinepen").equals("true")) {
        svgDoc.SVGTraverseChildren(jGoDoc, jGoChildElement, null, false);
      }
      String pen = jGoChildElement.getAttribute("linepen");
      svgDoc.registerReferencingObject(this, "linepen", pen);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }

  // Event hints
  public static final int LinePenChanged = JGoDocumentEvent.LAST + 10032;
  public static final int InsetsChanged = JGoDocumentEvent.LAST + 10033;
  public static final int SpacingChanged = JGoDocumentEvent.LAST + 10034;
  public static final int ItemWidthChanged = JGoDocumentEvent.LAST + 10036;
  public static final int ItemChanged = JGoDocumentEvent.LAST + 10037;
  public static final int BackgroundChanged = JGoDocumentEvent.LAST + 10022;
  public static final int TopPortChanged = JGoDocumentEvent.LAST + 10023;
  public static final int BottomPortChanged = JGoDocumentEvent.LAST + 10025;
  public static final int LeftPortChanged = JGoDocumentEvent.LAST + 10038;
  public static final int RightPortChanged = JGoDocumentEvent.LAST + 10039;
  public static final int ItemInsertedChanged = JGoDocumentEvent.LAST + 10040;
  public static final int ItemRemovedChanged = JGoDocumentEvent.LAST + 10041;


  // State

  // array of JGoObject items, typically JGoTexts
  private ArrayList myVector = new ArrayList();

  // arrays of JGoObjects, typically JGoPorts
  private ArrayList myLeftPorts = new ArrayList();
  private ArrayList myRightPorts = new ArrayList();

  private JGoObject myTopPort = null;
  private JGoObject myBottomPort = null;

  // the background drawable, typically a JGoRectangle
  private JGoDrawable myBack = null;

  private Insets myInsets = new Insets(1, 1, 1, 1);
  private int mySpacing = 0;
  private JGoPen myLinePen = null;
  private int myItemWidth = 150;
}
