/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;
import com.nwoods.jgo.examples.ListArea;

public class RecordNode extends JGoNode
{
  public RecordNode()
  {
    super();
  }

  public void initialize()
  {
    myListArea = new ListArea();
    myListArea.initialize();
    myListArea.setVertical(true);
    myListArea.setSelectable(false);
    myListArea.getRect().setSelectable(false);
    addObjectAtHead(myListArea);
  }

  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    RecordNode newobj = (RecordNode)newarea;

    super.copyChildren(newarea, env);

    newobj.myListArea = (ListArea)env.get(myListArea);
    newobj.myHeader = (JGoObject)env.get(myHeader);
    newobj.myFooter = (JGoObject)env.get(myFooter);
  }


  // RecordNode parts: a header, a list area, and a footer

  public ListArea getListArea() { return myListArea; }

  public JGoObject getHeader() { return myHeader; }

  public void setHeader(JGoObject obj)
  {
    JGoObject oldobj = myHeader;
    if (oldobj != obj) {
      if (oldobj != null)
        removeObject(oldobj);
      myHeader = obj;
      if (obj != null) {
        obj.setSelectable(false);
        addObjectAtHead(obj);
      }
      layoutChildren(obj);
      update(HeaderChanged, 0, oldobj);
    }
  }

  public JGoObject getFooter() { return myFooter; }

  public void setFooter(JGoObject obj)
  {
    JGoObject oldobj = myFooter;
    if (oldobj != obj) {
      if (oldobj != null)
        removeObject(oldobj);
      myFooter = obj;
      if (obj != null) {
        obj.setSelectable(false);
        addObjectAtHead(obj);
      }
      layoutChildren(obj);
      update(FooterChanged, 0, oldobj);
    }
  }


  // figure out the minimum width and height for resizing
  protected Rectangle handleResize(Graphics2D g, JGoView view, Rectangle prevRect,
                                   Point newPoint, int whichHandle, int event,
                                   int minWidth, int minHeight)
  {
    Dimension minSize = myListArea.getMinimumSize();
    if (getHeader() != null)
      minSize.height += getHeader().getHeight();
    if (getFooter() != null)
      minSize.height += getFooter().getHeight();
    return super.handleResize(g, view, prevRect, newPoint, whichHandle, event,
                              minSize.width, minSize.height);
  }

  // constrain to the minimum width and height
  public void setBoundingRect(int left, int top, int width, int height)
  {
    Dimension minSize = myListArea.getMinimumSize();
    if (getHeader() != null)
      minSize.height += getHeader().getHeight();
    if (getFooter() != null)
      minSize.height += getFooter().getHeight();
    super.setBoundingRect(left, top,
                          Math.max(width, minSize.width),
                          Math.max(height, minSize.height));
  }

  // since setVisible() doesn't automatically call setVisible() on all the
  // children, we need to do this manually to handle the scroll bar in
  // the list area
  public void setVisible(boolean bFlag)
  {
    if (myListArea != null)
      myListArea.setVisible(bFlag);
    super.setVisible(bFlag);
  }

  public void layoutChildren(JGoObject childchanged)
  {
    if (isInitializing() || myListArea == null) return;
    setInitializing(true);

    // get this record node's bounding rect
    int rx = getLeft();
    int ry = getTop();
    int rw = getWidth();
    int rh = getHeight();

    // get the header and footer heights, which are not changed
    int headh = 0;
    if (getHeader() != null)
      headh = getHeader().getHeight();
    int footh = 0;
    if (getFooter() != null)
      footh = getFooter().getHeight();

    // position the listarea between the two headers
    myListArea.setBoundingRect(rx, ry + headh, rw, rh - headh - footh);

    // get the difference between the ListArea and its ListAreaRect
    // because we want to line up the header and footer with the
    // ListAreaRect, not with the ListArea
    JGoRectangle larect = myListArea.getRect();
    int diffx = larect.getLeft() - myListArea.getLeft();
    int diffy = larect.getTop() - myListArea.getTop();

    int law = larect.getWidth();
    int lay = larect.getHeight();
    int diffw = myListArea.getWidth() - law;
    int diffh = myListArea.getHeight() - lay;

    // position the header and footer
    if (getHeader() != null)
      getHeader().setBoundingRect(rx + diffx, ry, law, headh);
    if (getFooter() != null)
      getFooter().setBoundingRect(rx + diffx, ry + rh - footh, law, footh);
    setInitializing(false);
  }


  // handle the list area changing width--reposition the header and footer too
  protected boolean geometryChangeChild(JGoObject child, Rectangle prevRect)
  {
    if (isInitializing()) return false;
    if (child == myListArea &&
        child.getWidth() != prevRect.width) {
      int w = myListArea.getRect().getWidth();
      if (getHeader() != null)
        getHeader().setWidth(w);
      if (getFooter() != null)
        getFooter().setWidth(w);
      setWidth(w);  // this will invoke layoutChildren
      return true;
    } else {
      return super.geometryChangeChild(child, prevRect);
    }
  }


  public void addItem(JGoObject obj, JGoPort leftport, JGoPort rightport, JGoObject icon)
  {
    if (leftport != null) {
      leftport.setValidSource(false);
      leftport.setValidDestination(true);
      if (myListArea.isVertical()) {
        leftport.setFromSpot(JGoObject.LeftCenter);
        leftport.setToSpot(JGoObject.LeftCenter);
      } else {
        leftport.setFromSpot(JGoObject.TopCenter);
        leftport.setToSpot(JGoObject.TopCenter);
      }
    }
    if (rightport != null) {
      rightport.setValidSource(true);
      rightport.setValidDestination(false);
      if (myListArea.isVertical()) {
        rightport.setFromSpot(JGoObject.RightCenter);
        rightport.setToSpot(JGoObject.RightCenter);
      } else {
        rightport.setFromSpot(JGoObject.BottomCenter);
        rightport.setToSpot(JGoObject.BottomCenter);
      }
    }
    myListArea.insertItem(getNumItems(), obj, leftport, rightport, icon);
  }

  public JGoPort getLeftPort(int i) { return (JGoPort)myListArea.getLeftPort(i); }
  public void setLeftPort(int i, JGoPort p)
  {
    if (p != null) {
      if (myListArea.isVertical()) {
        p.setFromSpot(JGoObject.LeftCenter);
        p.setToSpot(JGoObject.LeftCenter);
      } else {
        p.setFromSpot(JGoObject.TopCenter);
        p.setToSpot(JGoObject.TopCenter);
      }
    }
    myListArea.setLeftPort(i, p);
  }

  public JGoPort getRightPort(int i) { return (JGoPort)myListArea.getRightPort(i); }
  public void setRightPort(int i, JGoPort p)
  {
    if (p != null) {
      if (myListArea.isVertical()) {
        p.setFromSpot(JGoObject.RightCenter);
        p.setToSpot(JGoObject.RightCenter);
      } else {
        p.setFromSpot(JGoObject.BottomCenter);
        p.setToSpot(JGoObject.BottomCenter);
      }
    }
    myListArea.setRightPort(i, p);
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("footer")) {
      myFooter = (JGoObject)referencedObject;
    }
    else if (attr.equals("header")) {
      myHeader = (JGoObject)referencedObject;
    }
    else if (attr.equals("listarea")) {
      myListArea = (ListArea)referencedObject;
    }
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add RecordNode element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement jBasicNode = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.RecordNode", jGoElementGroup);
      // The following elements are all children of this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myFooter != null) {
        svgDoc.registerReferencingNode(jBasicNode, "footer", myFooter);
      }
      if (myHeader != null) {
        svgDoc.registerReferencingNode(jBasicNode, "header", myHeader);
      }
      if (myListArea != null) {
        svgDoc.registerReferencingNode(jBasicNode, "listarea", myListArea);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a RecordNode element
      String footer = jGoChildElement.getAttribute("footer");
      svgDoc.registerReferencingObject(this, "footer", footer);
      String header = jGoChildElement.getAttribute("header");
      svgDoc.registerReferencingObject(this, "header", header);
      String listarea = jGoChildElement.getAttribute("listarea");
      svgDoc.registerReferencingObject(this, "listarea", listarea);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }

  // convenience routines, from ListArea
  public int getNumItems() { return myListArea.getNumItems(); }
  public JGoObject getItem(int i) { return myListArea.getItem(i); }
  public void setItem(int i, JGoObject obj) { myListArea.setItem(i, obj); }
  public int findItem(JGoObject obj) { return myListArea.findItem(obj); }
  public void removeItem(int i) { myListArea.removeItem(i); }

  public JGoObject getIcon(int i) { return myListArea.getIcon(i); }
  public void setIcon(int i, JGoObject obj) { myListArea.setIcon(i, obj); }

  public boolean isVertical() { return myListArea.isVertical(); }
  public void setVertical(boolean b) { myListArea.setVertical(b); }
  public boolean isScrollBarOnRight() { return myListArea.isScrollBarOnRight(); }
  public void setScrollBarOnRight(boolean rb) { myListArea.setScrollBarOnRight(rb); }
  public JGoPen getLinePen() { return myListArea.getLinePen(); }
  public void setLinePen(JGoPen p) { myListArea.setLinePen(p); }
  public Insets getInsets() { return myListArea.getInsets(); }
  public void setInsets(Insets x) { myListArea.setInsets(x); }
  public int getSpacing() { return myListArea.getSpacing(); }
  public void setSpacing(int newspace) { myListArea.setSpacing(newspace); }
  public int getAlignment() { return myListArea.getAlignment(); }
  public void setAlignment(int align) { myListArea.setAlignment(align); }
  public int getIconAlignment() { return myListArea.getIconAlignment(); }
  public int getIconSpacing() { return myListArea.getIconSpacing(); }
  public void setIconSpacing(int newspace) { myListArea.setIconSpacing(newspace); }
  public void setIconAlignment(int align) { myListArea.setIconAlignment(align); }
  public int getFirstVisibleIndex() { return myListArea.getFirstVisibleIndex(); }
  public void setFirstVisibleIndex(int i) { myListArea.setFirstVisibleIndex(i); }
  public int getLastVisibleIndex() { return myListArea.getLastVisibleIndex(); }


  // undo/redo support

  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getFlags()) {
      case HeaderChanged:
        e.setNewValue(getHeader());
        return;
      case FooterChanged:
        e.setNewValue(getFooter());
        return;
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getFlags()) {
      case HeaderChanged:
        setHeader((JGoObject)e.getValue(undo));
        return;
      case FooterChanged:
        setFooter((JGoObject)e.getValue(undo));
        return;
      default:
        super.changeValue(e, undo);
        return;
    }
  }


  // Event hints
  public static final int HeaderChanged = JGoDocumentEvent.LAST + 10048;
  public static final int FooterChanged = JGoDocumentEvent.LAST + 10049;


  // State
  private ListArea myListArea = null;
  private JGoObject myHeader = null;
  private JGoObject myFooter = null;
}
