/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;

/**
 * A SimpleNode has an icon, a label, and up to two ports.
 * The label is a SimpleNodeLabel, positioned at the bottom center
 * of the icon.
 * The two ports are SimpleNodePort's, one for "input" and one for
 * "output", assuming a left-to-right "flow".
 * <p>
 * SimpleNode supports the notion of a minimum size.
 * Resizing the node now only resizes the icon,
 * not the label or the ports.  Furthermore, resizing the icon now
 * maintains its original aspect ratio.
 */
public class SimpleNode extends JGoNode
{
  /** Create an empty SimpleNode.  Call initialize() before using it. */
  public SimpleNode()
  {
    super();
  }

  // The location is the top-left corner of the node;
  // the size is the dimension of the icon;
  // the labeltext may be null if no label is desired.
  public void initialize(Point loc, Dimension size, JGoObject icon,
                         String labeltext, boolean hasinport, boolean hasoutport)
  {
    setInitializing(true);
    // the user can move this node around
    setDraggable(true);
    // the user cannot resize this node
    setResizable(false);
    // if it does become resizable, only show four resize handles
    set4ResizeHandles(true);

    myIcon = icon;
    if (myIcon != null) {
      myIcon.setBoundingRect(loc,size);
      myIcon.setSelectable(false);
      addObjectAtHead(myIcon);
    }

    // the label is a SimpleNodeLabel, centered underneath the icon
    if (labeltext != null) {
      myLabel = new SimpleNodeLabel(labeltext, this);
    }

    // create an input port and an output port, each instances of SimpleNodePort
    if (hasinport) {
      myInputPort = createPort(true);
    }
    if (hasoutport) {
      myOutputPort = createPort(false);
    }

    setInitializing(false);
    layoutChildren(null);
    setTopLeft(loc);
  }

  protected JGoPort createPort(boolean input) {
    return new SimpleNodePort(input, this);
  }

  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    SimpleNode newobj = (SimpleNode)newarea;

    super.copyChildren(newarea, env);

    newobj.myIcon = (JGoObject)env.get(myIcon);
    newobj.myLabel = (JGoText)env.get(myLabel);
    newobj.myInputPort = (JGoPort)env.get(myInputPort);
    newobj.myOutputPort = (JGoPort)env.get(myOutputPort);
  }

  /**
   * When an object is removed, make sure there are no more references from fields.
   */
  public JGoObject removeObjectAtPos(JGoListPosition pos)
  {
    JGoObject child = super.removeObjectAtPos(pos);
    if (child == myIcon)
      myIcon = null;
    else if (child == myLabel)
      myLabel = null;
    else if (child == myInputPort)
      myInputPort = null;
    else if (child == myOutputPort)
      myOutputPort = null;
    return child;
  }

  /**
   * Keep the parts of a SimpleNode positioned relative to each other
   * by setting their locations using some of the standard spots of
   * any JGoObject.
   */
  public void layoutChildren(JGoObject childchanged)
  {
    if (isInitializing()) return;
    setInitializing(true);

    JGoObject icon = getIcon();
    JGoObject label = getLabel();
    JGoObject inport = getInputPort();
    JGoObject outport = getOutputPort();

    if (label != null) {
      if (icon != null)
        label.setSpotLocation(TopCenter, icon, BottomCenter);
      else
        label.setSpotLocation(BottomCenter, this, BottomCenter);
    }
    if (inport != null) {
      if (icon != null)
        inport.setSpotLocation(RightCenter, icon, LeftCenter);
      else
        inport.setSpotLocation(LeftCenter, this, LeftCenter);
    }
    if (outport != null) {
      if (icon != null)
        outport.setSpotLocation(LeftCenter, icon, RightCenter);
      else
        outport.setSpotLocation(RightCenter, this, RightCenter);
    }

    setInitializing(false);
  }

  /**
   * If this object is resized, do the part positioning lay out again.
   * The ports and the text label do not get resized; only the icon
   * changes size, while keeping its old aspect ratio.
   */
  public void rescaleChildren(Rectangle prevRect)
  {
    // only change size of icon; need to calculate its new size while
    // keeping its old aspect ratio
    if (myIcon != null) {
      int oldw = myIcon.getWidth();
      int oldh = myIcon.getHeight();
      if (oldw <= 0) oldw = 1;
      double ratio = oldh/((double)oldw);
      // figure out how much space is left in the area after accounting
      // for any ports and label
      int iconw = getWidth();
      int iconh = getHeight();
      if (myInputPort != null)
        iconw -= myInputPort.getWidth();
      if (myOutputPort != null)
        iconw -= myOutputPort.getWidth();
      if (myLabel != null)
        iconh -= myLabel.getHeight();
      // now we have the maximum bounds for the icon, figure out the
      // right width and height that fit while maintaining the aspect ratio
      double maxratio = iconh/((double)iconw);
      if (ratio < maxratio)
        iconh = (int)Math.rint(ratio*iconw);
      else
        iconw = (int)Math.rint(iconh/ratio);
      myIcon.setSize(iconw, iconh);
    }
  }

  public Dimension getMinimumIconSize()
  {
    return new Dimension(20, 20);
  }

  public Dimension getMinimumSize()
  {
    int w = 0;
    int h = 0;
    // account for any ports and label
    if (myInputPort != null)
      w += myInputPort.getWidth();
    if (myOutputPort != null)
      w += myOutputPort.getWidth();
    // now account for any minimum desired icon size
    Dimension minIconSize = getMinimumIconSize();
    w += minIconSize.width;
    h += minIconSize.height;
    if (myLabel != null) {
      w = Math.max(w, myLabel.getWidth());
      h += myLabel.getHeight();
    }
    return new Dimension(w, h);
  }

  // constrain to the minimum width and height
  public void setBoundingRect(int left, int top, int width, int height)
  {
    Dimension minSize = getMinimumSize();
    super.setBoundingRect(left, top,
                          Math.max(width, minSize.width),
                          Math.max(height, minSize.height));
  }

  // limit the minimum width and height for resizing
  protected Rectangle handleResize(Graphics2D g, JGoView view, Rectangle prevRect,
                                   Point newPoint, int whichHandle, int event,
                                   int minWidth, int minHeight)
  {
    Dimension minSize = getMinimumSize();
    Rectangle newRect = super.handleResize(g, view, prevRect, newPoint, whichHandle, event,
                                           Math.max(minWidth, minSize.width), Math.max(minHeight, minSize.height));
    // resize continuously (default only does setBoundingRect on MouseUp)
    if (event == JGoView.EventMouseMove)
      setBoundingRect(newRect);
    return null;
  }


  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add <SimpleNode> element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement simpleNode = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.SimpleNode", jGoElementGroup);
      // The following elements are all children is this area and so will be writen out
      // by JGoArea.SVGWriteObject().  We just need to update the references to them.
      if (myIcon != null) {
        svgDoc.registerReferencingNode(simpleNode, "simpleicon", myIcon);
      }
      if (myInputPort != null) {
        svgDoc.registerReferencingNode(simpleNode, "simpleinputport",
                                       myInputPort);
      }
      if (myOutputPort != null) {
        svgDoc.registerReferencingNode(simpleNode, "simpleoutputport",
                                       myOutputPort);
      }
      if (myLabel != null) {
        svgDoc.registerReferencingNode(simpleNode, "simplelabel", myLabel);
      }
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a <SimpleNode> element
      String simpleIcon = jGoChildElement.getAttribute("simpleicon");
      svgDoc.registerReferencingObject(this, "simpleicon", simpleIcon);
      String simpleInputPort = jGoChildElement.getAttribute("simpleinputport");
      svgDoc.registerReferencingObject(this, "simpleinputport", simpleInputPort);
      String simpleOutputPort = jGoChildElement.getAttribute("simpleoutputport");
      svgDoc.registerReferencingObject(this, "simpleoutputport", simpleOutputPort);
      String simpleLabel = jGoChildElement.getAttribute("simplelabel");
      svgDoc.registerReferencingObject(this, "simplelabel", simpleLabel);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingElement());
    }
    return svgElement.getNextSibling();
  }

  public void SVGUpdateReference(String attr, Object referencedObject)
  {
    super.SVGUpdateReference(attr, referencedObject);
    if (attr.equals("simpleicon")) {
      myIcon = (JGoObject)referencedObject;
    }
    if (attr.equals("simpleinputport")) {
      myInputPort = (JGoPort)referencedObject;
    }
    if (attr.equals("simpleoutputport")) {
      myOutputPort = (JGoPort)referencedObject;
    }
    if (attr.equals("simplelabel")) {
      myLabel = (JGoText)referencedObject;
    }
  }

  public JGoText getLabel() { return myLabel; }
  public JGoObject getIcon() { return myIcon; }
  public JGoPort getInputPort() { return myInputPort; }
  public JGoPort getOutputPort() { return myOutputPort; }

    // State
  protected JGoText myLabel = null;
  protected JGoObject myIcon = null;
  protected JGoPort myInputPort = null;
  protected JGoPort myOutputPort = null;
  /**
   * A real application will have some other data associated with
   * the node, holding state and methods to be called according to
   * the needs of the application.
   */
}
