/*
 *  Copyright (c) Northwoods Software Corporation, 1999-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples;

import com.nwoods.jgo.*;
import java.awt.*;

/**
 * SimpleNodePort has been implemented as a port of StyleTriangle,
 * assuming ports are only on the left and right sides of a SimpleNode.
 * <p>
 * SimpleNodePort also has some smarts about the kinds of links
 * that can be made to this kind of port.
 * <p>
 * You can easily change the appearance of the port by calling
 * setPortStyle(JGoPort.StyleObject) and then setPortObject to
 * change the object used to represent the port.
 * For example you may want to use an instance of JGoImage.
 * These objects can be shared by more than one JGoPort.
 */
public class SimpleNodePort extends JGoPort
{
  /** This creates a StyleEllipse port.  Call initialize() before using. */
  public SimpleNodePort()
  {
    super();
  }

  /** This creates a light gray StyleTriangle port of the appropriate direction. */
  public SimpleNodePort(boolean input, JGoArea parent)
  {
    super(TriangleRect());
    initialize(input, parent);
  }

  public void initialize(boolean input, JGoArea parent)
  {
    setSelectable(false);
    setDraggable(false);
    setResizable(false);
    setVisible(true);
    // assume an outlined light gray triangle
    setStyle(StyleTriangle);
    setPen(JGoPen.darkGray);
    setBrush(JGoBrush.lightGray);
    // assume inputs are on the left, outputs are on the right
    if (input) {
      setValidSource(false);
      setValidDestination(true);
      setToSpot(LeftCenter);
    } else {
      setValidSource(true);
      setValidDestination(false);
      setFromSpot(RightCenter);
    }
    // put in the SimpleNode area
    setTopLeft(parent.getLeft(), parent.getTop());
    parent.addObjectAtTail(this);
  }

  // ports remember whether they are inputs or outputs
  public final boolean isInput()
  {
    return isValidDestination();
  }

  public final boolean isOutput()
  {
    return isValidSource();
  }

  // Only allow links from output ports to input ports,
  // only between different nodes, and
  // only if there isn't already a link to "to".
  public boolean validLink(JGoPort to)
  {
    return (super.validLink(to) &&
            isOutput() &&
            (to instanceof SimpleNodePort) &&
            ((SimpleNodePort)to).isInput());
  }

  /**
   * A convenience method for returning the parent as a SimpleNode.
   */
  public SimpleNode getNode() { return (SimpleNode)getParent(); }



  // return the bounding rectangle for the triangle-shaped port
  static public Rectangle TriangleRect()
  {
    return myTriangleRect;
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add <SimpleNodePort> element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement simpleNodePort = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.SimpleNodePort", jGoElementGroup);
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a <SimpleNodePort> element
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingElement());
    }
    return svgElement.getNextSibling();
  }



  static private Rectangle myTriangleRect = new Rectangle(0, 0, 8, 8);
}

