/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.boxapp;

import java.awt.*;
import com.nwoods.jgo.*;

public class BoxArea extends JGoSubGraphBase implements JGoLabeledPart {
  public BoxArea() {}  // default constructor, just for copying

  public BoxArea(String text) {
    init();
    setText(text);
  }

  public void init() {
    setInitializing(true);
    setSelectable(true);
    setDragsNode(false);
    myBorder = new JGoRoundRect();
    myBorder.setSelectable(false);
    myBorder.setPen(myDefaultPen);
    myBorder.setBrush(myDefaultBrush);
    myBorder.setSize(100, 100);
    addObjectAtHead(myBorder);
    myLabel = new JGoText();
    myLabel.setText("Box");
    myLabel.setSelectable(false);
    myLabel.setEditable(true);
    myLabel.setEditOnSingleClick(true);
    addObjectAtTail(myLabel);
    setInitializing(false);
    layoutChildren(myBorder);
  }

  private static JGoPen myDefaultPen = JGoPen.make(JGoPen.DASHED, 2, JGoBrush.ColorBlue);
  private static JGoBrush myDefaultBrush = JGoBrush.makeStockBrush(new Color(200, 200, 255, 63));
  private static JGoPen myHighlightPen = JGoPen.make(JGoPen.SOLID, 2, JGoBrush.ColorRed);

  // override to keep fields up-to-date
  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env) {
    super.copyChildren(newarea, env);
    BoxArea newobj = (BoxArea)newarea;
    newobj.myBorder = (JGoDrawable)env.get(myBorder);
    newobj.myLabel = (JGoText)env.get(myLabel);
  }
  
  // override to keep fields up-to-date
  public JGoObject removeObjectAtPos(JGoListPosition pos) {
    JGoObject child = super.removeObjectAtPos(pos);
    if (child == myBorder)
      myBorder = null;
    else if (child == myLabel)
      myLabel = null;
    return child;
  }

  // Determine the smallest rectangle that encloses all of the other children of this area.
  // This might return null, if there are no other objects besides the Border.
  public Rectangle computeBorder() {
    Rectangle rect = null;
    JGoListPosition pos = getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = getObjectAtPos(pos);
      pos = getNextObjectPosAtTop(pos);

      if (obj == getBorder()) continue;
      if (obj == getLabel()) continue;
      if (obj instanceof JGoPort) continue;
      if (obj instanceof JGoLink) continue;
      if (rect == null) {
        Rectangle b = obj.getBoundingRect();
        rect = new Rectangle(b.x, b.y, b.width, b.height);
      } else {
        // add the object's bounding rect to this one
        rect.add(obj.getBoundingRect());
      }
    }
    if (rect != null) {
      // leave some room as a margin
      rect.grow(4, 4);
    }
    return rect;
  }

  // override to keep myBorder surrounding the other child objects
  public void layoutChildren(JGoObject childchanged) {
    if (isInitializing()) return;
    super.layoutChildren(childchanged);
    setInitializing(true);
    JGoObject border = getBorder();
    if (border == null) return;
    addObjectAtHead(border);  // keep behind all other children
    if (childchanged == border) {
      Rectangle rect = border.getBoundingRect();
      getLabel().setSpotLocation(BottomLeft, rect.x, rect.y-2);
    }
    setInitializing(false);
  }

  public void expandBorder() {
    // compute the minimum rectangle needed to enclose the children except for the Border
    Rectangle rect = computeBorder();
    if (rect != null) {
      // but don't have the box shrink to minimum size continuously
      rect.add(getBorder().getBoundingRect());
      getBorder().setBoundingRect(rect);
      if (getParent() instanceof BoxArea) {
        ((BoxArea)getParent()).expandBorder();
      }
    }
  }

  protected Rectangle handleResize(Graphics2D g, JGoView view, Rectangle origRect,
                                   Point newPoint, int whichHandle, int event, int minWidth, int minHeight) {
    // compute rectangle for normal resizing
    Rectangle newRect = computeResize(origRect, newPoint, whichHandle, minWidth, minHeight);
    // account for the height of the Label
    newRect.y += getLabel().getHeight() + 2;
    newRect.height -= getLabel().getHeight() + 2;
    // compute rectangle required by the child objects
    Rectangle minRect = computeBorder();
    if (minRect != null) {
      newRect.add(minRect);
    }
    // update the bounding rect of the Border
    getBorder().setBoundingRect(newRect);
    if (getParent() instanceof BoxArea) {
      ((BoxArea)getParent()).expandBorder();
    }
    return null;
  }

  // implement highlighting by changing the Pen of the Border
  public void setHighlit(boolean lit) {
    if (getBorder() != null) {
      setSkipsUndoManager(true);  // don't let the UndoManager notice these changes!
      if (lit)
        getBorder().setPen(myHighlightPen);
      else
        getBorder().setPen(myDefaultPen);
      setSkipsUndoManager(false);
    }
  }

  public boolean isHighlit() {
    return (getBorder() != null && getBorder().getPen() == myHighlightPen);
  }

  public JGoDrawable getBorder() { return myBorder; }
  
  // JGoLabeledPart interface:
  public JGoText getLabel() { return myLabel; }
  public String getText() { return getLabel().getText(); }

  // for convenience:
  public void setText(String s) {
    if (getLabel() != null) getLabel().setText(s);
  }

  private JGoDrawable myBorder = null;
  private JGoText myLabel = null;
}
