/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.boxapp;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.nwoods.jgo.*;

public class BoxView extends JGoView {
  public BoxView() {}

  public boolean doMouseUp(int modifiers, Point dc, Point vc) {
    if (getState() == MouseStateMove) considerBox(dc);  // update highlighting of potential BoxArea
    return super.doMouseUp(modifiers, dc, vc);
  }

  public boolean doMouseMove(int modifiers, Point dc, Point vc) {
    if (getState() == MouseStateMove) considerBox(dc);  // update highlighting of potential BoxArea
    return super.doMouseMove(modifiers, dc, vc);
  }

  public void doUncapturedMouseMove(int modifiers, Point dc, Point vc) {
    setHighlit(null);  // don't highlight any BoxArea when not dragging the selection
    super.doUncapturedMouseMove(modifiers, dc, vc);
  }

  private void considerBox(Point dc) {
    JGoObject area = pickObjectExcluding(dc, true, getSelection());
    while (area != null && !(area instanceof BoxArea)) {
      area = area.getParent();
    }
    if (area instanceof BoxArea) {
      setHighlit((BoxArea)area);
    } else {
      setHighlit(null);
    }
    BoxApp.setStatus(" ");
  }

  // this BoxArea is under the current mouse point and is not selected
  public BoxArea getHighlit() { return myHighlitBox; }

  public void setHighlit(BoxArea box) {
    // only do something when the value changes
    BoxArea old = getHighlit();
    if (old != box) {
      if (old != null) old.setHighlit(false);
      myHighlitBox = box;
      if (box != null) box.setHighlit(true);
    }
  }

  private BoxArea myHighlitBox = null;


  public JGoObject pickObjectExcluding(Point p, boolean selectableOnly, JGoObjectSimpleCollection ignore) {
    ArrayList coll = new ArrayList();
    // iterate over document layers in reverse order
    JGoLayer layer = getLastLayer();
    while (layer != null) {
      coll.clear();
      // find all objects at point P
      layer.pickObjects(p, selectableOnly, coll, 999999);
      for (int i = 0; i < coll.size(); i++) {
        JGoObject obj = (JGoObject)coll.get(i);
        // see if the object is selected or is part of a selected object
        boolean ignored = false;
        JGoListPosition pos = ignore.getFirstObjectPos();
        while (pos != null) {
          JGoObject s = ignore.getObjectAtPos(pos);
          pos = ignore.getNextObjectPosAtTop(pos);
          if (obj == s || obj.isChildOf(s)) {
            ignored = true;
            break;
          }
          // also ignore a BoxArea that is the parent of a selected object if the point P is outside of the BoxArea's Border;
          // this lets the user drag a node from inside a BoxArea outside of it, to be reparented elsewhere
          if (obj instanceof BoxArea && s.getParent() == obj && !((BoxArea)obj).getBorder().isPointInObj(p)) {
            ignored = true;
            break;
          }
        }
        if (!ignored) return obj;
      }
      layer = getPrevLayer(layer);
    }
    return null;
  }


  // implement a view listener for the selection having been moved or copied...
  public void fireUpdate(int hint, int flags, Object x, Point vc, Point dc, int modifiers) {
    super.fireUpdate(hint, flags, x, vc, dc, modifiers);
    if ((hint == JGoViewEvent.SELECTION_MOVED || hint == JGoViewEvent.SELECTION_COPIED)) {
      // what unselected BoxArea is at the drop point?
      BoxArea box = getHighlit();
      JGoLayer deflayer = getDocument().getDefaultLayer();
      if (box != null) {
        // make children of a BoxArea
        getSelection().addCollection(box.addCollection(getSelection(), true, deflayer));
        box.expandBorder();
        BoxApp.setStatus("added to " + box.getText());
      } else {
        // no box at mouse point: make selection top-level objects in default layer
        getSelection().addCollection(deflayer.addCollection(getSelection(), true, deflayer));
        BoxApp.setStatus("moved to top-level");
      }
    } else if (hint == JGoViewEvent.LINK_CREATED && x instanceof JGoLink) {
      JGoLink link = (JGoLink)x;
      link.setArrowHeads(false, true);  // user-drawn links get arrowheads
    }
  }


  public void onKeyEvent(KeyEvent evt) {  // handle common commands
    super.onKeyEvent(evt);
    int code = evt.getKeyCode();
    if (code == KeyEvent.VK_DELETE) {  // delete the currently selected objects
      deleteSelection();
    } else if (evt.isControlDown() && code == KeyEvent.VK_A) {  // select all
      selectAll();
    } else if (evt.isControlDown() && code == KeyEvent.VK_X) {  // cut
      cut();
    } else if (evt.isControlDown() && code == KeyEvent.VK_C) {  // copy
      copy();
    } else if (evt.isControlDown() && code == KeyEvent.VK_V) {  // paste
      paste();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Z) {  // undo
      getDocument().undo();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Y) {  // redo
      getDocument().redo();
    }
  }
}
