/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.boxapp;

import java.awt.*;
import com.nwoods.jgo.*;

public class TestSubGraph extends BoxArea {
  public TestSubGraph() {}  // default constructor, just for copying

  public TestSubGraph(String s) {
    super(s);
    initPorts();
  }

  /**
   * Create an input port and an output port, each a green triangle.
   */
  public void initPorts() {
    setInitializing(true);

    myInput = new JGoPort();
    myInput.setSize(7, 7);
    myInput.setBrush(JGoBrush.green);
    myInput.setStyle(JGoPort.StyleTriangle);
    myInput.setValidSource(false);
    addObjectAtTail(myInput);

    myOutput = new JGoPort();
    myOutput.setSize(7, 7);
    myOutput.setBrush(JGoBrush.green);
    myOutput.setStyle(JGoPort.StyleTriangle);
    myOutput.setValidDestination(false);
    addObjectAtTail(myOutput);

    setInitializing(false);
    layoutChildren(null);
  }

  // override to keep fields up-to-date
  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env) {
    super.copyChildren(newarea, env);
    TestSubGraph newobj = (TestSubGraph)newarea;
    newobj.myInput = (JGoPort)env.get(myInput);
    newobj.myOutput = (JGoPort)env.get(myOutput);
  }

  // override to keep fields up-to-date
  public JGoObject removeObjectAtPos(JGoListPosition pos) {
    JGoObject child = super.removeObjectAtPos(pos);
    if (child == myInput)
      myInput = null;
    else if (child == myOutput)
      myOutput = null;
    return child;
  }

  /**
   * The input port is positioned inside the center left point of the subgraph;
   * the output port is position inside the center right spot.
   */
  public void layoutChildren(JGoObject childchanged) {
    if (isInitializing()) return;
    super.layoutChildren(childchanged);
    if (getBorder() == null) return;
    Rectangle b = getBorder().getBoundingRect();
    setInitializing(true);
    if (getInput() != null) {
      getInput().setSpotLocation(CenterRight, new Point(b.x, b.y + b.height / 2));
    }
    if (getOutput() != null) {
      getOutput().setSpotLocation(CenterLeft, new Point(b.x+b.width, b.y+b.height/2));
    }
    setInitializing(false);
  }

  protected Rectangle handleResize(Graphics2D g, JGoView view, Rectangle origRect,
                                   Point newPoint, int whichHandle, int event, int minWidth, int minHeight) {
    // compute rectangle for normal resizing
    Rectangle newRect = computeResize(origRect, newPoint, whichHandle, minWidth, minHeight);
    // account for the height of the Label
    newRect.y += getLabel().getHeight() + 2;
    newRect.height -= getLabel().getHeight() + 2;
    // account for widths of ports
    if (getInput() != null && getOutput() != null) {
      newRect.x += getInput().getWidth();
      newRect.width -= getInput().getWidth() + getOutput().getWidth();
    }
    // compute rectangle required by the child objects
    Rectangle minRect = computeBorder();
    if (minRect != null) {
      newRect.add(minRect);
    }
    // update the bounding rect of the Border
    getBorder().setBoundingRect(newRect);
    if (getParent() instanceof BoxArea) {
      ((BoxArea)getParent()).expandBorder();
    }
    return null;
  } 
  
  public JGoPort getInput() { return myInput; }
  public JGoPort getOutput() { return myOutput; }

  private JGoPort myInput = null;
  private JGoPort myOutput = null;
}
