/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.classier;

import java.awt.*;
import java.lang.reflect.*;
import com.nwoods.jgo.*;

/**
 * ClassNode is just a JGoTextNode that displays either the name of a
 * class or all of its public and protected methods.
 * <p>
 * Selecting a ClassNode changes the node's background color.
 * <p>
 * When expanded to hold all of the method information, the background
 * is changed to yet a third color which is translucent so that the
 * user can see the class hierarchy "through" the method information.
 */
class ClassNode extends JGoTextNode
{
  public ClassNode() {}
  public ClassNode(String s) { super(s); }

  // Remember the Class object associated with this ClassNode
  public Object getObject() { return myObject; }
  public void setObject(Object x)
  {
    if (x != null && x instanceof Class) {
      Class c = (Class)x;
      myObject = c;
      setText(c.getName());
      getLabel().setMultiline(false);
      if (c.isInterface()) {
        setPen(myInterfacePen);
        setBrush(myInterfaceBrush);
        getLabel().setItalic(true);
      } else {
        setPen(myClassPen);
        setBrush(myClassBrush);
        getLabel().setItalic(false);
      }
    }
  }


  // Expand or contract the node to show or hide methods
  public void toggleInfo()
  {
    Class nodeclass = (Class)getObject();
    // use the Multiline property to keep track of expanded state
    if (getLabel().isMultiline()) {
      getLabel().setMultiline(false);
      if (nodeclass.isInterface())
        setBrush(myInterfaceBrush);
      else
        setBrush(myClassBrush);
      setText(nodeclass.getName());
    } else {
      getLabel().setMultiline(true);
      setBrush(myDetailBrush);
      String msg = nodeclass.getName();
      msg += getClassInfo(nodeclass);
      setText(msg);
    }
  }


  // Return a string showing all of the methods of the class.
  //
  // The default textual representation of Methods and Constructors
  // is more verbose than it needs to be for most people.  We strip
  // out common package prefixes such as "java.lang.".
  public String getClassInfo(Class c)
  {
    String info = "";
    try {
      String prefix = c.getName() + ".";
      Constructor[] ctors = c.getDeclaredConstructors();
      int i;
      for (i = 0; i < ctors.length; i++) {
        Constructor m = ctors[i];
        if (isDisplayable(m)) {
          info += "\n";
          String str = m.toString();
          str = replaceSubstring(str, prefix, "");
          str = replaceSubstring(str, "com.nwoods.jgo.", "");
          str = replaceSubstring(str, "java.lang.", "");
          info += str;
        }
      }
      Method[] methods = c.getDeclaredMethods();
      for (i = 0; i < methods.length; i++) {
        Method m = methods[i];
        if (isDisplayable(m)) {
          info += "\n";
          String str = m.toString();
          str = replaceSubstring(str, prefix, "");
          str = replaceSubstring(str, "com.nwoods.jgo.", "");
          str = replaceSubstring(str, "java.lang.", "");
          info += str;
        }
      }
    } catch (Exception e) {
      System.err.println(e);
      e.printStackTrace();
    }
    return info;
  }


  // This predicate decides whether to display the Member.
  // By default we only show public and protected methods.
  public boolean isDisplayable(Member m)
  {
    int mod = m.getModifiers();
    return Modifier.isPublic(mod) || Modifier.isProtected(mod);
  }


  // This handy little function replaces one substring with another
  // from the given string.
  private String replaceSubstring(String str, String search, String rep)
  {
    int found = str.indexOf(search);
    if (found < 0)
      return str;
    return str.substring(0, found) +
           rep +
           replaceSubstring(str.substring(found + search.length()),
                            search,
                            rep);
  }


  // don't create handles, but just change the color
  protected void gainedSelection(JGoSelection selection)
  {
    getDocument().bringObjectToFront(this);
    Class c = (Class)getObject();
    if (c.isInterface())
      setBrush(mySelInterfaceBrush);
    else
      setBrush(mySelClassBrush);
  }

  protected void lostSelection(JGoSelection selection)
  {
    Class c = (Class)getObject();
    if (getLabel().isMultiline())
      setBrush(myDetailBrush);
    else if (c.isInterface())
      setBrush(myInterfaceBrush);
    else
      setBrush(myClassBrush);
  }


  // pens and brushes, shared by all ClassNode instances

  public static JGoPen myClassPen = JGoPen.lightGray;
  public static JGoBrush myClassBrush = JGoBrush.lightGray;
  public static JGoBrush mySelClassBrush = JGoBrush.makeStockBrush(new Color(200, 255, 255));

  public static JGoPen myInterfacePen = JGoPen.gray;
  public static JGoBrush myInterfaceBrush = JGoBrush.white;
  public static JGoBrush mySelInterfaceBrush = JGoBrush.makeStockBrush(new Color(200, 255, 255));

  // a light yellow that shows through a little bit
  //   (so the classes can be seen underneath)
  public static JGoBrush myDetailBrush =
      JGoBrush.makeStockBrush(new Color(250, 250, 200, 200));

  // ClassNode state
  private Object myObject = null;
}

