/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.classier;

import java.awt.*;
import java.awt.event.*;
import java.awt.dnd.*;
import javax.swing.*;
import java.util.HashMap;
import java.lang.reflect.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.layout.*;
import com.nwoods.jgo.instruments.*;
import com.nwoods.jgo.examples.*;

/**
 * This app is a simple JGo class hierarchy browser.
 * <p>
 * Besides showing the relationships of the classes and the interfaces
 * they implement, users can see all the public and protected methods
 * declared for each class.
 */
public class Classier extends JApplet implements Runnable
{
  public Classier()
  {
    // Create the main graphics window
    myView = new JGoView();
    myDoc = myView.getDocument();

    myView.setInternalMouseActions(DnDConstants.ACTION_MOVE);

    // Process KeyEvents from the JGoView--handle the Delete key.
    //
    // Normally one overrides JGoView.onKeyEvent() to make use of the
    // default behavior supplied by JGoView.  Using a listener,
    // as this example does, may lead to duplicate and/or conflicting
    // actions if the same key is handled, and does not allow
    // inhibiting the default behavior.
    myView.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent evt) {
        int t = evt.getKeyCode();
        if (t == KeyEvent.VK_DELETE) {
          // go through selection and get rid of any mapping of Class to ClassNode
          JGoSelection sel = myView.getSelection();
          JGoListPosition pos = sel.getFirstObjectPos();
          while (pos != null) {
            JGoObject obj = sel.getObjectAtPos(pos);
            pos = sel.getNextObjectPos(pos);

            if (obj instanceof ClassNode) {
              ClassNode node = (ClassNode)obj;
              Object key = node.getObject();
              if (key != null)
                myMap.remove(key);
            }
          }
          // now delete all selected ClassNodes
          myView.deleteSelection();
        } else if (evt.isControlDown() && t == KeyEvent.VK_Q) {
          System.exit(0);
        } else if (evt.isControlDown() && t == KeyEvent.VK_P) {
          myView.print();
        } else if (t == KeyEvent.VK_F7) {
            double newscale = Math.rint(myView.getScale() * 0.9f * 100f) / 100f;
            myView.setScale(newscale);
        } else if (t == KeyEvent.VK_F8) {
            double newscale = Math.rint(myView.getScale() / 0.9f * 100f) / 100f;
            myView.setScale(newscale);
        }
      }
    });

    // This will handle double clicks
    myView.addViewListener(new JGoViewListener() {
      public void viewChanged(JGoViewEvent e) {
        switch(e.getHint()) {
          case JGoViewEvent.DOUBLE_CLICKED:
          {
            JGoObject obj = e.getJGoObject();
            obj = obj.getTopLevelObject();
            if (obj instanceof ClassNode) {
              ClassNode node = (ClassNode)obj;
              node.toggleInfo();
              myView.getSelection().clearSelection();
            }
            break;
          }
          default: break;
        }
      }
    });

    // the view is the main window
    Container contentPane = getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(myView, BorderLayout.CENTER);
    contentPane.validate();
  }

  public void init()
  {
    // initialize the document/view with some nodes
    initNodes();

    javax.swing.JOptionPane.showMessageDialog(this,
          "Double click a class to see its public and protected methods.\n" +
          "Double click again to hide the information.",
          "View Class Methods",
          javax.swing.JOptionPane.INFORMATION_MESSAGE);
  }

  public void start()
  {
    // enable drag-and-drop from separate thread
    new Thread(this).start();
  }

  public void run() {
    myView.initializeDragDropHandling();
  }

  static public void main(String args[])
  {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

      final JFrame mainFrame = new JFrame();

      // close the application when the main window closes
      mainFrame.addWindowListener(new WindowAdapter() {
        public void windowClosing(java.awt.event.WindowEvent event) {
          Object object = event.getSource();
          if (object == mainFrame)
            System.exit(0);
        }
      });

      mainFrame.setTitle("JGo Class Hierarchy Viewer");
      Dimension screensize = Toolkit.getDefaultToolkit().getScreenSize();
      mainFrame.setBounds(0, 0, screensize.width, screensize.height);

      Classier app = new Classier();
      Container contentPane = mainFrame.getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(app, BorderLayout.CENTER);
      contentPane.validate();

      mainFrame.setVisible(true);

      app.init();
      app.start();
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }
  }

  // This method specifies the minimum set of classes to display.
  // Because Java does not provide access to all the extending or
  // implementing classes for a class or interface, we need to
  // explicitly specify all the classes we care about.
  public void initNodes()
  {
    Cursor oldcursor = myView.getTopLevelAncestor().getCursor();
    myView.getTopLevelAncestor().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

    // don't care about undo/redo, so don't need to call myDoc.fireForedate here
    myDoc.setSuspendUpdates(true);

    addClassNode("com.nwoods.jgo.DomDoc");
    addClassNode("com.nwoods.jgo.DomElement");
    addClassNode("com.nwoods.jgo.DomList");
    addClassNode("com.nwoods.jgo.DomNode");
    addClassNode("com.nwoods.jgo.DomText");
    addClassNode("com.nwoods.jgo.JGo3DNoteRect");
    addClassNode("com.nwoods.jgo.JGo3DRect");
    addClassNode("com.nwoods.jgo.JGoArea");
    addClassNode("com.nwoods.jgo.JGoBasicNode");
    addClassNode("com.nwoods.jgo.JGoBrush");
    addClassNode("com.nwoods.jgo.JGoButton");
    addClassNode("com.nwoods.jgo.JGoCollection");
    addClassNode("com.nwoods.jgo.JGoControl");
    addClassNode("com.nwoods.jgo.JGoCopyMap");
    addClassNode("com.nwoods.jgo.JGoDocument");
    addClassNode("com.nwoods.jgo.JGoDocumentChangedEdit");
    addClassNode("com.nwoods.jgo.JGoDocumentEvent");
    addClassNode("com.nwoods.jgo.JGoDrawable");
    addClassNode("com.nwoods.jgo.JGoEllipse");
    addClassNode("com.nwoods.jgo.JGoGlobal");
    addClassNode("com.nwoods.jgo.JGoHandle");
    addClassNode("com.nwoods.jgo.JGoIconicNode");
    addClassNode("com.nwoods.jgo.JGoImage");
    addClassNode("com.nwoods.jgo.JGoLabeledLink");
    addClassNode("com.nwoods.jgo.JGoLayer");
    addClassNode("com.nwoods.jgo.JGoLink");
    addClassNode("com.nwoods.jgo.JGoLinkLabel");
    addClassNode("com.nwoods.jgo.JGoListPosition");
    addClassNode("com.nwoods.jgo.JGoNode");
    addClassNode("com.nwoods.jgo.JGoObject");
    addClassNode("com.nwoods.jgo.JGoOverview");
    addClassNode("com.nwoods.jgo.JGoPalette");
    addClassNode("com.nwoods.jgo.JGoPen");
    addClassNode("com.nwoods.jgo.JGoPolygon");
    addClassNode("com.nwoods.jgo.JGoPort");
    addClassNode("com.nwoods.jgo.JGoPrintPreview");
    addClassNode("com.nwoods.jgo.JGoRectangle");
    addClassNode("com.nwoods.jgo.JGoRoundRect");
    addClassNode("com.nwoods.jgo.JGoScrollBar");
    addClassNode("com.nwoods.jgo.JGoSelection");
    addClassNode("com.nwoods.jgo.JGoStroke");
    addClassNode("com.nwoods.jgo.JGoSubGraph");
    addClassNode("com.nwoods.jgo.JGoSubGraphBase");
    addClassNode("com.nwoods.jgo.JGoSubGraphHandle");
    addClassNode("com.nwoods.jgo.JGoText");
    addClassNode("com.nwoods.jgo.JGoTextEdit");
    addClassNode("com.nwoods.jgo.JGoTextNode");
    addClassNode("com.nwoods.jgo.JGoUndoManager");
    addClassNode("com.nwoods.jgo.JGoView");
    addClassNode("com.nwoods.jgo.JGoViewEvent");

    addClassNode("com.nwoods.jgo.examples.Comment");
    addClassNode("com.nwoods.jgo.examples.Diamond");
    addClassNode("com.nwoods.jgo.examples.GeneralNode");
    addClassNode("com.nwoods.jgo.examples.GeneralNodeLabel");
    addClassNode("com.nwoods.jgo.examples.GeneralNodePort");
    addClassNode("com.nwoods.jgo.examples.GeneralNodePortLabel");
    addClassNode("com.nwoods.jgo.examples.ListArea");
    addClassNode("com.nwoods.jgo.examples.ListAreaRect");
    addClassNode("com.nwoods.jgo.examples.MultiPortNode");
    addClassNode("com.nwoods.jgo.examples.MultiPortNodeLabel");
    addClassNode("com.nwoods.jgo.examples.MultiPortNodePort");
    addClassNode("com.nwoods.jgo.examples.MultiTextNode");
    addClassNode("com.nwoods.jgo.examples.RecordNode");
    addClassNode("com.nwoods.jgo.examples.SimpleNode");
    addClassNode("com.nwoods.jgo.examples.SimpleNodeLabel");
    addClassNode("com.nwoods.jgo.examples.SimpleNodePort");
    addClassNode("com.nwoods.jgo.examples.SVGGoView");

    addClassNode("com.nwoods.jgo.layout.JGoAutoLayout");
    addClassNode("com.nwoods.jgo.layout.JGoForceDirectedAutoLayout");
    addClassNode("com.nwoods.jgo.layout.JGoLayeredDigraphAutoLayout");
    addClassNode("com.nwoods.jgo.layout.JGoNetwork");
    addClassNode("com.nwoods.jgo.layout.JGoNetworkLink");
    addClassNode("com.nwoods.jgo.layout.JGoNetworkNode");
    addClassNode("com.nwoods.jgo.layout.JGoRandomAutoLayout");

    addClassNode("com.nwoods.jgo.instruments.AbstractGraduatedScale");
    addClassNode("com.nwoods.jgo.instruments.AbstractIndicator");
    addClassNode("com.nwoods.jgo.instruments.GraduatedScaleElliptical");
    addClassNode("com.nwoods.jgo.instruments.GraduatedScaleLinear");
    addClassNode("com.nwoods.jgo.instruments.IndicatorBar");
    addClassNode("com.nwoods.jgo.instruments.IndicatorBarElliptical");
    addClassNode("com.nwoods.jgo.instruments.IndicatorKnob");
    addClassNode("com.nwoods.jgo.instruments.IndicatorNeedle");
    addClassNode("com.nwoods.jgo.instruments.IndicatorSlider");
    addClassNode("com.nwoods.jgo.instruments.IndicatorSliderElliptical");
    addClassNode("com.nwoods.jgo.instruments.Meter");
    addClassNode("com.nwoods.jgo.instruments.MultipleIndicatorMeter");

    layoutNodes();

    myDoc.setSuspendUpdates(false);
    // don't care about undo/redo, so don't need to call myDoc.fireUpdate here

    myView.getTopLevelAncestor().setCursor(oldcursor);
  }

  // This is an overload of addClassNode(Class) that lets
  // us specify classes by name.
  public ClassNode addClassNode(String n)
  {
    Class c = null;
    try {
      c = Class.forName(n);
    } catch (Exception e) {
      javax.swing.JOptionPane.showMessageDialog(this,
        "Unable to find the Class for " + n,
        "Class not found",
        javax.swing.JOptionPane.INFORMATION_MESSAGE);
    }
    return addClassNode(c);
  }

  // Add a ClassNode representing a particular class,
  // and add all of its super classes and interfaces.
  // If a ClassNode already exists for the given Class,
  // just return it.
  public ClassNode addClassNode(Class c)
  {
    if (c == null) return null;

    ClassNode cnode = findNode(c);
    if (cnode != null) return cnode;

    cnode = getNode(c);

    addSuperClass(c, cnode);
    addInterfaces(c, cnode);

    return cnode;
  }

  // Given a Class and its ClassNode, make sure its superclass, if there
  // is one, also has a ClassNode, and then draw an "extends" link to it
  public void addSuperClass(Class c, ClassNode cnode)
  {
    Class superclass = c.getSuperclass();
    if (superclass != null) {
      // also recurses up class hierarchy
      ClassNode parentnode = addClassNode(superclass);
      if (parentnode != null) {
        // add an "extends" link
        JGoLink link = new JGoLink(parentnode.getRightPort(), cnode.getLeftPort());
        link.setSelectable(false);
        link.setResizable(false);
        link.setPen(myExtendsPen);

        myDoc.addObjectAtHead(link);
      }
    }
  }

  // Given a Class and its ClassNode, make sure the Interfaces it
  // implements also have ClassNodes, and then draw "implements" links to them
  public void addInterfaces(Class c, ClassNode cnode)
  {
    Class[] interfaces = c.getInterfaces();
    for (int i = 0; i < interfaces.length; i++) {
      Class intf = interfaces[i];
      // also recurses up interface hierarchy
      ClassNode parentnode = addClassNode(intf);
      if (parentnode != null) {
        // add an "implements" link
        JGoLink link = new JGoLink(parentnode.getRightPort(), cnode.getLeftPort());
        link.setSelectable(false);
        link.setResizable(false);
        link.setPen(myImplementsPen);

        myDoc.addObjectAtHead(link);
      }
    }
  }

  // First layout Object and its subclasses, as a tree.
  // Then layout all interfaces without moving any already positioned nodes.

// To use the JGoLayout automatic layout library,
// uncomment the import com.nwoods.jgo.layout.* line at the beginning of this file,
// use this definition of the layoutNodes method instead of the following one.
//
// *****Note*****
// You must have purchased the JGoAutoLayout package, or have the
// evaluation version in order to use the layered digraph auto layout code.
/*
  public void layoutNodes()
  {
    JGoLayeredDigraphAutoLayout l = new JGoLayeredDigraphAutoLayout(myDoc,
        myHorizSeparation, myVertSeparation, JGoLayeredDigraphAutoLayout.LD_DIRECTION_RIGHT,
        JGoLayeredDigraphAutoLayout.LD_CYCLEREMOVE_DFS,
        JGoLayeredDigraphAutoLayout.LD_LAYERING_LONGESTPATHSOURCE,
        JGoLayeredDigraphAutoLayout.LD_INITIALIZE_DFSOUT, 4,
        JGoLayeredDigraphAutoLayout.LD_AGGRESSIVE_FALSE);
    l.performLayout();
  }
*/

  public void layoutNodes()
  {
    int oldx = 10;  // initial top-left position
    int oldy = 10;

    // first do Object
    ClassNode classNode = addClassNode("java.lang.Object");
    int newy = layoutTree(classNode, oldx, oldy);
    oldy = newy + classNode.getHeight() + myVertSeparation;

    // find all interfaces nodes and lay them out
    JGoListPosition pos = myDoc.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = myDoc.getObjectAtPos(pos);
      pos = myDoc.getNextObjectPosAtTop(pos);

      if (obj instanceof ClassNode) {
        ClassNode node = (ClassNode)obj;

        // skip all classes, just do interfaces
        Class c = (Class)node.getObject();
        if (!c.isInterface()) continue;

        if (node.getLeftPort().hasNoLinks()) {
          newy = layoutTree(node, oldx, oldy);
          oldy = newy + node.getHeight() + myVertSeparation;
        }
      }
    }
  }

  // Implement a simple tree-layout algorithm
  public int layoutTree(ClassNode node, int oldx, int oldy)
  {
    int origy = oldy;
    int newy = oldy;

    // recurse through all the node's children
    JGoPort outp = node.getRightPort();

    // for each link attached to the output port...
    JGoListPosition pos = outp.getFirstLinkPos();
    while (pos != null) {
      JGoLink link = outp.getLinkAtPos(pos);
      pos = outp.getNextLinkPos(pos);

      // in case we link things up in the wrong direction sometime in the future,
      // this gets the other end of the link no matter which way we're going
      JGoPort inp = link.getOtherPort(outp);

      // get the child node, and lay it out recursively if not already positioned
      ClassNode child = (ClassNode)inp.getParent();
      if (child.getTop() <= 0) {
        newy = layoutTree(child, oldx + node.getWidth() + myHorizSeparation, oldy);

        // increment the Y position for the next child, if any
        oldy = newy + node.getHeight() + myVertSeparation;
      }
    }

    // position this node at the average Y position of the children
    node.setTopLeft(oldx, (origy + newy)/2);

    return newy;  // updated by recursive layoutTree calls
  }


  // If we already know about a ClassNode for a given object,
  // return it, else return null
  public ClassNode findNode(Object key)
  {
    Object val = myMap.get(key);
    if (val instanceof ClassNode)
      return (ClassNode)val;
    else
      return null;
  }

  // Get a ClassNode for a given object, using an existing one
  // if possible, otherwise creating one
  public ClassNode getNode(Object key)
  {
    ClassNode node = findNode(key);
    if (node == null) {
      node = new ClassNode(key.toString());
      node.setObject(key);  // keep back pointer
      myDoc.addObjectAtTail(node);
      myMap.put(key, node);
    }
    return node;
  }

  // node spacing parameters
  protected int myHorizSeparation = 40;
  protected int myVertSeparation = 4;

  // link appearances
  protected JGoPen myExtendsPen = JGoPen.black;
  protected JGoPen myImplementsPen = JGoPen.make(JGoPen.DASHED, 1, Color.black);

  protected JGoView myView;
  protected JGoDocument myDoc;

  // maintain a map of Class instances to corresponding ClassNode instances
  protected HashMap myMap = new HashMap();
}
