/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.demo1;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.image.BufferedImage;
import java.awt.print.PrinterJob;
import java.awt.print.PageFormat;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.undo.*;
import javax.swing.table.*;
import java.io.*;
import java.lang.reflect.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.layout.*;
import com.nwoods.jgo.examples.*;

/**
 * A simple application demonstrating many features of JGo.
 */
public class Demo1 extends JFrame
{
  /**
   * Constructor for the Demo1 sample application
   */
  public Demo1()
  {
    Container contentPane = getContentPane();
    contentPane.setBackground(new Color(0xFF, 0xCC, 0xCC));
    setSize(800,650);
    setTitle("JGo Demonstration Application");

    //==============================================================
    // Create the main graphics window
    //==============================================================
    myView = new Demo1View();
    myDoc = myView.getDocument();
    myDoc.setMaintainsPartID(true);
    myView.setBackground(new Color(0xFF, 0xFF, 0xDD));

    // Set up a background image
    //myView.setBackgroundImage((new javax.swing.ImageIcon("C:/jdk1.3.1/demo/applets/ImageMap/images/jim.graham.gif")).getImage());

    myMainLayer = myDoc.getFirstLayer();
    myForegroundLayer = myDoc.addLayerAfter(myMainLayer);
    myForegroundLayer.setIdentifier("in foreground layer");
    myBackgroundLayer = myDoc.addLayerBefore(myMainLayer);
    myBackgroundLayer.setIdentifier("in read-only semitransparent background layer");
    myBackgroundLayer.setTransparency(0.5f);
    myBackgroundLayer.setModifiable(false);
    // by default new user-drawn links go into this layer:
    myDoc.setLinksLayer(myDoc.addLayerBefore(myMainLayer));

    // Set up main view properties
    myView.setGridWidth(20);
    myView.setGridHeight(20);
    myView.setBorder(new TitledBorder("Demo1 View"));

    //==============================================================
    // Add event listeners
    //==============================================================

    // Note: An alternate way to get notification of events from a JGoDocument
    // is to create a subclass of JGoView and override the documentChanged() method.
    myDoc.addDocumentListener(new JGoDocumentListener() {
      public void documentChanged(JGoDocumentEvent e) {
        processDocChange(e);
      }
    });

    // This will handle selection changes and double clicks
    myView.addViewListener(new JGoViewListener() {
      public void viewChanged(JGoViewEvent e) {
        processViewChange(e);
      }
    });

    // Process KeyEvents from the JGoView--handle the Delete key.
    //
    // Normally one overrides JGoView.onKeyEvent() to make use of the
    // default behavior supplied by JGoView.  Using a listener,
    // as this example does, may lead to duplicate and/or conflicting
    // actions if the same key is handled, and does not allow
    // inhibiting the default behavior.
    myView.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent evt) {
        int t = evt.getKeyCode();
        if (t == KeyEvent.VK_DELETE) {
          myView.deleteSelection();
        } else if (evt.isControlDown() && t == KeyEvent.VK_Q) {
          System.exit(0);
        } else if (evt.isControlDown() && t == KeyEvent.VK_S) {
          if (evt.isShiftDown()) {
            FileInputStream fstream = null;
            try {
              fstream = new FileInputStream("Demo1.serialized");
              JGoDocument doc = loadObjects(fstream);
              if (doc != null)
                getCurrentView().setDocument(doc);
            } catch (Exception ex) {
              System.err.println(ex);
            } finally {
              try { if (fstream != null) fstream.close(); } catch (Exception x) {}
            }
          } else {
            FileOutputStream fstream = null;
            try {
              fstream = new FileOutputStream("Demo1.serialized");
              storeObjects(fstream);
            } catch (Exception ex) {
              System.err.println(ex);
            } finally {
              try { if (fstream != null) fstream.close(); } catch (Exception x) {}
            }
          }
        }
      }
    });

    addWindowListener(new WindowAdapter() {
      public void windowClosing(java.awt.event.WindowEvent event) {
        Object object = event.getSource();
        if (object == Demo1.this)
          System.exit(0);             // close the application
      }
    });

    initMenus();

    //==============================================================
    // Create the palette
    // Add a bunch of objects to it in initPalette()
    //==============================================================
    myPalette = new JGoPalette();

    //==============================================================
    // Create the list and add a bunch of actions
    //==============================================================
    myList = new Demo1List();
    AppAction[] insertitems =
          {
            InsertStuffAction,
            InsertStrokeAction,
            InsertPolygonAction,
            InsertDiamondAction,
            InsertRectangleAction,
            InsertRoundedRectangleAction,
            Insert3DRectAction,
            InsertCommentAction,
            InsertEllipseAction,
            InsertMultiPortNodeAction,
            InsertSimpleNodeAction,
            InsertIconicNodeAction,
            InsertDraggableLabelIconicNodeAction,
            InsertBasicNodeAction,
            InsertCenterLabelBasicNodeAction,
            InsertFixedSizeCenterLabelBasicNodeAction,
            InsertRectBasicNodeAction,
            InsertSelfLoopBasicNodeAction,
            InsertMultiSpotNodeAction,
            InsertGeneralNodeAction,
            InsertTextNodeAction,
            InsertFixedSizeTextNodeAction,
            InsertRoundTextNodeAction,
            InsertMultiTextNodeAction,
            InsertTextAction,
            InsertListAreaAction,
            InsertRecordNodeAction,
            InsertGraphOfGraphsAction,
            InsertSanKeyNodesAction
          };
    myList.setListData(insertitems);

    //==============================================================
    // Set up the left hand part of the app
    //==============================================================
    myPaletteAndList = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    myPaletteAndList.setContinuousLayout(true);
    myPaletteAndList.setTopComponent(myPalette);
    myListScrollPane = new JScrollPane(myList);
    myPaletteAndList.setBottomComponent(myListScrollPane);

    //==============================================================
    // Add the windows to the frame
    //==============================================================
    contentPane.setLayout(new BorderLayout());

    contentPane.add(myView, BorderLayout.CENTER);
    contentPane.add(myPaletteAndList, BorderLayout.WEST);
    contentPane.validate();
}



  //==============================================================
  // Define all the command actions
  //==============================================================
    AppAction AboutAction = new AppAction("About", this) {
      public void actionPerformed(ActionEvent e) { aboutAction(); } };

    AppAction ExitAction = new AppAction("Exit", this) {
      public void actionPerformed(ActionEvent e) { exitAction(); } };

    AppAction InsertStuffAction = new AppAction("Lots of stuff", this) {
      public void actionPerformed(ActionEvent e) { stuffAction(); } };

    AppAction InsertRectangleAction = new AppAction("Rectangle", this) {
      public void actionPerformed(ActionEvent e) { rectangleAction(); } };

    AppAction InsertRoundedRectangleAction = new AppAction("Rounded Rectangle", this) {
      public void actionPerformed(ActionEvent e) { roundedRectangleAction(); } };

    AppAction InsertEllipseAction = new AppAction("Ellipse", this) {
      public void actionPerformed(ActionEvent e) { ellipseAction(); } };

    AppAction InsertMultiPortNodeAction = new AppAction("MultiPort Node", this) {
      public void actionPerformed(ActionEvent e) { multiPortNodeAction(); } };

    AppAction InsertSimpleNodeAction = new AppAction("Simple Node", this) {
      public void actionPerformed(ActionEvent e) { simpleNodeAction(); } };

    AppAction InsertIconicNodeAction = new AppAction("Iconic Node", this) {
      public void actionPerformed(ActionEvent e) { iconicNodeAction(); } };

    AppAction InsertDraggableLabelIconicNodeAction = new AppAction("Iconic Node with Draggable Label", this) {
      public void actionPerformed(ActionEvent e) { draggableLabelIconicNodeAction(); } };

    AppAction InsertBasicNodeAction = new AppAction("Basic Node", this) {
      public void actionPerformed(ActionEvent e) { basicNodeAction(); } };

    AppAction InsertCenterLabelBasicNodeAction = new AppAction("Basic Node with Center Label", this) {
      public void actionPerformed(ActionEvent e) { centerLabelBasicNodeAction(); } };

    AppAction InsertFixedSizeCenterLabelBasicNodeAction = new AppAction("Fixed Size Basic Node", this) {
      public void actionPerformed(ActionEvent e) { fixedSizeCenterLabelBasicNodeAction(); } };

    AppAction InsertRectBasicNodeAction = new AppAction("Rectangular Basic Node", this) {
      public void actionPerformed(ActionEvent e) { rectBasicNodeAction(); } };

    AppAction InsertSelfLoopBasicNodeAction = new AppAction("Basic Node with Self Loop", this) {
      public void actionPerformed(ActionEvent e) { selfLoopBasicNodeAction(); } };

    AppAction InsertMultiSpotNodeAction = new AppAction("MultiSpot Node", this) {
      public void actionPerformed(ActionEvent e) { multiSpotNodeAction(); } };

    AppAction InsertGeneralNodeAction = new AppAction("General Node", this) {
      public void actionPerformed(ActionEvent e) { generalNodeAction(); } };

    AppAction AddLeftPortAction = new AppAction("Add Left Port", this) {
      public void actionPerformed(ActionEvent e) { addLeftPortAction(); } };

    AppAction AddRightPortAction = new AppAction("Add Right Port", this) {
      public void actionPerformed(ActionEvent e) { addRightPortAction(); } };

    AppAction InsertTextNodeAction = new AppAction("Text Node", this) {
      public void actionPerformed(ActionEvent e) { textNodeAction(); } };

    AppAction InsertFixedSizeTextNodeAction = new AppAction("Fixed Size Text Node", this) {
      public void actionPerformed(ActionEvent e) { fixedSizeTextNodeAction(); } };

    AppAction InsertRoundTextNodeAction = new AppAction("Rounded Text Node", this) {
      public void actionPerformed(ActionEvent e) { roundTextNodeAction(); } };

    AppAction InsertMultiTextNodeAction = new AppAction("MultiText Node", this) {
      public void actionPerformed(ActionEvent e) { multiTextNodeAction(); } };

    AppAction InsertListAreaAction = new AppAction("List Area", this) {
      public void actionPerformed(ActionEvent e) { listAreaAction(); } };

    AppAction InsertRecordNodeAction = new AppAction("Record Nodes", this) {
      public void actionPerformed(ActionEvent e) { recordNodeAction(); } };

    AppAction InsertStrokeAction = new AppAction("Stroke", this) {
      public void actionPerformed(ActionEvent e) { strokeAction(); } };

    AppAction InsertPolygonAction = new AppAction("Polygon", this) {
      public void actionPerformed(ActionEvent e) { polygonAction(); } };

    AppAction InsertDiamondAction = new AppAction("Diamond", this) {
      public void actionPerformed(ActionEvent e) { diamondAction(); } };

    AppAction InsertTextAction = new AppAction("Text", this) {
      public void actionPerformed(ActionEvent e) { textAction(); } };

    AppAction Insert3DRectAction = new AppAction("3D Rectangle", this) {
      public void actionPerformed(ActionEvent e) { threeDRectAction(); } };

    AppAction InsertCommentAction = new AppAction("Comment", this) {
      public void actionPerformed(ActionEvent e) { commentAction(); } };

    AppAction Insert10000Action = new AppAction("10000 Objects", this) {
      public void actionPerformed(ActionEvent e) { insert10000Action(); } };

    AppAction InsertDrawingStrokeAction = new AppAction("Draw Stroke", this) {
      public void actionPerformed(ActionEvent e) { insertDrawingStroke(); } };

    AppAction InsertGraphOfGraphsAction = new AppAction("Graph of Graphs", this) {
      public void actionPerformed(ActionEvent e) { insertGraphOfGraphs(); } };

    AppAction InsertSanKeyNodesAction = new AppAction("SanKeyNodes", this) {
      public void actionPerformed(ActionEvent e) { insertSanKeyNodes(); } };

    AppAction CutAction = new AppAction("Cut", this) {
      public void actionPerformed(ActionEvent e) { cutAction(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    AppAction CopyAction = new AppAction("Copy", this) {
      public void actionPerformed(ActionEvent e) { getView().copy(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    AppAction PasteAction = new AppAction("Paste", this) {
      public void actionPerformed(ActionEvent e) { pasteAction(); } };

    AppAction SelectAllAction = new AppAction("Select All", this) {
      public void actionPerformed(ActionEvent e) { getView().selectAll(); } };

    AppAction MoveToFrontAction = new AppAction("Move to Front", this) {
      public void actionPerformed(ActionEvent e) { moveToFrontAction(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    AppAction MoveToBackAction = new AppAction("Move to Back", this) {
      public void actionPerformed(ActionEvent e) { moveToBackAction(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    AppAction ChangeLayersAction = new AppAction("Change Layers", this) {
      public void actionPerformed(ActionEvent e) { changeLayersAction(); } };

    AppAction GroupAction = new AppAction("Group", this) {
      public void actionPerformed(ActionEvent e) { groupAction(); }
      public boolean canAct() { return super.canAct() &&
                                       (getView().getSelection().getNumObjects() >= 2); } };

    AppAction SubgraphAction = new AppAction("Make SubGraph", this) {
      public void actionPerformed(ActionEvent e) { subgraphAction(); }
      public boolean canAct() { return super.canAct() &&
                                       (getView().getSelection().getNumObjects() >= 2); } };

    AppAction UngroupAction = new AppAction("Ungroup", this) {
      public void actionPerformed(ActionEvent e) { ungroupAction(); }
      public boolean canAct() { return super.canAct() &&
                                          !getView().getSelection().isEmpty() &&
                                          (getView().getSelection().getPrimarySelection() instanceof JGoArea); } };

    AppAction InspectAction = new AppAction("Inspect", this) {
      public void actionPerformed(ActionEvent e) { inspectAction(); } };

    AppAction PropertiesAction = new AppAction("Properties", this) {
      public void actionPerformed(ActionEvent e) { propertiesAction(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    AppAction ZoomInAction = new AppAction("Zoom In", this) {
      public void actionPerformed(ActionEvent e) { zoomInAction(); }
      public boolean canAct() { return super.canAct() && (getView().getScale() < 8.0f); } };

    AppAction ZoomOutAction = new AppAction("Zoom Out", this) {
      public void actionPerformed(ActionEvent e) { zoomOutAction(); }
      public boolean canAct() { return super.canAct() && (getView().getScale() > 0.13f); } };

    AppAction ZoomNormalAction = new AppAction("Zoom Normal Size", this) {
      public void actionPerformed(ActionEvent e) { zoomNormalAction(); } };

    AppAction ZoomToFitAction = new AppAction("Zoom To Fit", this) {
      public void actionPerformed(ActionEvent e) { zoomToFitAction(); } };

    AppAction PrintPreviewAction = new AppAction("Print Preview", this) {
      public void actionPerformed(ActionEvent e) {
        PrinterJob pj = PrinterJob.getPrinterJob();
        PageFormat defaultpf = pj.validatePage(pj.defaultPage());
        // Ask the user what page size they're using, what the margins are,
        // and whether they are printing in landscape or portrait mode.
        PageFormat pf = pj.pageDialog(defaultpf);
        if (pf != defaultpf)
          getView().printPreview("Print Preview", pf);
      }
    };

    AppAction PrintAction = new AppAction("Print", this) {
      public void actionPerformed(ActionEvent e) { getView().print(); } };

    AppAction ToggleLinkIconicNodesAction = new AppAction("Toggle Link IconicNodes Mode", this) {
      public void actionPerformed(ActionEvent e) { toggleLinkIconicNodesAction(); } };

    AppAction GridAction = new AppAction("Grid", this) {
      public void actionPerformed(ActionEvent e) { gridAction(); } };

    AppAction OverviewAction = new AppAction("Overview", this) {
      public void actionPerformed(ActionEvent e) { overviewAction(); } };

    AppAction LeftAction = new AppAction("Align Left Sides", this) {
      public void actionPerformed(ActionEvent e) { leftAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction HorizontalAction = new AppAction("Align Horizontal Centers", this) {
      public void actionPerformed(ActionEvent e) { horizontalAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction RightAction = new AppAction("Align Right Sides", this) {
      public void actionPerformed(ActionEvent e) { rightAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction TopAction = new AppAction("Align Tops", this) {
      public void actionPerformed(ActionEvent e) { topAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction BottomAction = new AppAction("Align Bottoms", this) {
      public void actionPerformed(ActionEvent e) { bottomAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction VerticalAction = new AppAction("Align Vertical Centers", this) {
      public void actionPerformed(ActionEvent e) { verticalAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction SameWidthAction = new AppAction("Make Same Size Widths", this) {
      public void actionPerformed(ActionEvent e) { sameWidthAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction SameHeightAction = new AppAction("Make Same Size Heights", this) {
      public void actionPerformed(ActionEvent e) { sameHeightAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction SameBothAction = new AppAction("Make Same Size Both", this) {
      public void actionPerformed(ActionEvent e) { sameBothAction(); }
      public boolean canAct() { return super.canAct() &&
                                          (getView().getSelection().getNumObjects() >= 2); } };

    AppAction LayeredDigraphAutoLayoutAction = new AppAction("AutoLayout", this) {
      public void actionPerformed(ActionEvent e) { layeredDigraphAutoLayoutAction(); } };

  JMenuItem UndoMenuItem = null;

    AppAction UndoAction = new AppAction("Undo", this) {
      public void actionPerformed(ActionEvent e) { getView().getDocument().undo(); AppAction.updateAllActions(); }
      public boolean canAct() { return super.canAct() && (getView().getDocument().canUndo()); }
      public void updateEnabled()
      {
        super.updateEnabled();
        if (UndoMenuItem != null && getView() != null) {
          JGoUndoManager mgr = getView().getDocument().getUndoManager();
          if (mgr != null)
            UndoMenuItem.setText(mgr.getUndoPresentationName());
        }
      } };

  JMenuItem RedoMenuItem = null;

    AppAction RedoAction = new AppAction("Redo", this) {
      public void actionPerformed(ActionEvent e) { getView().getDocument().redo(); AppAction.updateAllActions(); }
      public boolean canAct() { return super.canAct() && (getView().getDocument().canRedo()); }
      public void updateEnabled()
      {
        super.updateEnabled();
        if (RedoMenuItem != null && getView() != null) {
          JGoUndoManager mgr = getView().getDocument().getUndoManager();
          if (mgr != null)
            RedoMenuItem.setText(mgr.getRedoPresentationName());
        }
      } };

  void initMenus()
  {
    //==============================================================
    // Set up menu bar
    //==============================================================
    filemenu.setText("File");
    filemenu.add(PrintPreviewAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,Event.CTRL_MASK|Event.SHIFT_MASK));
    filemenu.add(PrintAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,Event.CTRL_MASK));
    filemenu.addSeparator();
    filemenu.add(ExitAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4,Event.CTRL_MASK));
    mainMenuBar.add(filemenu);

    editmenu.setText("Edit");
    editmenu.add(CutAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,Event.CTRL_MASK));
    editmenu.add(CopyAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,Event.CTRL_MASK));
    editmenu.add(PasteAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V,Event.CTRL_MASK));
    editmenu.add(SelectAllAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A,Event.CTRL_MASK));
    editmenu.addSeparator();

    // add commands for undo/redo
    UndoMenuItem = editmenu.add(UndoAction);
    UndoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z,Event.CTRL_MASK));
    UndoMenuItem.setMnemonic('U');

    RedoMenuItem = editmenu.add(RedoAction);
    RedoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y,Event.CTRL_MASK));
    RedoMenuItem.setMnemonic('R');

    editmenu.addSeparator();

    // turn on undo/redo
    // Because this example does not have a JGoDocument subclass where we could override
    // endTransaction(String), we do it here as part of the undo manager.
    // The problem is that we need to update all of the action menu items after each
    // transaction.
    myDoc.setUndoManager(new JGoUndoManager()
      {
        public void endTransaction(String s) {
          super.endTransaction(s);
          Inspector.refresh();
          updateActions();
        }
      });

    editmenu.add(MoveToFrontAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F,Event.CTRL_MASK));
    editmenu.add(MoveToBackAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B,Event.CTRL_MASK));
    editmenu.add(ChangeLayersAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R,Event.CTRL_MASK));
    editmenu.addSeparator();
    editmenu.add(ZoomOutAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6,Event.SHIFT_MASK));
    editmenu.add(ZoomInAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6,Event.CTRL_MASK));
    editmenu.add(ZoomNormalAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6,Event.CTRL_MASK | Event.SHIFT_MASK));
    editmenu.add(ZoomToFitAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0));
    editmenu.addSeparator();
    editmenu.add(GroupAction);
    editmenu.add(SubgraphAction);
    editmenu.add(UngroupAction);
    editmenu.addSeparator();
    editmenu.add(InsertDrawingStrokeAction);
    editmenu.add(ToggleLinkIconicNodesAction);
    editmenu.addSeparator();
    editmenu.add(InspectAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4, 0));
    editmenu.add(PropertiesAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,Event.CTRL_MASK));
    editmenu.add(GridAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G,Event.CTRL_MASK));
    editmenu.add(OverviewAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O,Event.CTRL_MASK));
    mainMenuBar.add(editmenu);

    insertmenu.setText("Insert");
    insertmenu.add(InsertStuffAction);
    insertmenu.add(InsertStrokeAction);
    insertmenu.add(InsertPolygonAction);
    insertmenu.add(InsertDiamondAction);
    insertmenu.add(InsertRectangleAction);
    insertmenu.add(InsertRoundedRectangleAction);
    insertmenu.add(Insert3DRectAction);
    insertmenu.add(InsertCommentAction);
    insertmenu.add(InsertEllipseAction);
    insertmenu.add(InsertMultiPortNodeAction);
    insertmenu.add(InsertSimpleNodeAction);
    insertmenu.add(InsertIconicNodeAction);
    insertmenu.add(InsertDraggableLabelIconicNodeAction);
    insertmenu.add(InsertBasicNodeAction);
    insertmenu.add(InsertCenterLabelBasicNodeAction);
    insertmenu.add(InsertFixedSizeCenterLabelBasicNodeAction);
    insertmenu.add(InsertRectBasicNodeAction);
    insertmenu.add(InsertSelfLoopBasicNodeAction);
    insertmenu.add(InsertMultiSpotNodeAction);
    insertmenu.add(InsertGeneralNodeAction);
    insertmenu.add(InsertTextNodeAction);
    insertmenu.add(InsertFixedSizeTextNodeAction);
    insertmenu.add(InsertRoundTextNodeAction);
    insertmenu.add(InsertMultiTextNodeAction);
    insertmenu.add(InsertTextAction);
    insertmenu.add(InsertListAreaAction);
    insertmenu.add(InsertRecordNodeAction);
    insertmenu.add(Insert10000Action);
    insertmenu.add(InsertGraphOfGraphsAction);
    insertmenu.add(InsertSanKeyNodesAction);
    insertmenu.addSeparator();
    insertmenu.add(AddLeftPortAction);
    insertmenu.add(AddRightPortAction);
    mainMenuBar.add(insertmenu);

    layoutmenu.setText("Layout");
    layoutmenu.add(LeftAction);
    layoutmenu.add(HorizontalAction);
    layoutmenu.add(RightAction);
    layoutmenu.add(TopAction);
    layoutmenu.add(BottomAction);
    layoutmenu.add(VerticalAction);
    layoutmenu.addSeparator();
    layoutmenu.add(SameWidthAction);
    layoutmenu.add(SameHeightAction);
    layoutmenu.add(SameBothAction);
    layoutmenu.addSeparator();
    layoutmenu.add(LayeredDigraphAutoLayoutAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L,Event.CTRL_MASK));
    mainMenuBar.add(layoutmenu);
    setJMenuBar(mainMenuBar);

    helpmenu.setText("Help");
    helpmenu.add(AboutAction);
    mainMenuBar.add(helpmenu);
    setJMenuBar(mainMenuBar);
  }

  protected void initPalette()
  {
    JGoDocument realDoc = myDoc;
    JGoLayer realBack = myBackgroundLayer;
    JGoLayer realMain = myMainLayer;
    JGoLayer realFore = myForegroundLayer;

    JGoDocument paletteDoc = myPalette.getDocument();

    // don't care about undo/redo for the palette,
    // so don't need to call paletteDoc.fireForedate here
    paletteDoc.setSuspendUpdates(true);

    myDoc = paletteDoc;  // temporarily, so objects get created in palette
    myBackgroundLayer = paletteDoc.getDefaultLayer();
    myMainLayer = paletteDoc.getDefaultLayer();
    myForegroundLayer = paletteDoc.getDefaultLayer();
    stuffAction();  // make a bunch of stuff in the palette, into myDoc

    // remove all links between nodes from palette
    ArrayList links = new ArrayList();
    JGoListPosition pos = paletteDoc.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = paletteDoc.getObjectAtPos(pos);
      pos = paletteDoc.getNextObjectPosAtTop(pos);
      if (obj instanceof JGoLink) {
        links.add(obj);
      }
    }
    for (int i = 0; i < links.size(); i++) {
      paletteDoc.removeObject((JGoObject)links.get(i));
    }

    myDoc = realDoc;  // switch back to normal document
    myBackgroundLayer = realBack;
    myMainLayer = realMain;
    myForegroundLayer = realFore;

    paletteDoc.setSuspendUpdates(false);
    // don't care about undo/redo for the palette,
    // so don't need to call paletteDoc.fireUpdate here
  }


  /**
   * Shows or hides the component depending on the boolean flag b.
   * @param b  if true, show the component; otherwise, hide the component.
   * @see javax.swing.JComponent#isVisible
   */
  public void setVisible(boolean b)
  {
    if (b)
    {
      setLocation(50, 50);
    }
    super.setVisible(b);
  }

  static public void main(String args[])
  {
    try
    {
        UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        //UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
        //Create a new instance of our application's frame, and make it visible.
        Demo1 theApp = new Demo1();
        theApp.setVisible(true);
        theApp.initPalette();
        theApp.stuffAction();

        theApp.updateActions();
    }
    catch (Throwable t)
    {
      System.err.println(t);
      t.printStackTrace();
      //Ensure the application exits with an error condition.
      System.exit(1);
    }
  }

  static private Demo1 theApp = null;
  static public Demo1 getApp() { return theApp; }

  // the default place to put stuff if not dragged there
  Point getDefaultLocation()
  {
    // to avoid constantly putting things in the same place,
    // keep shifting the default location
    if (myDefaultLocation != null) {
      myDefaultLocation.x += 4;
      myDefaultLocation.y += 4;
    }
    return myDefaultLocation;
  }

  public void setDefaultLocation(Point loc)
  {
    myDefaultLocation = loc;
  }

  public void setRNorg(boolean bVertical)
  {
    myRNorg = bVertical;
  }
  public void setLAorg(int nOrg)
  {
    myLAorg = nOrg;
  }


  // This public method exists so that any public demo1 action methods
  // can be shared by other applications.  In particular, the JGo
  // regression test application makes used of several of these methods.
  public void setMainLayer(JGoLayer layer)
  {
    myMainLayer = layer;
  }
  public void setForegroundLayer(JGoLayer layer)
  {
    myForegroundLayer = layer;
  }

  //==============================================================
  // Process JGoDocumentEvents from the JGoDocument
  //==============================================================
  public void processDocChange(JGoDocumentEvent e)
  {
    switch(e.getHint()) {
      case JGoDocumentEvent.STARTED_TRANSACTION:
//        System.err.println("Started Transaction");
        break;
      case JGoDocumentEvent.FINISHED_TRANSACTION:
//        if (e.getPreviousValue() != null)
//          System.err.println("Finished Transaction " + e.getPreviousValue().toString());
//        else
//          System.err.println("Finished Transaction");
//        // look at all of the JGoDocumentChangedEdits that happened in this transaction
//        if (e.getObject() instanceof JGoUndoManager.JGoCompoundEdit) {
//          JGoUndoManager.JGoCompoundEdit cedit = (JGoUndoManager.JGoCompoundEdit)e.getObject();
//          Vector edits = cedit.getAllEdits();
//          for (int i = 0; i < edits.size(); i++) {
//            if (edits.get(i) instanceof JGoDocumentChangedEdit) {
//              JGoDocumentChangedEdit edit = (JGoDocumentChangedEdit)edits.get(i);
//              System.err.println("  " + edit.toString());
//            }
//          }
//        }
        break;
      case JGoDocumentEvent.ABORTED_TRANSACTION:
//        System.err.println("Aborted Transaction");
        break;
      case JGoDocumentEvent.STARTING_UNDO:
//        if (e.getObject() instanceof UndoableEdit)
//          System.err.println("Starting Undo " + ((UndoableEdit)e.getObject()).getUndoPresentationName());
//        else
//          System.err.println("Starting Undo");
        break;
      case JGoDocumentEvent.FINISHED_UNDO:
//        System.err.println("Finished Undo");
        break;
      case JGoDocumentEvent.STARTING_REDO:
//        if (e.getObject() instanceof UndoableEdit)
//          System.err.println("Starting Redo " + ((UndoableEdit)e.getObject()).getRedoPresentationName());
//        else
//          System.err.println("Starting Redo");
        break;
      case JGoDocumentEvent.FINISHED_REDO:
//        System.err.println("Finished Redo");
        break;
      case JGoDocumentEvent.INSERTED:
        if (e.getJGoObject() instanceof JGoLink) {

          // make sure each link that's created has an arrowhead at its front,
          // if the "to" port is a port on a JGoBasicNode
          JGoLink link = (JGoLink)e.getJGoObject();
          JGoPort port = link.getToPort();
          if (port != null) {
            if (port instanceof MultiPortNodePort) {
              link.setArrowHeads(false, true);
            } else if (port.getParent() instanceof JGoBasicNode) {
              // only have an arrowhead at the "to" end
              link.setArrowHeads(false, true);

              // the default values produce a traditional looking arrowhead;
              // the following values produce a diamond instead
              //link.setArrowAngle(Math.PI/3);
              //link.setArrowLength(8);
              //link.setArrowShaftLength(16);
            }

            // if the "from" port is the same as the "to" port, make sure
            // it doesn't have labels at either end, and change its middle label
            if (port == link.getFromPort() && link instanceof JGoLabeledLink) {
              JGoLabeledLink ll = (JGoLabeledLink)link;
              JGoObject mid = ll.getMidLabel();
              ll.setFromLabel(null);
              ll.setToLabel(null);
              if (mid != null && mid instanceof JGoText)
                ((JGoText)mid).setText("loop");
            }
          }

          // if it's connected to a MultiSpotNode, make it orthogonally routed
          // (not implemented as a separate class, so we're using a flag on
          //  the object as a marker)
          if (port != null &&
              port.getParent() != null) {
            if ((port.getParent().getFlags() & flagMultiSpotNode) != 0) {
              link.setOrthogonal(true);
              link.setAvoidsNodes(true);
              link.setJumpsOver(true);
            } else if (port.getParent() instanceof MultiTextNode) {
              link.setOrthogonal(true);
              link.setArrowHeads(false, true);
              link.setArrowShaftLength(0);
            }
          }

          // if the link connects ports on different kinds of objects,
          // highlight it in red
          if (link.getFromPort() != null &&
              port.getClass() != link.getFromPort().getClass()) {
            link.setHighlight(JGoPen.make(JGoPen.SOLID,
                                          link.getPen().getWidth()+4, Color.red));
          }
        }
        break;
    }
  }

  //==============================================================
  // Process JGoViewEvents from the JGoView
  //==============================================================
  public void processViewChange(JGoViewEvent e)
  {
    // if the selection changed, maybe some commands need to
    // be disabled or re-enabled
    switch(e.getHint()) {
      case JGoViewEvent.UPDATE_ALL:
      case JGoViewEvent.SELECTION_GAINED:
      case JGoViewEvent.SELECTION_LOST:
        UpdateControls();
        break;
      case JGoViewEvent.SELECTION_STARTING:
        myUpdatingSelection = true;
        break;
      case JGoViewEvent.SELECTION_FINISHED:
        myUpdatingSelection = false;
        UpdateControls();
        break;
      case JGoViewEvent.SCALE_CHANGED:
        myView.updateBorder();
        break;
    }
  }
  private boolean myUpdatingSelection = false;
  public void UpdateControls() {
    if (myUpdatingSelection) return;
    myView.updateBorder();
    if (Inspector.getInspector() != null && Inspector.getInspector().isVisible()) {
      Inspector.inspect(myView.getSelection().getPrimarySelection());
    }
    updateActions();
  }

  public void startTransaction()
  {
    myDoc.startTransaction();
  }

  public void endTransaction(String pname)
  {
    myDoc.endTransaction(pname);
  }

  //==============================================================
  // Handle the actions
  //==============================================================
  void aboutAction()
  {
    try {
      // AboutDialog Create and show as modal
      (new AboutDialog(this, "Demo1 Application - About", true)).setVisible(true);
    } catch (Exception e) {
      System.err.println(e);
    }
  }

  void exitAction()
  {
    setVisible(false);    // Hide the invoking frame
    dispose();            // Free system resources
    System.exit(0);       // close the application
  }


 public void rectangleAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(180, 250);
    JGoRectangle obj = new JGoRectangle(loc, new Dimension(75,75));
    obj.setBrush(JGoBrush.makeStockBrush(Color.red));
    myMainLayer.addObjectAtTail(obj);
    endTransaction(InsertRectangleAction.toString());
  }

  void roundedRectangleAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(180, 150);
    JGoRoundRect roundRect = new Demo1RoundRect(loc, new Dimension(80,80), new Dimension(15,30));
    roundRect.setPen(JGoPen.make(JGoPen.DASHDOTDOT, 3, Color.darkGray));
    // brush gets determined by Demo1RoundRect
    myMainLayer.addObjectAtTail(roundRect);
    endTransaction(InsertRoundedRectangleAction.toString());
  }

  public void ellipseAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 250);
    JGoEllipse ellipse = new JGoEllipse(loc, new Dimension(50,75));
    ellipse.setBrush(JGoBrush.makeStockBrush(Color.green));
    myMainLayer.addObjectAtTail(ellipse);
    endTransaction(InsertEllipseAction.toString());
  }

  void multiPortNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(10, 170);
    MultiPortNode snode = new MultiPortNode();
    JGoImage nodeicon = new JGoImage(new Rectangle(0,0,40,40));
    nodeicon.loadImage(Demo1.class.getResource("doc.gif"), true);
    snode.initialize(loc, nodeicon, "multiport node");

    int numports = (int)(getRandom()*5)+1;
    for (int i = 0; i < numports; i++) {
      Point offset = new Point(8*((int)(getRandom()*6)-1), 8*((int)(getRandom()*6)-1));
      int spot = JGoObject.Center;
      if (offset.x > offset.y) {
        if (offset.x < 40 - offset.y)
          spot = JGoObject.TopMiddle;
        else
          spot = JGoObject.SideRight;
      } else {
        if (offset.x > 40 - offset.y)
          spot = JGoObject.BottomMiddle;
        else
          spot = JGoObject.SideLeft;
      }
      MultiPortNodePort p = new MultiPortNodePort(true, true, spot,
                                            offset, new Dimension(8, 8), nodeicon, snode);
    }

    myMainLayer.addObjectAtTail(snode);
    endTransaction(InsertMultiPortNodeAction.toString());
  }

  public void simpleNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 70);
    SimpleNode snode = new SimpleNode();
    JGoImage nodeicon = new JGoImage(new Rectangle(0,0,50,50));
    nodeicon.loadImage(Demo1.class.getResource("doc.gif"), true);
    snode.initialize(loc, new Dimension(50,50), nodeicon, "a simple node", true, true);
    myMainLayer.addObjectAtTail(snode);
    endTransaction(InsertSimpleNodeAction.toString());
  }

  public JGoPort iconicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(340, 110);
    JGoIconicNode snode = new JGoIconicNode("an iconic node");
    // modify the iconic node's standard image to display a GIF
    snode.getImage().setSize(50, 50);
    snode.getImage().loadImage(Demo1.class.getResource("doc.gif"), true);
    snode.setLocation(loc);
    myMainLayer.addObjectAtTail(snode);
    endTransaction(InsertIconicNodeAction.toString());
    return snode.getPort();
  }

  JGoPort draggableLabelIconicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(440, 110);
    JGoIconicNode snode = new JGoIconicNode("draggable label");
    // replace the iconic node's standard image with a new one displaying a GIF
    JGoImage nodeicon = new JGoImage(new Rectangle(0,0,50,50));
    nodeicon.loadImage(Demo1.class.getResource("doc.gif"), true);
    snode.setIcon(nodeicon);
    snode.setLocation(loc);
    snode.setDraggableLabel(true);
    myMainLayer.addObjectAtTail(snode);
    endTransaction(InsertDraggableLabelIconicNodeAction.toString());
    return snode.getPort();
  }


  // let each newly created JGoBasicNode show an integer label
  int basicNodeCounter = 1;

  public JGoPort basicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(40, 140);
    JGoBasicNode bnode = new JGoBasicNode(Integer.toString(basicNodeCounter));
    if (basicNodeCounter%18 >= 9) {
      JGoRectangle rect = new JGoRectangle();
      rect.setSize(20, 20);
      bnode.setDrawable(rect);
    }
    bnode.setLocation(loc);
    bnode.setBrush(JGoBrush.makeStockBrush(JGoBrush.ColorOrange));
    bnode.setLabelSpot(basicNodeCounter%9);
    myMainLayer.addObjectAtTail(bnode);
    basicNodeCounter++;
    endTransaction(InsertBasicNodeAction.toString());
    return bnode.getPort();
  }

  JGoPort centerLabelBasicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(270, 30);
    JGoBasicNode tnode = new JGoBasicNode("basic node\ncenter label");
    tnode.setInsets(new Insets(10, 18, 10, 18));
    tnode.getLabel().setMultiline(true);
    tnode.setLabelSpot(JGoObject.Center);
    tnode.setBrush(JGoBrush.makeStockBrush(new Color(0x90, 0xEE, 0x90)));
    tnode.setPen(JGoPen.makeStockPen(new Color(0xFF, 0x14, 0x93)));
    tnode.setLocation(loc);
    myMainLayer.addObjectAtTail(tnode);
    endTransaction(InsertCenterLabelBasicNodeAction.toString());
    return tnode.getPort();
  }

  JGoPort fixedSizeCenterLabelBasicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(340, 350);
    JGoBasicNode tnode = new JGoBasicNode("resizable basic node with AutoResize false");
    tnode.setInsets(new Insets(10, 18, 10, 18));
    tnode.getLabel().setMultiline(true);
    tnode.setLabelSpot(JGoObject.Center);
    tnode.setBrush(JGoBrush.makeStockBrush(new Color(0x7F, 0xFF, 0xD4)));
    tnode.setLocation(loc);
    tnode.setResizable(true);
    tnode.setAutoResize(false);
    tnode.setSize(200, 50);
    myMainLayer.addObjectAtTail(tnode);
    endTransaction(InsertFixedSizeCenterLabelBasicNodeAction.toString());
    return tnode.getPort();
  }

  JGoPort rectBasicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(90, 350);
    JGoBasicNode tnode = new JGoBasicNode("rectangular basic node");
    tnode.setLabelSpot(JGoObject.Center);
    tnode.setDrawable(new JGoRectangle());
    tnode.setBrush(JGoBrush.makeStockBrush(new Color(0xC0, 0, 0xC0)));
    tnode.setLocation(loc);
    myMainLayer.addObjectAtTail(tnode);
    endTransaction(InsertRectBasicNodeAction.toString());
    return tnode.getPort();
  }

  public JGoPort selfLoopBasicNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(270, 100);
    JGoBasicNode tnode = new JGoBasicNode("self loop");
    tnode.setInsets(new Insets(10, 18, 10, 18));
    tnode.getLabel().setMultiline(true);
    tnode.setLabelSpot(JGoObject.Center);
    tnode.setLocation(loc);
    tnode.setBrush(JGoBrush.makeStockBrush(new Color(0x80, 0x00, 0x80)));
    tnode.getLabel().setBold(true);
    tnode.getLabel().setTextColor(Color.white);
    tnode.getPort().setValidSelfNode(true);
    myMainLayer.addObjectAtTail(tnode);
    JGoLink l = new JGoLink(tnode.getPort(), tnode.getPort());
    l.setArrowShaftLength(0);
    l.setArrowLength(6);
    l.setArrowWidth(8);
    l.setPen(JGoPen.makeStockPen(new Color(0x80, 0x00, 0x80)));
    myMainLayer.addObjectAtTail(l);
    l.calculateStroke();
    endTransaction(InsertSelfLoopBasicNodeAction.toString());
    return tnode.getPort();
  }

  JGoPort textNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(120, 20);
    JGoTextNode tnode = new JGoTextNode("text node");
    tnode.setTopLeft(loc);
    tnode.setBrush(JGoBrush.makeStockBrush(Color.pink));
    myMainLayer.addObjectAtTail(tnode);
    endTransaction(InsertTextNodeAction.toString());
    return tnode.getBottomPort();
  }

  JGoPort fixedSizeTextNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(290, 230);
    JGoTextNode tnode = new JGoTextNode("resizable text node with AutoResize false");
    tnode.setResizable(true);
    tnode.setAutoResize(false);
    tnode.setTopLeft(loc);
    tnode.setSize(100, 100);
    tnode.setBrush(JGoBrush.makeStockBrush(Color.yellow));
    tnode.getLabel().setEditable(true);
    tnode.getLabel().setAlignment(JGoText.ALIGN_CENTER);
    myMainLayer.addObjectAtTail(tnode);
    endTransaction(InsertTextNodeAction.toString());
    return tnode.getLeftPort();
  }

  JGoPort roundTextNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(320, 10);
    JGoTextNode rnode = new JGoTextNode("rounded text node\ndisplaying three lines\nwith wider insets");
    rnode.setTopLeft(new Point(loc.x+40, loc.y));
    rnode.getLabel().setEditable(true);
    JGoRoundRect rrect = new JGoRoundRect(new Dimension(10, 10));
    rrect.setBrush(JGoBrush.white);
    rnode.setBackground(rrect);
    rnode.setInsets(new Insets(4, 10, 4, 10));
    myMainLayer.addObjectAtTail(rnode);
    endTransaction(InsertRoundTextNodeAction.toString());
    return rnode.getTopPort();
  }

  void multiTextNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(370, 160);

    MultiTextNode tnode = new MultiTextNode();
    tnode.initialize();
    tnode.setTopLeft(loc);
    JGoText t = (JGoText)tnode.addString("MultiTextNode");
    t.setAlignment(JGoText.ALIGN_CENTER);
    t = (JGoText)tnode.addString("second line");
    t.setAlignment(JGoText.ALIGN_CENTER);
    t.setBold(true);
    t = (JGoText)tnode.addString("third line");
    t.setAlignment(JGoText.ALIGN_CENTER);
    tnode.setLinePen(JGoPen.darkGray);
    //JGoRoundRect rr = new JGoRoundRect();
    //rr.setSelectable(false);
    //rr.setArcDimension(new Dimension(20, 20));
    //tnode.setBackground(rr);
    //tnode.setInsets(new Insets(5, 5, 5, 5));  // make room for corners
    tnode.setInsets(new Insets(0, 2, 0, 2));  // add space on sides
    tnode.setItemWidth(100);
    myForegroundLayer.addObjectAtTail(tnode);

    endTransaction(InsertMultiTextNodeAction.toString());
  }

  public void recordNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(500, 100);

    RecordNode right = makeRecordNode(true, myRNorg);
    right.setTopLeft(loc.x + 200, loc.y);
    right.setHeight(150);
    right.setLinePen(JGoPen.gray);
    right.setSpacing(3);

    RecordNode left = makeRecordNode(false, myRNorg);
    left.setTopLeft(loc);
    left.setHeight(150);

    myRNorg = !myRNorg;

    JGoPort left5 = left.getRightPort(5);
    JGoPort left6 = left.getRightPort(6);
    JGoPort left7 = left.getRightPort(7);
    JGoPort left8 = left.getRightPort(8);
    JGoPort right5 = right.getLeftPort(5);
    JGoPort right10 = right.getLeftPort(10);
    JGoPort right0 = right.getLeftPort(0);
    JGoPort right19 = right.getLeftPort(19);
    if (left5 != null && right5 != null)
      myForegroundLayer.addObjectAtTail(new JGoLink(left5, right5));
    JGoImage star = new JGoImage();
    star.loadImage(Demo1.class.getResource("star.gif"), true);
    if (right5 != null) {
      right5.setSize(7, 7);
      right5.setStyle(JGoPort.StyleObject);
      right5.setPortObject(star);
    }
    if (left6 != null && right10 != null)
      myForegroundLayer.addObjectAtTail(new JGoLink(left6, right10));
    if (right10 != null) {
      right10.setSize(7, 7);
      right10.setPortObject(star);
      right10.setStyle(JGoPort.StyleObject);
    }
    if (left7 != null && right0 != null)
      myForegroundLayer.addObjectAtTail(new JGoLink(left7, right0));
    if (right0 != null) {
      right0.setSize(7, 7);
      right0.setPortObject(star);
      right0.setStyle(JGoPort.StyleObject);
    }
    if (left8 != null && right19 != null)
      myForegroundLayer.addObjectAtTail(new JGoLink(left8, right19));
    if (right19 != null) {
      right19.setSize(7, 7);
      right19.setPortObject(star);
      right19.setStyle(JGoPort.StyleObject);
    }
    if (left5 != null && right0 != null)
      myForegroundLayer.addObjectAtTail(new JGoLink(left5, right0));
    JGoPort left13 = left.getRightPort(13);
    if (left13 != null) {
      left13.setSize(9, 9);
      left13.setStyle(JGoPort.StyleDiamond);
      left13.setBrush(JGoBrush.makeStockBrush(Color.magenta));
    }

    endTransaction(InsertRecordNodeAction.toString());
  }

  static boolean myRNorg = true;

  RecordNode makeRecordNode(boolean onRight, boolean vertical)
  {
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(20, 250);
    RecordNode tnode = new RecordNode();
    tnode.initialize();
    tnode.setScrollBarOnRight(onRight);
    tnode.setVertical(vertical);
    //tnode.setAlignment(ListArea.ALIGN_RIGHT);  // icons on left, text on right

    JGoText title = new JGoText("a Record");
    title.setAlignment(JGoText.ALIGN_CENTER);
    title.setBkColor(Color.blue);
    title.setTextColor(Color.white);
    title.setBold(true);
    title.setAutoResize(false);
    title.setClipping(true);

    if (!tnode.isVertical() &&
        onRight)
      tnode.setFooter(title);
    else
      tnode.setHeader(title);

    for (int i = 0; i < 20; i++) {
      JGoObject item = makeListItem(i);

      JGoImage icon = new JGoImage();
      icon.loadImage(Demo1.class.getResource("doc.gif"), true);
      icon.setSelectable(false);
      if (tnode.isVertical())
        icon.setSize(10, 12);
      else
        icon.setSize(30, 40);

      JGoPort leftport = null;
      JGoPort rightport = null;
      if (tnode.isScrollBarOnRight()) {
        leftport = new JGoPort();
        leftport.setSize(5, 5);
        leftport.setFromSpot(JGoObject.LeftCenter);
        leftport.setToSpot(JGoObject.LeftCenter);
      } else {
        rightport = new JGoPort();
        rightport.setSize(5, 5);
        rightport.setFromSpot(JGoObject.RightCenter);
        rightport.setToSpot(JGoObject.RightCenter);
      }

      tnode.addItem(item, leftport, rightport, icon);
    }

    myForegroundLayer.addObjectAtTail(tnode);
    return tnode;
  }

  public ListArea listAreaAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(420, 350);
    ListArea tnode = new ListArea();
    tnode.initialize();
    // have fun trying different orientations and scroll bar arrangements
    switch (myLAorg) {
      default:
      case 0:  tnode.setVertical(true);  tnode.setScrollBarOnRight(true);
               myLAorg++; break;
      case 1:  tnode.setVertical(true);  tnode.setScrollBarOnRight(false);
               myLAorg++; break;
      case 2:  tnode.setVertical(false); tnode.setScrollBarOnRight(true);
               myLAorg++; break;
      case 3:  tnode.setVertical(false); tnode.setScrollBarOnRight(false);
               myLAorg = 0; break;
    }
    for (int i = 0; i < 10; i++) {
      tnode.addItem(makeListItem(i), null, null, null);
    }
    myMainLayer.addObjectAtTail(tnode);
    tnode.setTopLeft(loc.x, loc.y);
    // start it not with the first object (0)
    tnode.setFirstVisibleIndex(3);
    tnode.setAlignment(ListArea.ALIGN_LEFT);
    endTransaction(InsertListAreaAction.toString());
    return tnode;
  }

  static int myLAorg = 0;

  JGoObject makeListItem(int i)
  {
    // just for fun, have a differently sized (and looking) item
    if (i == 7) {
      JGoPolygon gon = new JGoPolygon();
      gon.setBrush(JGoBrush.makeStockBrush(Color.red));
      gon.setPen(JGoPen.make(JGoPen.SOLID, 3, Color.white));
      gon.addPoint(new Point(10,  0));
      gon.addPoint(new Point(20,  0));
      gon.addPoint(new Point(30, 10));
      gon.addPoint(new Point(30, 20));
      gon.addPoint(new Point(20, 30));
      gon.addPoint(new Point(10, 30));
      gon.addPoint(new Point( 0, 20));
      gon.addPoint(new Point( 0, 10));
      return gon;
    } else if (i == 13) {
      JGoText t = new JGoText("start a link");
      t.setDraggable(false);
      t.setTransparent(true);
      t.setSelectBackground(true);
      t.setTextColor(Color.blue);
      t.setBkColor(myView.getSecondarySelectionColor());
      t.setItalic(true);
      t.setFontSize(16);
      return t;
    } else if (i == 14) {
      JGoText t = new JGoText("not selectable");
      t.setSelectable(false);
      t.setTransparent(true);
      t.setSelectBackground(true);
      t.setTextColor(Color.red);
      t.setBkColor(myView.getSecondarySelectionColor());
      t.setItalic(true);
      return t;
    } else {
      String m = "Item " + i;
      JGoText t = new JGoText(m);
      t.setTransparent(true);
      t.setSelectBackground(true);
      t.setBkColor(myView.getSecondarySelectionColor());
      t.setFaceName("Helvetica");
      // for fun, if it's a prime number, make it bold
      if (i == 2 || i == 3 || i == 5 || i == 7 ||
          i == 11 || i == 13 || i == 17 || i == 19)
        t.setBold(true);
      return t;
    }
  }

  // use a flag on JGoObject to recognize a particular kind of area
  // (could have implemented our own subclass of JGoArea, but didn't
  //  want to bother this time)
  public static final int flagMultiSpotNode = 0x10000;

  void multiSpotNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(280, 175);
    JGoArea node = new JGoNode();
    node.setFlags(node.getFlags() | flagMultiSpotNode);
    JGoRectangle rect = new JGoRectangle(loc, new Dimension(50,50));
    rect.setSelectable(false);
    rect.setBrush(JGoBrush.makeStockBrush(Color.gray));
    node.addObjectAtHead(rect);
    addMultiSpotPort(node, rect, JGoObject.Center);
    addMultiSpotPort(node, rect, JGoObject.TopLeft);
    addMultiSpotPort(node, rect, JGoObject.TopCenter);
    addMultiSpotPort(node, rect, JGoObject.TopRight);
    addMultiSpotPort(node, rect, JGoObject.RightCenter);
    addMultiSpotPort(node, rect, JGoObject.BottomRight);
    addMultiSpotPort(node, rect, JGoObject.BottomCenter);
    addMultiSpotPort(node, rect, JGoObject.BottomLeft);
    addMultiSpotPort(node, rect, JGoObject.LeftCenter);
    myMainLayer.addObjectAtTail(node);
    endTransaction(InsertMultiSpotNodeAction.toString());
  }

  void addMultiSpotPort(JGoArea area, JGoObject rect, int spot)
  {
    JGoPort port = new JGoPort();
    port.setSize(6,6);
    port.setStyle(JGoPort.StyleDiamond);
    port.setBrush(JGoBrush.makeStockBrush(Color.magenta));
    port.setSpotLocation(spot, rect, spot);
    port.setFromSpot(spot);
    port.setToSpot(spot);
    area.addObjectAtTail(port);
  }

  // create a GeneralNode with a random number of input and output ports
  void generalNodeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(10, 230);
    GeneralNode node = new GeneralNode();
    JGoImage nodeicon = new JGoImage(new Rectangle(0,0,50,50));
    nodeicon.loadImage(Demo1.class.getResource("doc.gif"), true);
    node.initialize(loc, new Dimension(50, 50), nodeicon, "top", "general node",
                    (int)(getRandom()*5), (int)(getRandom()*5));
    myMainLayer.addObjectAtTail(node);
    endTransaction(InsertGeneralNodeAction.toString());
  }

  void addLeftPortAction()
  {
    JGoObject obj = myView.getSelection().getPrimarySelection();
    if (obj != null && obj.getParentNode() instanceof GeneralNode) {
      startTransaction();
      GeneralNode gn = (GeneralNode)obj.getParentNode();
      String pname = "in" + Integer.toString(gn.getNumLeftPorts());
      GeneralNodePort p = new GeneralNodePort(true, pname, gn);
      GeneralNodePortLabel l = new GeneralNodePortLabel(pname, p);
      gn.addLeftPort(p);
      endTransaction(AddLeftPortAction.toString());
    }
  }

  void addRightPortAction()
  {
    JGoObject obj = myView.getSelection().getPrimarySelection();
    if (obj != null && obj.getParentNode() instanceof GeneralNode) {
      startTransaction();
      GeneralNode gn = (GeneralNode)obj.getParentNode();
      String pname = "out" + Integer.toString(gn.getNumRightPorts());
      GeneralNodePort p = new GeneralNodePort(false, pname, gn);
      GeneralNodePortLabel l = new GeneralNodePortLabel(pname, p);
      gn.addRightPort(p);
      endTransaction(AddRightPortAction.toString());
    }
  }

  void strokeAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(10, 10);
    JGoStroke stroke = new JGoStroke();
    stroke.addPoint(new Point(loc.x+0, loc.y+0));
    stroke.addPoint(new Point(loc.x+68, loc.y+46));
    stroke.addPoint(new Point(loc.x+68, loc.y+90));
    stroke.addPoint(new Point(loc.x+53, loc.y+57));
    stroke.addPoint(new Point(loc.x+85, loc.y+34));
    myMainLayer.addObjectAtTail(stroke);
    endTransaction(InsertStrokeAction.toString());
  }

  void diamondAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(175, 100);
    Diamond dia3 = new Diamond(loc, new Dimension(70, 40));
    dia3.setPen(JGoPen.make(JGoPen.SOLID, 3, Color.magenta));
    myMainLayer.addObjectAtTail(dia3);
    endTransaction(InsertDiamondAction.toString());
  }

  void polygonAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(10, 80);
    JGoPolygon gon = new JGoPolygon();
    gon.setBrush(JGoBrush.makeStockBrush(Color.red));
    gon.setPen(JGoPen.make(JGoPen.SOLID, 3, Color.lightGray));
    gon.addPoint(new Point(loc.x+10, loc.y+ 0));
    gon.addPoint(new Point(loc.x+20, loc.y+ 0));
    gon.addPoint(new Point(loc.x+30, loc.y+10));
    gon.addPoint(new Point(loc.x+30, loc.y+20));
    gon.addPoint(new Point(loc.x+20, loc.y+30));
    gon.addPoint(new Point(loc.x+10, loc.y+30));
    gon.addPoint(new Point(loc.x+ 0, loc.y+20));
    gon.addPoint(new Point(loc.x+ 0, loc.y+10));
    myMainLayer.addObjectAtTail(gon);
    endTransaction(InsertPolygonAction.toString());
  }

  void textAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(150, 50);
    JGoText text = new JGoText(loc,
                  12,
                  "Sample Text",
                  "Serif",
                  true,
                  true,
                  true,
                  JGoText.ALIGN_CENTER,
                  false,
                  true);
    text.setResizable(true);
    text.setEditOnSingleClick(true);
    myMainLayer.addObjectAtTail(text);
    endTransaction(InsertTextAction.toString());
  }

  void threeDRectAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 150);
    JGo3DRect obj = new JGo3DRect(loc, new Dimension(55,25));
    obj.setBrush(JGoBrush.makeStockBrush(Color.cyan));
    myMainLayer.addObjectAtTail(obj);
    endTransaction(Insert3DRectAction.toString());
  }

  void commentAction()
  {
    startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 200);
    Comment obj = new Comment("This is a\nmultiline comment.");
    obj.setTopLeft(loc.x, loc.y);
    myMainLayer.addObjectAtTail(obj);
    endTransaction(InsertCommentAction.toString());
  }

  void insert10000Action()
  {
    startTransaction();
    JGoPen red = JGoPen.makeStockPen(Color.red);
    JGoPen green = JGoPen.makeStockPen(Color.green);
    JGoPen blue = JGoPen.makeStockPen(Color.blue);
    for (int i = 0; i < 100; i++) {
      for (int j = 0; j < 100; j++) {
        // JGoDrawable obj = new JGoRectangle();
        // obj.setResizable(false);
        // obj.setTopLeft(i*40, j*40);
        // obj.setSize(20, 20);
        JGoBasicNode obj = new JGoBasicNode(Integer.toString(i) + "," + Integer.toString(j));
        obj.setLocation(new Point(i*40, j*40));
        switch ((i*100+j) % 6) {
          default:
          case 0: obj.setPen(JGoPen.black); break;
          case 1: obj.setPen(red); break;
          case 2: obj.setPen(JGoPen.gray); break;
          case 3: obj.setPen(JGoPen.black); break;
          case 4: obj.setPen(green); break;
          case 5: obj.setPen(blue); break;
        }
        myMainLayer.addObjectAtTail(obj);
      }
    }
    endTransaction(Insert10000Action.toString());
  }

  void insertDrawingStroke()
  {
    javax.swing.JOptionPane.showMessageDialog(this,
          "Starting a mode to allow you to draw a stroke by specifying its points.\n" +
          "This mode stops when you type ENTER to accept the stroke or ESCAPE to cancel the new link.",
          "Drawing Stroke Mode",
          javax.swing.JOptionPane.INFORMATION_MESSAGE);

    myView.startDrawingStroke();
  }

  public void insertGraphOfGraphs()
  {
    startTransaction();
    JGoSubGraph gr1 = makeSubGraph("one", "1");
    gr1.setTopLeft(300, 100);
    JGoSubGraph gr2 = makeSubGraph("two", "2");
    gr2.setCollapsible(false);
    gr2.setTopLeft(100, 300);
    gr2.setBackgroundColor(new Color(255, 200, 200, 63));
    gr2.setBorderPen(JGoPen.make(JGoPen.DASHDOT, 2, Color.gray));
    JGoSubGraph gr3 = makeSubGraph("three", "3");
    gr3.setTopLeft(500, 300);
    gr3.setBackgroundColor(new Color(200, 255, 200, 63));
    gr3.setBorderPen(JGoPen.makeStockPen(Color.magenta));
    JGoLabeledLink l2 = new JGoLabeledLink(findBasicNode(gr1, "1b").getPort(),
                                           findBasicNode(gr2, "2a").getPort());
    JGoLinkLabel l2l = new JGoLinkLabel("one to two");
    l2.setMidLabel(l2l);
    JGoLabeledLink l3 = new JGoLabeledLink(findBasicNode(gr1, "1c").getPort(),
                                           findBasicNode(gr3, "3a").getPort());
    JGoLinkLabel l3l = new JGoLinkLabel("one to three");
    l3.setMidLabel(l3l);
    myMainLayer.addObjectAtTail(gr1);
    myMainLayer.addObjectAtTail(gr2);
    myMainLayer.addObjectAtTail(gr3);
    myMainLayer.addObjectAtTail(l2);
    myMainLayer.addObjectAtTail(l3);
    JGoSubGraph gr4 = makeSubGraph("four", "4");
    gr4.setTopLeft(550, 350);
    gr3.addObjectAtTail(gr4);
    endTransaction(InsertGraphOfGraphsAction.toString());
  }

  JGoSubGraph makeSubGraph(String labtext, String prefix)
  {
    JGoBasicNode a = new JGoBasicNode(prefix + "a");
    a.setLocation(new Point(30, 0));
    JGoBasicNode b = new JGoBasicNode(prefix + "b");
    b.setLocation(new Point(0, 60));
    JGoBasicNode c = new JGoBasicNode(prefix + "c");
    c.setLocation(new Point(60, 60));
    JGoLink ab = new JGoLink(a.getPort(), b.getPort());
    JGoLink ac = new JGoLink(a.getPort(), c.getPort());
    JGoSubGraph gr = new JGoSubGraph(labtext);
    gr.addObjectAtTail(a);
    gr.addObjectAtTail(b);
    gr.addObjectAtTail(c);
    gr.addObjectAtTail(ab);
    gr.addObjectAtTail(ac);
    return gr;
  }

  JGoBasicNode findBasicNode(JGoObjectSimpleCollection sg, String lab)
  {
    JGoListPosition pos = sg.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = sg.getObjectAtPos(pos);
      pos = sg.getNextObjectPosAtTop(pos);
      if (obj instanceof JGoBasicNode) {
        JGoBasicNode n = (JGoBasicNode)obj;
        if (n.getLabel().getText().equals(lab)) {
          return n;
        }
      }
    }
    return null;
  }

  void insertSanKeyNodes() {
    startTransaction();
    SanKeyNode snode = new SanKeyNode("star");
    // modify the iconic node's standard image to display a GIF
    snode.getImage().setSize(50, 50);
    snode.getImage().loadImage(Demo1.class.getResource("star.gif"), true);
    snode.setLocation(450, 350);
    myForegroundLayer.addObjectAtTail(snode);

    SanKeyNode snode2 = new SanKeyNode("doc1");
    snode2.getImage().setSize(50, 50);
    snode2.getImage().loadImage(Demo1.class.getResource("doc.gif"), true);
    snode2.setLocation(600, 300);
    myForegroundLayer.addObjectAtTail(snode2);

    SanKeyNode snode3 = new SanKeyNode("doc2");
    snode3.getImage().setSize(50, 50);
    snode3.getImage().loadImage(Demo1.class.getResource("doc.gif"), true);
    snode3.setLocation(600, 400);
    myForegroundLayer.addObjectAtTail(snode3);

    JGoLink link1 = new JGoLink(snode.getPort(), snode2.getPort());
    link1.setPen(JGoPen.make(JGoPen.SOLID, 20, Color.blue));
    myMainLayer.addObjectAtTail(link1);

    JGoLink link2 = new JGoLink(snode.getPort(), snode3.getPort());
    link2.setPen(JGoPen.make(JGoPen.SOLID, 10, new Color(0, 128, 0)));
    myMainLayer.addObjectAtTail(link2);
    endTransaction(InsertGraphOfGraphsAction.toString());
  }

  public void stuffAction()
  {
    startTransaction();
    Point oldloc = getDefaultLocation();
    setDefaultLocation(null); // position everything at absolute default locations
    rectangleAction();
    ellipseAction();
    multiPortNodeAction();
    simpleNodeAction();
    JGoPort p1 = iconicNodeAction();
    JGoPort p2 = draggableLabelIconicNodeAction();
    basicNodeAction();
    roundedRectangleAction();
    centerLabelBasicNodeAction();
    JGoPort p3 = fixedSizeCenterLabelBasicNodeAction();
    JGoPort p5 = rectBasicNodeAction();
    selfLoopBasicNodeAction();
    multiSpotNodeAction();
    generalNodeAction();
    textNodeAction();
    JGoPort p4 = fixedSizeTextNodeAction();

    JGoLink link1 = new JGoLink();
    link1.setArrowHeads(false, true);
    link1.setBrush(JGoBrush.makeStockBrush(Color.magenta));
    link1.setCubic(true);
    link1.setFromPort(p1);
    link1.setToPort(p2);
    myMainLayer.addObjectAtTail(link1);

    JGoLink link2 = new JGoLink(p3, p4);
    myMainLayer.addObjectAtTail(link2);

    JGoLabeledLink link3 = new JGoLabeledLink(p5, p3);
    JGoLinkLabel lab3 = new JGoLinkLabel("mid");
    link3.setArrowHeads(false, true);
    link3.setArrowShaftLength(0);
    link3.setMidLabel(lab3);
    link3.setBrush(null);
    myMainLayer.addObjectAtTail(link3);

    JGoImage img = new JGoImage(new Rectangle(25,375,50,50));
    img.loadImage(Demo1.class.getResource("star.gif"), false);
    myDoc.addObjectAtTail(img);

    TestIconicNode tin = new TestIconicNode("test");
    tin.getImage().loadImage(Demo1.class.getResource("star.gif"), false);
    tin.setTopLeft(10, 450);
    myDoc.addObjectAtTail(tin);

    TestSubGraph sg = new TestSubGraph("test subgraph");

    JGoImage sgimg = new JGoImage();
    sgimg.loadImage(Demo1.class.getResource("star.gif"), true);
    sgimg.setSize(40, 40);
    sgimg.setSelectable(false);
    sgimg.setDragsNode(true);
    sgimg.setVisible(false);
    sg.setCollapsedObject(sgimg);

    JGoBasicNode sgbn1 = new JGoBasicNode("bn1");
    sgbn1.setLocation(100, 500);
    sg.addObjectAtTail(sgbn1);
    JGoBasicNode sgbn2 = new JGoBasicNode("bn2");
    sgbn2.setLocation(200, 500);
    sg.addObjectAtTail(sgbn2);
    JGoLabeledLink sglink1 = new JGoLabeledLink(sgbn1.getPort(), sgbn2.getPort());
    JGoLinkLabel lllabel = new JGoLinkLabel();
    lllabel.setText("middle");
    sglink1.setMidLabel(lllabel);
    sglink1.setArrowHeads(false, true);
    sg.addObjectAtTail(sglink1);
    myDoc.addObjectAtTail(sg);

    TestSubGraph2 sg2 = new TestSubGraph2("test subgraph2");

    JGoImage sgimg2 = new JGoImage();
    sgimg2.loadImage(Demo1.class.getResource("star.gif"), true);
    sgimg2.setSize(40, 40);
    sgimg2.setSelectable(false);
    sgimg2.setDragsNode(true);
    sgimg2.setVisible(false);
    sg2.setCollapsedObject(sgimg2);

    JGoBasicNode sg2bn1 = new JGoBasicNode("bn1");
    sg2bn1.setLocation(300, 500);
    sg2.addObjectAtTail(sg2bn1);
    JGoBasicNode sg2bn2 = new JGoBasicNode("bn2");
    sg2bn2.setLocation(400, 500);
    sg2.addObjectAtTail(sg2bn2);
    JGoLink sg2link1 = new JGoLink(sg2bn1.getPort(), sg2bn2.getPort());
    sg2link1.setArrowHeads(false, true);
    sg2.addObjectAtTail(sg2link1);
    myDoc.addObjectAtTail(sg2);

    roundTextNodeAction();
    multiTextNodeAction();
    strokeAction();
    polygonAction();
    diamondAction();
    textAction();
    threeDRectAction();
    commentAction();
//    listAreaAction();
//    recordNodeAction();
    insertSanKeyNodes();

    DecoratedTextNode dtn = new DecoratedTextNode("decorated");
    dtn.setLocation(123, 123);
    myDoc.addObjectAtTail(dtn);

    LinearGradientEllipse lge = new LinearGradientEllipse();
    lge.setBoundingRect(100, 380, 100, 50);
    myDoc.addObjectAtTail(lge);

    setDefaultLocation(oldloc);
    endTransaction(InsertStuffAction.toString());
  }

  void inspectAction()
  {
    Inspector.inspect(myView.getSelection().getPrimarySelection());
    JDialog inspector = Inspector.getInspector();
    if (inspector != null && !inspector.isVisible())
      inspector.setVisible(true);
  }

  void cutAction()
  {
    startTransaction();
    myView.cut();
    endTransaction(CutAction.toString());
  }

  void pasteAction()
  {
    startTransaction();
    myView.paste();
    endTransaction(PasteAction.toString());
  }

  void propertiesAction()
  {
    startTransaction();
    try {
      doEditProperties();
    } catch (Exception e) {
      System.err.println(e);
    }
    endTransaction(PropertiesAction.toString());
  }

   /**
    * Display a form to edit the selected JGoObject's properties
    */
    public void doEditProperties()
    {
      JGoSelection selection = myView.getSelection();
      if (!selection.isEmpty()) {
        JGoListPosition pos = selection.getFirstObjectPos();
        while (pos != null) {
          JGoObject obj = selection.getObjectAtPos(pos);
          pos = selection.getNextObjectPos(pos);
          callDialog(obj);
        }
      }
    }

  boolean myIconicNodesAreLinkable = false;
  void toggleLinkIconicNodesAction()
  {
    myIconicNodesAreLinkable = !myIconicNodesAreLinkable;
    JGoDocument doc = myView.getDocument();
    JGoListPosition pos = doc.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = doc.getObjectAtPos(pos);
      pos = doc.getNextObjectPosAtTop(pos);
      if (obj instanceof JGoIconicNode) {
        JGoIconicNode n = (JGoIconicNode)obj;
        n.getPort().setVisible(myIconicNodesAreLinkable);
      }
    }
  }

  // create an image for a JGoObject
  public BufferedImage getObjectImage(JGoObject obj, JGoView view) {
    Rectangle r = new Rectangle(obj.getBoundingRect());
    obj.expandRectByPenWidth(r);
    BufferedImage img = new BufferedImage(r.width, r.height, BufferedImage.TYPE_INT_ARGB);
    Graphics2D g2 = img.createGraphics();
    g2.translate(-r.x, -r.y);
    view.applyRenderingHints(g2);
    obj.paint(g2, view);
    g2.dispose();
    return img;
  }

  void gridAction()
  {
    try {
      GridOptionsDialog dlg = new GridOptionsDialog(this, "Grid View Options", true, myView);
      dlg.setVisible(true);
    } catch (Exception e) {
      System.err.println(e);
      e.printStackTrace();
    }
  }


  // make the overview window visible
  void overviewAction()
  {
    if (myOverview == null) {
      myOverview = new JGoOverview();
      myOverview.setObserved(myView);
      myOverviewDialog = new JDialog((Frame)Demo1.getApp(), "Overview", false);
      myOverviewDialog.getContentPane().setLayout(new BorderLayout());
      myOverviewDialog.getContentPane().add(myOverview, BorderLayout.CENTER);
    }
    myOverviewDialog.pack();
    myOverviewDialog.setVisible(true);
  }


  // these two actions now work for selected objects that
  // are part of areas as well as for top-level document objects
  void moveToFrontAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);

      if (obj.getParent() != null)
        obj.getParent().bringObjectToFront(obj);
      else
        myDoc.bringObjectToFront(obj);
    }
    endTransaction(MoveToFrontAction.toString());
  }

  void moveToBackAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);

      if (obj.getParent() != null)
        obj.getParent().sendObjectToBack(obj);
      else
        myDoc.sendObjectToBack(obj);
    }
    endTransaction(MoveToBackAction.toString());
  }


  void zoomInAction()
  {
    double newscale = Math.rint(myView.getScale() / 0.9f * 100f) / 100f;
    myView.setScale(newscale);
    updateActions();
  }

  void zoomOutAction()
  {
    double newscale = Math.rint(myView.getScale() * 0.9f * 100f) / 100f;
    myView.setScale(newscale);
    updateActions();
  }

  void zoomNormalAction()
  {
    double newscale = 1;
    myView.setScale(newscale);
    updateActions();
  }

  void zoomToFitAction()
  {
    double newscale = 1;
    if (!myDoc.isEmpty()){
      double extentWidth = myView.getExtentSize().width;
      double printWidth = myView.getPrintDocumentSize().width;
      double extentHeight = myView.getExtentSize().height;
      double printHeight = myView.getPrintDocumentSize().height;
      newscale = Math.min((extentWidth / printWidth),(extentHeight / printHeight));
    }
    if (newscale > 2)
      newscale = 1;
    newscale *= myView.getScale();
    myView.setScale(newscale);
    myView.setViewPosition(0, 0);
    updateActions();
  }

  public void centerInView(Rectangle r, JGoView view) {
    Dimension s = view.getExtentSize();
    int newx, newy;
    if (r.width < s.width)
      newx = r.x + r.width/2 - s.width/2;
    else
      newx = r.x;
    if (r.height < s.height)
      newy = r.y + r.height/2 - s.height/2;
    else
      newy = r.y;
    view.setViewPosition(newx, newy);
  }


  // make a JGoNode out of the current selection
  void groupAction() {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject primary = selection.getPrimarySelection();
    JGoArea area = new JGoNode();
    // now add the area to the document
    primary.getLayer().addObjectAtTail(area);
    // add the selected objects to the area
    ArrayList arr = area.addCollection(selection, false, myMainLayer);
    // update the selection too, for the user's convenience
    selection.selectObject(area);
    // make each object not Selectable
    for (int i = 0; i < arr.size(); i++) {
      JGoObject obj = (JGoObject)arr.get(i);
      obj.setSelectable(false);
    }
    endTransaction(GroupAction.toString());
  }

  // make a JGoSubGraph out of the current selection
  void subgraphAction() {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject primary = selection.getPrimarySelection();
    JGoSubGraph sg = new JGoSubGraph();
    sg.initialize("subgraph!");
    // now add the area to the document
    primary.getLayer().addObjectAtTail(sg);
    // add the selected objects to the area
    sg.addCollection(selection, true, myMainLayer);
    // update the selection too, for the user's convenience
    selection.selectObject(sg);
    endTransaction(SubgraphAction.toString());
  }

  // this action only works on the primary selection, which should
  // be a JGoSubGraph
  void ungroupAction() {
    JGoSelection selection = myView.getSelection();
    JGoObject primary = selection.getPrimarySelection();
    if (primary == null) return;
    if (!(primary instanceof JGoArea)) return;
    JGoLayer layer = primary.getLayer();
    if (layer == null) return;
    startTransaction();
    JGoArea area = (JGoArea)primary;
    ArrayList arr = null;

    // special cases for subgraphs--don't promote Handle or Label
    // or other special subgraph children as stand-alone objects
    if (area instanceof JGoSubGraph) {
      JGoSubGraph sg = (JGoSubGraph)area;
      ArrayList children = new ArrayList();
      JGoListPosition pos = sg.getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = sg.getObjectAtPos(pos);
        pos = sg.getNextObjectPosAtTop(pos);
        // check for special subgraph children
        if (obj != sg.getHandle() && obj != sg.getLabel() &&
            !(obj instanceof JGoPort) && obj != sg.getCollapsedObject())
          children.add(obj);
      }
      arr = layer.addCollection(children, true, myMainLayer);
    } else {
      arr = layer.addCollection(area, false, myMainLayer);
    }

    for (int i = 0; i < arr.size(); i++) {
      JGoObject obj = (JGoObject)arr.get(i);
      // update the selectability and visibility, just in case
      obj.setSelectable(true);
      obj.setVisible(true);
      // update the selection too, for the user's convenience
      selection.extendSelection(obj);
    }
    
    // delete the area
    layer.removeObject(area);
    endTransaction(UngroupAction.toString());
  }


  //align all object's left sides to the left side of the primary selection
  void leftAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    Point p = obj.getSpotLocation(JGoObject.TopLeft);
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (!(temp instanceof JGoLink))
        temp.setSpotLocation(JGoObject.TopLeft, new Point(p.x,temp.getTop()));
    }
    endTransaction(LeftAction.toString());
  }

  //align all object's horizontal centers to the horizontal center of the primary selection
  void horizontalAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    Point p = obj.getSpotLocation(JGoObject.Center);
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (!(temp instanceof JGoLink))
        temp.setSpotLocation(JGoObject.TopCenter, new Point(p.x,temp.getTop()));
    }
    endTransaction(HorizontalAction.toString());
  }

  //align all object's right sides to the right side of the primary selection
  void rightAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    Point p = obj.getSpotLocation(JGoObject.TopRight);
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (!(temp instanceof JGoLink))
        temp.setSpotLocation(JGoObject.TopRight, new Point(p.x,temp.getTop()));
    }
    endTransaction(RightAction.toString());
  }

  //align all object's tops to the top of the primary selection
  void topAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    Point p = obj.getSpotLocation(JGoObject.Top);
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (!(temp instanceof JGoLink))
        temp.setSpotLocation(JGoObject.TopLeft, new Point(temp.getLeft(),p.y));
    }
    endTransaction(TopAction.toString());
  }

  //align all object's bottoms to the bottom of the primary selection
  void bottomAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    Point p = obj.getSpotLocation(JGoObject.Bottom);
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (!(temp instanceof JGoLink))
        temp.setSpotLocation(JGoObject.BottomLeft, new Point(temp.getLeft(),p.y));
    }
    endTransaction(BottomAction.toString());
  }

  //align all object's vertical centers to the vertical center of the primary selection
  void verticalAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    Point p = obj.getSpotLocation(JGoObject.Center);
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (!(temp instanceof JGoLink))
        temp.setSpotLocation(JGoObject.LeftCenter, new Point(temp.getLeft(),p.y));
    }
    endTransaction(VerticalAction.toString());
  }

  // this makes the widths of all objects equal to the width of the main selection.
  void sameWidthAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    int width = obj.getWidth();
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (temp.isTopLevel() && !(temp instanceof JGoLink))
        temp.setWidth(width);
    }
    endTransaction(SameWidthAction.toString());
  }

  // this makes the heights of all objects equal to the height of the main selection.
  void sameHeightAction()
  {
    startTransaction();
    JGoSelection selection = myView.getSelection();
    JGoObject obj = selection.getPrimarySelection();
    JGoListPosition pos = selection.getFirstObjectPos();
    while (obj != null && obj instanceof JGoLink && pos != null){
      obj = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
    }
    if (obj == null)
      return;
    int width = obj.getHeight();
    while (pos != null) {
      JGoObject temp = selection.getObjectAtPos(pos);
      pos = selection.getNextObjectPos(pos);
      if (temp.isTopLevel() && !(temp instanceof JGoLink))
        temp.setHeight(width);
    }
    endTransaction(SameHeightAction.toString());
  }

  // this makes the heights and widths of all objects equal to the height and
  //width of the main selection.
  void sameBothAction()
  {
    startTransaction();
    sameHeightAction();
    sameWidthAction();
    endTransaction(SameBothAction.toString());
  }

  void layeredDigraphAutoLayoutAction() {
    JGoDocument doc = myView.getDocument();
    doc.startTransaction();
    layoutCollection(doc, doc);
    doc.endTransaction("LayeredDigraph AutoLayout");
  }

  public void layoutCollection(JGoDocument doc, JGoObjectSimpleCollection coll) {
    JGoListPosition pos = coll.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = coll.getObjectAtPos(pos);
      pos = coll.getNextObjectPosAtTop(pos);

      if (obj instanceof JGoSubGraph) {  // recursively lay out subgraphs; make sure they are expanded
        JGoSubGraph sg = (JGoSubGraph)obj;
        boolean collapsed = !sg.isExpanded();
        if (collapsed) sg.expand();
        layoutCollection(doc, sg);
        if (collapsed) sg.collapse();
      }
    }

    // create a JGoNetwork so we can control exactly which nodes (and links) are to be laid out
    JGoNetwork net = new JGoNetwork(coll);
    pos = coll.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = coll.getObjectAtPos(pos);
      pos = coll.getNextObjectPosAtTop(pos);

      if (obj instanceof JGoLabeledLink) {  // don't lay out the labels of a labeled link
        JGoLabeledLink ll = (JGoLabeledLink)obj;
        net.deleteNode(ll.getFromLabel());
        net.deleteNode(ll.getMidLabel());
        net.deleteNode(ll.getToLabel());
      } else if (!(obj instanceof JGoNode) && !(obj instanceof JGoLink)) {  // ignore other objects
        net.deleteNode(obj);
      }
    }

    if (coll instanceof JGoSubGraph) {  // don't lay out any of the special subgraph children
      JGoSubGraph sg = (JGoSubGraph)coll;
      net.deleteNode(sg.getHandle());
      net.deleteNode(sg.getLabel());
      net.deleteNode(sg.getPort());
      net.deleteNode(sg.getCollapsedObject());
    }

    JGoLayeredDigraphAutoLayout layout = new JGoLayeredDigraphAutoLayout(doc, net);
    layout.setDirectionOption(JGoLayeredDigraphAutoLayout.LD_DIRECTION_RIGHT);
    layout.performLayout();
  }

  void changeLayersAction()
  {
    startTransaction();
    JGoSelection sel = myView.getSelection();
    if (sel.isEmpty()) {
      //myDoc.bringLayerToFront(myDoc.getFirstLayer());
      myMainLayer.setTransparency((float)getRandom());
      myBackgroundLayer.setVisible(getRandom() >= 0.5d);
    } else {
      // find which layer the selection is in
      // (well, the first object in the selection)
      JGoObject firstobj = sel.getObjectAtPos(sel.getFirstObjectPos());
      JGoLayer layer = firstobj.getLayer();
      // let's choose the layer behind that one
      layer = layer.getPrevLayer();
      if (layer == null)
        layer = myForegroundLayer;
      // now move all the selected objects to that layer
      JGoListPosition pos = sel.getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = sel.getObjectAtPos(pos);
        pos = sel.getNextObjectPosAtTop(pos);

        layer.addObjectAtTail(obj);
      }
      myBackgroundLayer.setVisible(true);
    }
    endTransaction(ChangeLayersAction.toString());
  }

  // this calls the appropriate dialog box for the type of object that obj is.
  void callDialog(JGoObject obj)
  {
    startTransaction();
    myView.getSelection().clearSelectionHandles(obj);
    String title = null;
    if (obj instanceof JGoStroke) {
      title = "Stroke Properties";
      StrokeDialog dlg = new StrokeDialog(myView.getFrame(), title, true, (JGoStroke)obj);
      dlg.setVisible(true);
    } else if (obj instanceof JGoText) {
      title = "Text Properties";
      TextPropsDialog dlg = new TextPropsDialog(myView.getFrame(), title, true, (JGoText)obj);
      dlg.setVisible(true);
    } else if (obj instanceof JGoPort) {
      title = "Port Properties";
      PortPropsDialog dlg = new PortPropsDialog(myView.getFrame(), title, true, (JGoPort)obj);
      dlg.setVisible(true);
    } else if (obj instanceof JGoDrawable) {
      title = "Drawable Properties";
      DrawablePropsDialog dlg = new DrawablePropsDialog(myView.getFrame(), title, true, (JGoDrawable)obj);
      dlg.setVisible(true);
    } else if (obj instanceof JGoImage) {
      title = "Image Properties";
      ImagePropsDialog dlg = new ImagePropsDialog(myView.getFrame(), title, true, (JGoImage)obj);
      dlg.setVisible(true);
    } else if (obj instanceof JGoObject) {
      title = "Object Properties";
      ObjectPropsDialog dlg = new ObjectPropsDialog(myView.getFrame(), title, true, obj);
      dlg.setVisible(true);
    }
    myDoc.update();
    myView.getSelection().restoreSelectionHandles(obj);
    endTransaction(title);
  }

  // this enables/disables all known actions
  static public void updateActions()
  {
    AppAction.updateAllActions();
  }

  static public JGoDocument loadObjects(InputStream ins)
    throws IOException, ClassNotFoundException
  {
    ObjectInputStream istream = new ObjectInputStream(ins);
    Object newObj = istream.readObject();
    if (newObj instanceof JGoDocument) {
      JGoDocument doc = (JGoDocument)newObj;
      return doc;
    } else {
      return null;
    }
  }

  public void storeObjects(OutputStream outs)
    throws IOException
  {
    ObjectOutputStream ostream = new ObjectOutputStream(outs);
    ostream.writeObject(getCurrentView().getDocument());
    ostream.flush();
  }

  protected double getRandom()
  {
    if (isRandom())
      return Math.random();
    else
      return 0.57;
  }

  public void setRandom(boolean isRandom)  {myRandom = isRandom;}

  public boolean isRandom() {return myRandom;}


  public Demo1View getCurrentView() { return myView; }


  // basic components for this app
  protected Demo1View myView;
  protected JGoDocument myDoc;
  protected JGoLayer myMainLayer;
  protected JGoLayer myForegroundLayer;
  protected JGoLayer myBackgroundLayer;

  protected JSplitPane myPaletteAndList;
  protected JScrollPane myListScrollPane;
  protected JDialog myOverviewDialog;

  protected JGoPalette myPalette;

  protected Demo1List myList;

  protected JGoOverview myOverview;

  // menu bar
  protected JMenuBar mainMenuBar = new JMenuBar();
  protected JMenu filemenu = new JMenu();
  protected JMenu editmenu = new JMenu();
  protected JMenu insertmenu = new JMenu();
  protected JMenu layoutmenu = new JMenu();
  protected JMenu helpmenu = new JMenu();

  protected Point myDefaultLocation = new Point(10, 10);
  protected boolean myRandom = true;
}
