/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.demo1;
import java.awt.*;
import java.awt.event.*;
import java.awt.dnd.*;
import java.awt.datatransfer.*;
import java.awt.geom.*;
import java.awt.print.*;
import javax.swing.*;
import javax.swing.border.*;
import java.util.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.examples.*;

// This example class also includes support for MouseWheel events,
// which is only available when using J2SDK/JRE 1.4 or later.

public class Demo1View extends JGoView implements MouseWheelListener
{
  Demo1View()
  {
    super();
    //setDebugFlags(DebugEvents);
    setIncludingNegativeCoords(true);
    setHidingDisabledScrollbars(true);
    //setDragsRealtime(false);
    //setDragsSelectionImage(false);
    addMouseWheelListener(this);
  }

  public void mouseWheelMoved(MouseWheelEvent e)
  {
    int clicks = e.getWheelRotation();

    if (e.isControlDown()) {
      Point oldpos = getViewPosition();
      Point oldpt = viewToDocCoords(e.getPoint());
      setScale(getScale() * (1 + ((double)clicks)/20));
      Point focus = viewToDocCoords(e.getPoint());
      setViewPosition(oldpos.x + oldpt.x - focus.x, oldpos.y + oldpt.y - focus.y);
      return;
    }

    JScrollBar bar;
    if (e.isShiftDown()) {
      bar = getHorizontalScrollBar();
    } else {
      bar = getVerticalScrollBar();
    }
    if (bar == null)
      return;

    int direction = clicks > 0 ? 1 : -1;
    int bound;
    if (direction == 1)
      bound = bar.getMinimum();
    else
      bound = bar.getMaximum();
    int step;
    if (e.getScrollType() == MouseWheelEvent.WHEEL_BLOCK_SCROLL)
      step = bar.getBlockIncrement(direction);
    else
      step = bar.getUnitIncrement(direction);

    int oldv = bar.getValue();
    int newv = oldv + direction * step;
    if (direction == 1) {
      newv = Math.max(newv, bound);
    } else {
      newv = Math.min(newv, bound);
    }
    bar.setValue(newv);
  }


  // new kinds of "modes"
  public static final int MouseStateDrawingStroke = MouseStateLast+1;

  public void doCancelMouse() {
    myMouseUpDocPoint.x = -9999;
    myMouseUpDocPoint.y = -9999;
    if (getState() == MouseStateDrawingStroke) {
      cancelDrawingStroke();
    } else {
      super.doCancelMouse();
    }
  }

  public void onKeyEvent(KeyEvent evt)
  {
    int t = evt.getKeyCode();
    if (t == KeyEvent.VK_ESCAPE) {
      doCancelMouse();
    } else if (t == KeyEvent.VK_ENTER && getState() == MouseStateDrawingStroke) {
      JGoStroke myStroke = (JGoStroke)getCurrentObject();
      if (myStroke != null && myStroke.getNumPoints() > 2) {
        // stroke always includes a point where the mouse is, which isn't
        // supposed to be included in the finished JGoStroke
        finishDrawingStroke();
      } else {
        doCancelMouse();
      }
    } else {
      super.onKeyEvent(evt);
    }
  }

  // This implements additional mouse behaviors on mouse down:
  // - start or continue creating a stroke with a user-defined path
  // - if on a record node field that is not Draggable, then we
  //   start drawing a link from that field's port
  public boolean doMouseDown(int modifiers, Point dc, Point vc)
  {
    if (getState() == MouseStateDrawingStroke) {
      addPointToDrawingStroke(modifiers, dc, vc);
      return true;
    }
    JGoObject obj = pickDocObject(dc, true);
    if (obj != null) {
      JGoObject field = obj;
      if (!field.isDraggable() &&
          field.getParent() != null &&
          field.getParent() instanceof ListArea &&
          field.getParent().getParent() != null &&
          field.getParent().getParent() instanceof RecordNode) {
        RecordNode node = (RecordNode)field.getParent().getParent();
        int idx = node.findItem(field);
        if (idx >= 0) {
          JGoPort port = node.getLeftPort(idx);
          if (port == null)
            port = node.getRightPort(idx);
          if (port != null &&
              startNewLink(port, dc)) {
            return true;
          }
        }
      }
    }
    // otherwise implement the default behavior
    return super.doMouseDown(modifiers, dc, vc);
  }

  // This implements additional mouse behaviors on mouse down:
  // - when creating a link with a user-defined path, have the end follow
  //   the mouse pointer between clicks
  public boolean doMouseMove(int modifiers, Point dc, Point vc)
  {
    if (getState() == MouseStateDrawingStroke) {
      followPointerForDrawingStroke(modifiers, dc, vc);
      return true;
    }
    return super.doMouseMove(modifiers, dc, vc);
  }

  private JPopupMenu myPopupMenu = new JPopupMenu();
  private Point myMouseUpDocPoint = new Point(0, 0);

  // This implements additional mouse behaviors on mouse up:
  // - mouse up must be ignored when the user is specifying stroke points
  // - if it's a right button mouse down bring up a properties
  //   dialog for the object
  public boolean doMouseUp(int modifiers, Point dc, Point vc)
  {
//    if ((modifiers & InputEvent.BUTTON3_MASK) != 0) {
//      ArrayList found = getDocument().pickObjects(dc, true, null, 99999);
//      String msg = "";
//      for (int i = 0; i < found.size(); i++) {
//        msg += found.get(i).toString() + "\r\n";
//      }
//      JOptionPane.showMessageDialog(null, msg);
//      return true;
//    }

    myMouseUpDocPoint.x = dc.x;
    myMouseUpDocPoint.y = dc.y;

    if (getState() == MouseStateDrawingStroke) {
      // don't need to do anything
      return false;
    }
    if ((modifiers & InputEvent.BUTTON3_MASK) != 0) {
      JGoObject obj = pickDocObject(dc, true);
      if (obj != null) {
        // the right-mouse-button was used
        // normally OBJ is a selectable object;  if the Control
        // key was also held down, we try to operate on the immediate
        // object at the point, even if not selectable
        if ((modifiers & InputEvent.CTRL_MASK) != 0) {
          JGoObject o = pickDocObject(dc, false);
          if (o != null)
            obj = o;
        } else {
          selectObject(obj);
        }
        if (obj instanceof JGoLink && (modifiers & InputEvent.CTRL_MASK) == 0) {
          JGoLink link = (JGoLink)obj;
          JPopupMenu popup = myPopupMenu;
          popup.removeAll();
          AppAction insertPointAction = new AppAction("Insert Point", getFrame()) {
            public void actionPerformed(ActionEvent e) { insertPointIntoLink(); } };
          popup.add(insertPointAction);
          if (link.getNumPoints() > (link.isOrthogonal() ? 6 : 2)) {
            AppAction removeSegmentAction = new AppAction("Remove Segment", getFrame()) {
              public void actionPerformed(ActionEvent e) { removeSegmentFromLink(); } };
            popup.add(removeSegmentAction);
          }
          popup.show(this, vc.x, vc.y);
        } else {
          doCancelMouse();
          Demo1 app = (Demo1)getFrame();
          app.callDialog(obj);
        }
        return true;
      }
    }
    // otherwise implement the default behavior
    return super.doMouseUp(modifiers, dc, vc);
  }

  // don't start drawing a new link if the port is selectable
  public boolean startNewLink(JGoPort port, Point dc) {
    if (port.isSelectable()) return false;
    return super.startNewLink(port, dc);
  }

  public void startDrawingStroke() {
    setState(MouseStateDrawingStroke);
    getDocument().startTransaction();
    setCurrentObject(null);
    setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
  }

  void cancelDrawingStroke() {
    if (!hasFocus())
      return;
    if (getCurrentObject() != null) {
      removeObject(getCurrentObject());
      setCurrentObject(null);
    }
    setCursor(getDefaultCursor());
    getDocument().endTransaction(false);
    setState(MouseStateNone);
  }

  void addPointToDrawingStroke(int modifiers, Point dc, Point vc) {
    // keep adding points to the stroke representing the future link
    JGoStroke s = (JGoStroke)getCurrentObject();
    if (s == null) {
      // create a new stroke starting at or near the first mouse down point
      s = new JGoStroke();
      s.addPoint(dc);
      addObjectAtTail(s);
      setCurrentObject(s);
    }
    s.addPoint(dc);
  }

  void followPointerForDrawingStroke(int modifiers, Point dc, Point vc) {
    JGoStroke myStroke = (JGoStroke)getCurrentObject();
    if (myStroke != null) {
      int numpts = myStroke.getNumPoints();
      myStroke.setPoint(numpts-1, dc);
    }
  }

  void finishDrawingStroke() {
    JGoStroke myStroke = (JGoStroke)getCurrentObject();
    if (myStroke != null) {
      // get rid of the stroke from the view
      removeObject(myStroke);
      // remove last point, which had been following the pointer
      myStroke.removePoint(myStroke.getNumPoints()-1);
      getDocument().getDefaultLayer().addObjectAtTail(myStroke);
      selectObject(myStroke);
      setCurrentObject(null);
      setCursor(getDefaultCursor());
      getDocument().endTransaction("created stroke");
      setState(MouseStateNone);
    }
  }


  // popup menu commands

  void insertPointIntoLink()
  {
    if (getSelection().getPrimarySelection() instanceof JGoLink) {
      JGoLink s = (JGoLink)getSelection().getPrimarySelection();
      int i = s.getSegmentNearPoint(myMouseUpDocPoint);
      if (s.getNumPoints() > 3) {
        if (i < 1)
          i = 1;  // don't add to first segment
        else if (i >= s.getNumPoints()-2)
          i = s.getNumPoints()-3;  // don't add to last segment
      }
      Point a = s.getPoint(i);
      Point b = s.getPoint(i+1);
      Point closest = new Point((a.x+b.x)/2, (a.y+b.y)/2);
      getDocument().startTransaction();
      s.insertPoint(i+1, closest);
      if (s.isOrthogonal())  // when orthogonal, gotta insert two points
        s.insertPoint(i+1, closest);
      getSelection().toggleSelection(s);
      selectObject(s);
      getDocument().endTransaction("inserted point into link stroke");
    }
  }

  void removeSegmentFromLink()
  {
    if (getSelection().getPrimarySelection() instanceof JGoLink) {
      JGoLink s = (JGoLink)getSelection().getPrimarySelection();
      int i = s.getSegmentNearPoint(myMouseUpDocPoint);
      getDocument().startTransaction();
        if (s.isOrthogonal()) {  // will have at least 7 points
          // don't remove either first two or last two segments
          i = Math.max(i, 2);
          i = Math.min(i, s.getNumPoints()-5);
          Point a = s.getPoint(i);
          Point b = s.getPoint(i+1);
          s.removePoint(i);
          // to maintain orthogonality, gotta remove two points
          s.removePoint(i);
          // now fix up following point to maintain orthogonality
//          Point c = new Point(s.getPoint(i));
          Point c = new Point(s.getPoint(i).x, s.getPoint(i).y);
          if (a.x == b.x) {
            c.y = a.y;
          } else {
            c.x = a.x;
          }
          s.setPoint(i, c);
        } else {  // will have at least 3 points
          i = Math.max(i, 1);  // don't remove point 0
          i = Math.min(i, s.getNumPoints()-2);  // don't remove last point
          s.removePoint(i);
        }
      getSelection().toggleSelection(s);
      selectObject(s);
      getDocument().endTransaction("removed segment from link stroke");
    }
  }

  // demonstrate how to detect a double-click in the background--i.e., not on any object
  // (for single-click in background, see doBackgroundClick below...)
  public boolean doMouseDblClick(int modifiers, Point dc, Point vc)
  {
    boolean result = super.doMouseDblClick(modifiers, dc, vc);
    if (!result && pickDocObject(dc,false) == null) {
      getDocument().startTransaction();
      LimitedNode n = new LimitedNode("limited");
      n.setLocation(dc);
      getDocument().addObjectAtTail(n);
      getDocument().endTransaction("added LimitedNode in background");
    }
    return result;
  }

  // If a right button mouse click, bring up the grid properties dialog
  public void doBackgroundClick(int modifiers, Point dc, Point vc)
  {
    // System.err.println("single background click " + Integer.toString(modifiers));
    if ((modifiers & InputEvent.BUTTON3_MASK) != 0) {
      Demo1 app = (Demo1)getFrame();
      app.gridAction();
    } else {
      super.doBackgroundClick(modifiers, dc, vc);
    }
  }


  // For this demo, provide visual feedback regarding where something will be
  // dropped from a different window, while dragging
  protected JGoRectangle myGhost = new JGoRectangle(new Point(0, 0), new Dimension());

  public void dragOver(DropTargetDragEvent e)
  {
    super.dragOver(e);
    if (e.getDropAction() != DnDConstants.ACTION_NONE) {
      if (myGhost.getView() != this) {
        // set a default size for the ghost rectangle
        myGhost.setSize(50, 50);
        addObjectAtTail(myGhost);
      }
      myGhost.setTopLeft(viewToDocCoords(e.getLocation()));
    }
  }

  public void dragExit(DropTargetEvent e)
  {
    if (myGhost.getView() == this) {
      removeObject(myGhost);
    }
    super.dragExit(e);
  }

  // support both dragging from the palette or other JGoView as well
  // as from the List of AppActions represented by their String names
  public boolean isDropFlavorAcceptable(DropTargetDragEvent e)
  {
    return super.isDropFlavorAcceptable(e) ||
           e.isDataFlavorSupported(DataFlavor.stringFlavor);
  }

  // Let drag-and-drops from other windows add JGoObjects to the JGoSubGraph
  // where the drop occurs.
  //
  //public boolean doDrop(DropTargetDropEvent e, JGoCopyEnvironment copyenv) {
  //  Point viewCoord = e.getLocation();
  //  Point docCoord = viewToDocCoords(viewCoord);
  //  JGoSubGraph sg = pickSubGraph(docCoord);
  //  boolean result = super.doDrop(e, copyenv);
  //  if (result && sg != null) {
  //    sg.addCollection(getSelection(), true, getDocument().getDefaultLayer());
  //  }
  //  return result;
  //}

  //public JGoSubGraph pickSubGraph(Point p) {
  //  JGoObject obj = pickDocObject(p, true);
  //  if (obj == null) return null;
  //  if (obj instanceof JGoSubGraph) return (JGoSubGraph)obj;
  //  if (obj.getParentNode() != null && obj.getParentNode().getParent() instanceof JGoSubGraph)
  //    return (JGoSubGraph)obj.getParentNode().getParent();
  //  return null;
  //}
  
  // Implement drop behavior when coming from another component
  public void drop(DropTargetDropEvent e)
  {
    // Try the standard drop action.
    // For this demo application, we'll modify the copied objects
    // to make them resizable.  Since the default implementation of
    // drop doesn't make the copy environment accessible, we need to
    // allocate it here.
    JGoCopyEnvironment map = getDocument().createDefaultCopyEnvironment();
    getDocument().startTransaction();
    if (doDrop(e, map)) {
      // if the default drop action succeeded, let's nudge the
      // copied top-level objects, just for fun
      Iterator i = map.values().iterator();
      while (i.hasNext()) {
        Object o = i.next();
        if (o instanceof JGoObject) {
          JGoObject obj = (JGoObject)o;
          if (obj.isTopLevel()) {
            obj.setTopLeft(obj.getLeft()+1, obj.getTop()+1);
          }
        }
      }
      fireUpdate(JGoViewEvent.EXTERNAL_OBJECTS_DROPPED, 0, null);
      // done with the document changes
      getDocument().endTransaction("Drop");
      return;
    }
    // otherwise try a command action
    try {
      if (e.isDataFlavorSupported(DataFlavor.stringFlavor)) {
        Transferable tr = e.getTransferable();
        Object data = tr.getTransferData(DataFlavor.stringFlavor);
        e.acceptDrop(e.getDropAction());
        // execute command
        String actname = (String)data;
        Vector allActs = AppAction.allActions();
        // find the appropriate command by name
        for (int i = 0; i < allActs.size(); i++) {
          AppAction act = (AppAction)allActs.elementAt(i);
          if (act.toString().equals(actname)) {
            // set default location to be mouse drop point
            Point docloc = viewToDocCoords(e.getLocation());
            Demo1 app = (Demo1)getFrame();
            app.setDefaultLocation(docloc);
            // execute the action
            act.actionPerformed(null);
            break;
          }
        }
        completeDrop(e, true);
        fireUpdate(JGoViewEvent.EXTERNAL_OBJECTS_DROPPED, 0, null);
        // done with the document changes
        getDocument().endTransaction("Dropped Action");
        return;
      }
    } catch (Exception x) {
      x.printStackTrace();
    }
    // can't do anything with the drop
    e.rejectDrop();
    // abort the document transaction
    getDocument().endTransaction(false);
  }

  // make use of the TitledBorder to show the current zoom scale
  // and (optionally) the number of objects selected
  public void updateBorder()
  {
    // change border to reflect current scale
    TitledBorder b = (TitledBorder)getBorder();
    String msg = "Demo1 View: ";
    int scale = (int)(getScale() * 100);
    msg = msg + String.valueOf(scale);
    msg = msg + "%";
// WARNING: turning on this code to display the number of selected
// objects will slow down the display.  In particular, selecting
// or deselecting a lot of objects will take a long time updating
// this counter display.
//    msg = msg + ",  ";
//    int numsel = getSelection().getNumObjects();
//    msg = msg + numsel;
//    msg = msg + " selected";
    b.setTitle(msg);
    Insets insets = getInsets();
    paintImmediately(0, 0, getWidth(), insets.top);
  }


  // this is a helper function for deciding whether
  // to create a labeled link or not
  public boolean makeLabeledLinkForPort(JGoPort p)
  {
    if (p.getParent() instanceof JGoBasicNode)
      return true;
    JGoArea area = p.getParent();
    if (area != null &&
        (area.getFlags() & Demo1.flagMultiSpotNode) != 0)
      return true;
    return false;
  }

  // This method is called by JGoView when the user has
  // performed the gestures for creating a new link between
  // two ports.  We need to create the link, add it to the
  // document, and finish the transaction.
  public void newLink(JGoPort from, JGoPort to)
  {
    JGoLink l = null;
    if (makeLabeledLinkForPort(from) &&
        makeLabeledLinkForPort(to)) {
      JGoLabeledLink link = new JGoLabeledLink(from, to);
      l = link;

      JGoLinkLabel text;
      if (!(from.getParent() instanceof JGoBasicNode)) {
        text = new JGoLinkLabel("from");
        text.setAlignment(JGoText.ALIGN_LEFT);
        text.setSelectable(true);
        text.setEditOnSingleClick(true);
        link.setFromLabel(text);
      }

      text = new JGoLinkLabel("middle");
      text.setAlignment(JGoText.ALIGN_CENTER);
      text.setSelectable(true);
      text.setEditable(true);
      text.setEditOnSingleClick(true);
      link.setMidLabel(text);

      if (!(to.getParent() instanceof JGoBasicNode)) {
        text = new JGoLinkLabel("to");
        text.setAlignment(JGoText.ALIGN_RIGHT);
        text.setEditOnSingleClick(true);
        link.setToLabel(text);
      }
    } else {
      l = new JGoLink(from, to);
    }

    JGoSubGraphBase.reparentToCommonSubGraph(l, from, to, true, from.getDocument().getLinksLayer());

    fireUpdate(JGoViewEvent.LINK_CREATED, 0, l);
    from.getDocument().endTransaction("new link");
  }


  // printing support

  //protected PageFormat printShowPageDialog(PrinterJob pj) {
  //  PageFormat defaultpf = pj.validatePage(pj.defaultPage());
  //  defaultpf.setOrientation(PageFormat.LANDSCAPE);
  //  return defaultpf;
  //}

  public Rectangle2D.Double getPrintPageRect(Graphics2D g2, PageFormat pf)
  {
    // leave some space at the bottom for a footer
    return new Rectangle2D.Double(pf.getImageableX(), pf.getImageableY(),
                                  pf.getImageableWidth(), pf.getImageableHeight() - 20);
  }


  public double getPrintScale(Graphics2D g2, PageFormat pf)
  {
    // three different options:

    // (A) always print at standard scale
/*
    return 1.0d;
*/

    // (B) print using view's currently selected scale
    return getScale();

    // (C) scale to fit printed page
/*
    Rectangle2D.Double pageRect = getPrintPageRect(g2, pf);
    Dimension docSize = getPrintDocumentSize();
    // make sure it doesn't get scaled too much! (especially if no objects in document)
    docSize.width = Math.max(docSize.width, 50);
    docSize.height = Math.max(docSize.height, 50);
    double hratio = pageRect.width / docSize.width;
    double vratio = pageRect.height / docSize.height;
    return Math.min(hratio, vratio);
*/
  }


  public void printDecoration(Graphics2D g2, PageFormat pf, int hpnum, int vpnum)
  {
    // draw corners around the getPrintPageRect area
    super.printDecoration(g2, pf, hpnum, vpnum);

    // print the n,m page number in the footer
    String msg = Integer.toString(hpnum);
    msg += ", ";
    msg += Integer.toString(vpnum);

    Paint oldpaint = g2.getPaint();
    g2.setPaint(Color.black);
    Font oldfont = g2.getFont();
    g2.setFont(new Font(JGoText.getDefaultFontFaceName(), Font.PLAIN, 10));
    g2.drawString(msg, (int)(pf.getImageableX() + pf.getImageableWidth()/2),
                       (int)(pf.getImageableY() + pf.getImageableHeight() - 10));
    g2.setPaint(oldpaint);
    g2.setFont(oldfont);
  }

  // provide tooltips indicating layer for all objects except those in the middle
  // layer, for which the default behavior applies--looking for object-specific
  // tooltip strings.
  public String getToolTipText(MouseEvent evt)
  {
    if (!isMouseEnabled()) return null;

    Point p = evt.getPoint();
    Point dc = viewToDocCoords(p);

    JGoObject obj = pickDocObject(dc,false);

    if (obj != null) {
      Object id = obj.getLayer().getIdentifier();
      if (id != null)
        id = id.toString();
      if (id != null)
        return (String)id;
    }

    // otherwise, just have the normal behavior
    return super.getToolTipText(evt);
  }

/*
  // a gradient for the view, regardless of where the view is scrolled
  protected void paintPaperColor(Graphics2D g2, Rectangle r)
  {
    Point vp = getViewPosition();
    Dimension vs = getExtentSize();
    GradientPaint gp = new GradientPaint(vp.x, vp.y, Color.white, vp.x+vs.width, vp.y+vs.height, new Color(178, 223, 238));
    g2.setPaint(gp);
    g2.fillRect(r.x, r.y, r.width, r.height);
  }
*/
/*
  // a gradient for the document, at a fixed document position
  GradientPaint myGP = new GradientPaint(100, 100, Color.white, 400, 400, new Color(178, 223, 238));
  protected void paintPaperColor(Graphics2D g2, Rectangle r)
  {
    g2.setPaint(myGP);
    g2.fillRect(r.x, r.y, r.width, r.height);
  }
*/

}
