/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.family;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.Vector;
import java.util.HashMap;
import java.util.Iterator;
import com.nwoods.jgo.*;
import com.nwoods.jgo.examples.Comment;
// If you want to use the JGoLayout package, use the following line
// and change the definition of layoutNodes below.
/*
import com.nwoods.jgo.layout.JGoLayeredDigraphAutoLayout;
*/

/**
 * FamilyTreeDoc is a JGoDocument that manages the JGoObjects used to display
 * family tree information in a JGoView.
 * <p>
 * The primary purpose is to convert the database information in a GenDB
 * object (with its Persons) into JGoObjects: PersonNodes and JGoLinks
 * representing marriages and resultant children.
 * <p>
 * There are three kinds of top-level objects in the document:
 * a PersonNode representing Person,
 * a JGoLabeledLink representing a marriage between two Persons,
 * a JGoLink linking a child with the marriage between its mother and father.
 * <p>
 * For each of these three kind of objects, there are "find..." and "get..."
 * methods:  findNode(), getNode(), findMarriageLink(), getMarriageLink(),
 * findChildLink(), and getChildLink().  The "find..." versions just look
 * for an existing object representing the Person or relationship between
 * Persons.  The "get..." versions return either the existing object or a
 * newly created JGoObject corresponding to the given Person or relationship.
 * <p>
 * This design is unusual in that it the label of a marriage link is not
 * a JGoLinkLabel (a JGoText), but a JGoPort, which in turn is attached to
 * links to all of that marriage's children.
 * <p>
 * There are also methods to lay out the PersonNodes in the document in
 * a potentially reasonable way, starting with older generations at the
 * top, placing spouses next to each other, with their children below them.
 */
public class FamilyTreeDoc extends JGoDocument
{
  public FamilyTreeDoc()
  {
    myDB.load();
  }

  public void initNodes()
  {
    // don't care about undo/redo, so don't need to call fireForedate here
    setSuspendUpdates(true);

    // Iterate over all the Persons in the database
    Iterator it = myDB.getIterator();
    while (it.hasNext()) {
      Person p = (Person)it.next();

      // Make sure the PersonNode exists for the Person
      PersonNode pn = getNode(p);

      // Get the mother and create the PersonNode for her
      // if such a Person exists
      Person mother = myDB.findPerson(p.mother);
      PersonNode mothernode = getNode(mother);

      // Get the father and create the PersonNode for him
      // if such a Person exists
      Person father = myDB.findPerson(p.father);
      PersonNode fathernode = getNode(father);

      // Create a child link
      getChildLink(fathernode, mothernode, pn);

      // Create any marriage links for this Person
      Vector marriages = p.spouseslist;
      for (int i = 0; i < marriages.size(); i++) {

        // Person stores references to other persons not
        // as pointers, but as integer identifiers
        Integer spouseID = (Integer)marriages.get(i);

        // Get the spouse as a Person
        Person spouse = myDB.findPerson(spouseID.intValue());

        // Make sure the PersonNode exists for the spouse
        PersonNode spousenode = getNode(spouse);

        // Now we can make sure the JGoLink exists between them
        getMarriageLink(pn, spousenode);
      }
    }

    setSuspendUpdates(false);
    // don't care about undo/redo, so don't need to call fireUpdate here

    // try to position all the PersonNodes nicely
    layoutNodes();

    // add a caption
    // don't care about undo/redo, so don't need to call fireForedate here
    setSuspendUpdates(true);
    Comment caption = new Comment(
                    "Some of the Tudors of English royalty in the 1500's.\n" +
                    "\n" +
                    "Women are pink; men are blue; marriage links are green.\n" +
                    "\n" +
                    "Tooltips provide more information about each person.\n" +
                    "DELETE to remove selected people from the tree;\n" +
                    "CTRL-L to reposition all the people;\n" +
                    "INSERT to start over with the original tree;\n" +
                    "CTRL-P to print; CTRL-Q to quit."
                    );
    caption.setTopLeft(350, 400);
    addObjectAtHead(caption);
    setSuspendUpdates(false);
    // don't care about undo/redo, so don't need to call fireUpdate here
  }


  // If we already know about a PersonNode for a given object,
  // return it, else return null
  public PersonNode findNode(Person key)
  {
    if (key == null) return null;
    Object val = myMap.get(key);
    if (val instanceof PersonNode)
      return (PersonNode)val;
    else
      return null;
  }

  // Get a PersonNode for a given object, using an existing one
  // if possible, otherwise creating one
  public PersonNode getNode(Person key)
  {
    if (key == null) return null;
    PersonNode node = findNode(key);
    if (node == null) {
      node = new PersonNode("");
      node.setPerson(key);  // initialize label and keep back pointer
      // PersonNodes are always in front of any links
      addObjectAtTail(node);
      // keep track of Person --> PersonNode mapping
      myMap.put(key, node);
    }
    return node;
  }

  // Override the standard document behavior to clean up the entry
  // in the hash table from Person to PersonNode
  public void removeObject(JGoObject obj)
  {
    if (obj instanceof PersonNode) {
      PersonNode node = (PersonNode)obj;
      Person key = node.getPerson();
      if (key != null)
        myMap.remove(key);
    }
    super.removeObject(obj);
  }

  // Override the standard document behavior to clean up all the
  // entries in the hash table from Person to PersonNode
  public void deleteContents()
  {
    myMap.clear();
    super.deleteContents();
  }

  // If we can find a marriage link between the two given people,
  // return it
  public JGoLabeledLink findMarriageLink(PersonNode p1, PersonNode p2)
  {
    if (p1 == null) return null;
    if (p2 == null) return null;

    // marriage links are always connected at the JGoTextNode's RightPort
    JGoPort p1p = p1.getRightPort();
    JGoPort p2p = p2.getRightPort();
    JGoListPosition pos = p1p.getFirstLinkPos();
    while (pos != null) {
      JGoLink link = p1p.getLinkAtPos(pos);
      pos = p1p.getNextLinkPos(pos);

      JGoPort other = link.getOtherPort(p1p);
      if (other == p2p)
        return (JGoLabeledLink)link;
    }
    return null;
  }

  // Create a marriage link between the two given people if it does
  // not already exist
  public JGoLabeledLink getMarriageLink(PersonNode p1, PersonNode p2)
  {
    if (p1 == null) return null;
    if (p2 == null) return null;

    JGoLabeledLink l = findMarriageLink(p1, p2);
    if (l == null) {
    // marriage links are always connected at the JGoTextNode's RightPort
      JGoPort p1p = p1.getRightPort();
      JGoPort p2p = p2.getRightPort();
      // create a labeled link, with the middle label being a port!
      l = new JGoLabeledLink(p1p, p2p);
      l.setSelectable(false);
      l.setPen(myMarriageLinkPen);
      JGoPort midp = new JGoPort();
      midp.setStyle(JGoPort.StyleHidden);
      midp.setFromSpot(JGoObject.BottomCenter);
      midp.setToSpot(JGoObject.BottomCenter);
      l.setMidLabel(midp);
      // marriage links are always behind any nodes
      addObjectAtHead(l);
    }
    return l;
  }

  // To improve the appearance of child links, the port on the
  // marriage link for the child links can be at either end as
  // well as at the middle
  public JGoPort getMarriageLinkPortForChild(JGoLabeledLink link)
  {
    JGoPort mp = (JGoPort)link.getMidLabel();
    if (mp == null)
      mp = (JGoPort)link.getToLabel();
    if (mp == null)
      mp = (JGoPort)link.getFromLabel();
    return mp;
  }

  // If we can find a child link between the given person and a marriage
  // link between the two given parents, return it
  public JGoLink findChildLink(PersonNode p1, PersonNode p2, PersonNode c)
  {
    if (p1 == null) return null;
    if (p2 == null) return null;
    if (c == null) return null;

    // make sure there's a marriage between p1 and p2
    JGoLabeledLink m = findMarriageLink(p1, p2);
    if (m == null) return null;
    JGoPort mp = getMarriageLinkPortForChild(m);

    // look at c's parentport's link
    JGoPort cp = c.getTopPort();
    JGoListPosition pos = cp.getFirstLinkPos();
    if (pos != null) {
      JGoLink link = cp.getLinkAtPos(pos);
      // found a child link--is it the right one?
      JGoPort cmp = link.getOtherPort(cp);
      if (cmp == mp)
        return link;
    }
    return null;
  }

  // Return an existing child link, if any, or else make sure a
  // marriage link exists between the two parents and then create a
  // child link from the marriage link to the child node
  public JGoLink getChildLink(PersonNode p1, PersonNode p2, PersonNode c)
  {
    if (p1 == null) return null;
    if (p2 == null) return null;
    if (c == null) return null;

    JGoLink cl = findChildLink(p1, p2, c);
    if (cl == null) {
      JGoLabeledLink m = getMarriageLink(p1, p2);
      JGoPort mp = getMarriageLinkPortForChild(m);
      // parent port is always the TopPort
      JGoPort cp = c.getTopPort();
      cl = new JGoLink(mp, cp);
      cl.setSelectable(false);
      if (c.getPerson().isMale())
        cl.setPen(mySonLinkPen);
      else if (c.getPerson().isFemale())
        cl.setPen(myDaughterLinkPen);
      else
        cl.setPen(JGoPen.black);
      // child links are always behind any nodes
      addObjectAtHead(cl);
    }
    return cl;
  }

  // Place all the people where they belong.  Start with the parent-less
  // person with the lowest ID, call layoutTree() for that person, and
  // then call layoutTree() repeatedly while unplaced people remain.

// To use the JGoLayout automatic layout library,
// uncomment the import com.nwoods.jgo.layout.* line at the beginning of this file,
// and use this definition of the layoutNodes method instead of the following one.
//
// Also, uncomment the contents of FamilyLDAL.java and FamilyNetwork.java
//
// *****Note*****
// You must have purchased the JGoAutoLayout package, or have the
// evaluation version in order to use the layered digraph auto layout code.
/*
  public void layoutNodes()
  {
    if (isEmpty()) return;
    FamilyNetwork net = new FamilyNetwork(this);
    FamilyLDAL f = new FamilyLDAL(this, net);
    f.performLayout();
  }
*/
  public void layoutNodes()
  {
    Rectangle rect = new Rectangle(10, 10, 0, 0);  // initial top-left position

    // don't care about undo/redo, so don't need to call fireForedate here
    setSuspendUpdates(true);

    PersonNode root = null;
    int lowestID = 99999999;

    // move all nodes so they appear unpositioned
    JGoListPosition pos = getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = getObjectAtPos(pos);
      pos = getNextObjectPosAtTop(pos);
      if (obj instanceof PersonNode) {
        PersonNode node = (PersonNode)obj;
        node.setTopLeft(0, 0);
        Person p = node.getPerson();
        if ((p.number < lowestID) &&
            node.getTopPort().hasNoLinks()) {
          root = node;
          lowestID = p.number;
        }
      }
    }

    // start with Person with lowest ID having no parents
    layoutTree(root, rect);
    rect.x = rect.x + rect.width;
    rect.width = 0;

    // find all parentless, unpositioned known person nodes and lay them out
    pos = getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = getObjectAtPos(pos);
      pos = getNextObjectPosAtTop(pos);

      if (obj instanceof PersonNode) {
        PersonNode node = (PersonNode)obj;
        Person p = node.getPerson();

        if (!isPositioned(node) &&
            node.getTopPort().hasNoLinks()) {
          layoutTree(node, rect);
          rect.x = rect.x + rect.width;
          rect.width = 0;
        }
      }
    }

    setSuspendUpdates(false);
    // don't care about undo/redo, so don't need to call fireUpdate here
  }

  // implement a simple tree-layout algorithm
  // ORIGRECT is modified to reflect the resultant placements
  private void layoutTree(PersonNode node, Rectangle origrect)
  {
    if (isPositioned(node)) {
      return;
    }

    int spousewidth = 0;

    Rectangle childrect = new Rectangle(0, 0, 0, 0);
    childrect.x = origrect.x + origrect.width;
    childrect.y = origrect.y + node.getHeight() + myVertSeparation;
    childrect.width = 0;
    // childrect.height is ignored

    // iterate through all the person's marriages
    JGoPort outp = node.getRightPort();
    JGoListPosition pos = outp.getFirstLinkPos();
    while (pos != null) {
      JGoLabeledLink link = (JGoLabeledLink)outp.getLinkAtPos(pos);
      pos = outp.getNextLinkPos(pos);

      JGoPort spousep = link.getOtherPort(outp);
      PersonNode spousenode = (PersonNode)spousep.getParent();  // parent area is PersonNode

      // iterate over the children of this marriage
      JGoPort mp = getMarriageLinkPortForChild(link);
      // put the marriageportforchild near the spouse, to handle
      // multiple spouses more reasonably
      if (link.getFromPort() == spousep)
        link.setFromLabel(mp);
      else if (link.getToPort() == spousep)
        link.setToLabel(mp);
      // now look at each child
      JGoListPosition childpos = mp.getFirstLinkPos();
      while (childpos != null) {
        JGoLink childlink = mp.getLinkAtPos(childpos);
        childpos = mp.getNextLinkPos(childpos);

        JGoPort childp = childlink.getOtherPort(mp);
        PersonNode childnode = (PersonNode)childp.getParent();

        layoutTree(childnode, childrect);
      }

      // now position the spouse immediately below the node
      if (!isPositioned(spousenode)) {
        spousenode.setTopLeft(origrect.x + origrect.width + spousewidth,
                              origrect.y + node.getHeight());
        spousewidth += spousenode.getWidth();
      }
    }

    // figure out the maximum width for the node, all the spouses,
    // and all the children
    int parentwidth = Math.max(node.getWidth(), spousewidth);
    int subtreewidth = Math.max(parentwidth, childrect.width);

    // place this node, centered
    node.setSpotLocation(JGoObject.TopCenter,
                origrect.x + origrect.width + subtreewidth/2, origrect.y);

    // place all the spouses, centered
    if (spousewidth < subtreewidth) {
      // figure out horizontal offset
      int offset = (subtreewidth - spousewidth)/2;
      pos = outp.getFirstLinkPos();
      while (pos != null) {
        JGoLabeledLink link = (JGoLabeledLink)outp.getLinkAtPos(pos);
        pos = outp.getNextLinkPos(pos);

        JGoPort spousep = link.getOtherPort(outp);
        PersonNode spousenode = (PersonNode)spousep.getParent();  // parent area is PersonNode

        // only adjust a spouse if it's located here
        if ((spousenode.getLeft() >= origrect.x + origrect.width) &&
            (spousenode.getLeft() <= origrect.x + origrect.width + subtreewidth))
          spousenode.setLeft(spousenode.getLeft() + offset);
      }
    }

    // return the effective width of this subtree
    origrect.width += subtreewidth;
  }

  public GenDB getDB()
  {
    return myDB;
  }
  
  private boolean isPositioned(PersonNode pn)
  {
    return (pn.getLeft() >= 10 || pn.getTop() >= 10);
  }


  // node spacing
  protected int myVertSeparation = 80;
  protected int myHorizSeparation = 4;

  // link parameterization
  protected JGoPen myMarriageLinkPen =
       JGoPen.make(JGoPen.SOLID, 2, new Color(0, 150, 0));
  protected JGoPen myDaughterLinkPen =
       JGoPen.make(JGoPen.SOLID, 1, new Color(200, 100, 100));
  protected JGoPen mySonLinkPen =
       JGoPen.make(JGoPen.SOLID, 1, new Color(100, 100, 200));

  // the "genealogical database"
  protected GenDB myDB = new GenDB();

  // map Persons to PersonNodes
  protected HashMap myMap = new HashMap();
}

