/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.family;

import java.io.Serializable;
import java.util.GregorianCalendar;
import java.util.Vector;
import java.util.TreeMap;
import java.util.Iterator;


/**
 * This represents the in-memory data base of family information.
 * It is composed of instances of Persons, searchable by both person
 * name and by identifier.  Because this is just an example program,
 * we have not put much effort into completing the database work,
 * including error checking or other work that is independent of
 * demonstrating JGo.
 * <p>
 * Both the GenDB and Person classes are completely independent of
 * any GUI (including JGo).
 * @see Person
 */
public class GenDB implements Serializable
{
  // instead of actually reading from persistent storage, this example program
  // just uses some hard-coded information
  public void load()
  {
    final int M = Person.MALE;
    final int F = Person.FEMALE;
    final int U = Person.UNKNOWN;

    makePerson(1,  "Henry VII", M, U, U, 1457, 1509)
      .alias = "King of England 1485-1509";
    makePerson(2,  "Elizabeth", F, U, U, 1465, 1503);
    makeMarriage(1, 2);

    makePerson(3,  "Arthur", M, 1, 2, 1486, 1502);
    makePerson(4,  "Catherine of Aragon", F, U, U, 1485, 1536);
    makeMarriage(3, 4);
    makePerson(5,  "Margaret", F, 1, 2, 1489, 1541);
    makePerson(6,  "James IV", M, U, U, 1473, 1513)
      .alias = "of Scotland";
    makeMarriage(5, 6);
    makePerson(7,  "Archibald Douglas", M, U, U, 1489, 1557)
      .alias = "Earl of Angus";
    makeMarriage(5, 7);
    makePerson(8,  "Henry VIII", M, 1, 2, 1491, 1547)
      .alias = "King of England 1509-1547";
    makeMarriage(8, 4);
    makePerson(9,  "Anne Boleyn", F, U, U, 1507, 1536);
    makeMarriage(8, 9);
    makePerson(10, "Jane Seymour", F, U, U, 1509, 1537);
    makeMarriage(8, 10);
    makePerson(11, "Anne of Cleves", F, U, U, 1515, 1557);
    makeMarriage(8, 11);
    makePerson(12, "Catherine Howard", F, U, U, -1, 1542);
    makeMarriage(8, 12);
    makePerson(13, "Catherine Parr", F, U, U, 1512, 1548);
    makeMarriage(8, 13);
    makePerson(14, "Elizabeth", F, 1, 2, 1492, 1495);
    makePerson(15, "Mary", F, 1, 2, 1496, 1533);
    makePerson(16, "Louis XII", M, U, U, 1462, 1515)
      .alias = "of France";
    makeMarriage(15, 16);
    makePerson(17, "Charles Brandon", M, U, U, -1, 1545)
      .alias = "Duke of Suffolk";
    makeMarriage(15, 17);
    makePerson(18, "Edmund", M, 1, 2, 1499, 1499);
    makePerson(19, "Catherine", F, 1, 2, 1503, 1503);

    makePerson(20, "James V", M, 5, 6, 1512, 1542)
      .alias = "of Scotland";
    makePerson(21, "Mary of Guise", F, U, U, 1515, 1560);
    makeMarriage(20, 21);
    makePerson(22, "5 died young", Person.UNSEXED, 5, 6, -1, -1);
    makePerson(23, "Margaret", F, 5, 7, 1515, 1578);
    makePerson(24, "Matthew Stewart", M, U, U, 1516, 1571)
      .alias = "Earl of Lennox";
    makeMarriage(23, 24);
    makePerson(25, "Mary I", F, 8, 4, 1516, 1558)
      .alias = "Queen of England 1553-1558";
    makePerson(26, "Philip II", M, U, U, 1527, 1598)
      .alias = "of Spain";
    makeMarriage(25, 26);
    makePerson(27, "3 died young", Person.UNSEXED, 8, 4, -1, -1);
    makePerson(28, "Elizabeth I", F, 8, 9, 1533, 1603)
      .alias = "Queen of England 1558-1603";
    makePerson(29, "Edward VI", M, 8, 10, 1537, 1553)
      .alias = "King of England 1547-1553";
    makePerson(30, "Henry", M, 15, 17, 1516, 1534)
      .alias = "Earl of Lincoln";
    makePerson(31, "Frances", F, 15, 17, 1517, 1559);
    makePerson(32, "Henry Grey", M, U, U, 1517, 1554)
      .alias = "Duke of Suffolk";
    makeMarriage(31, 32);
    makePerson(33, "Eleanor", F, 15, 17, 1519, 1547);
    makePerson(34, "Henry Clifford", M, U, U, 1517, 1570)
      .alias = "Earl of Cumberland";
    makeMarriage(33, 34);

    makePerson(35, "2 died young", Person.UNSEXED, 20, 21, -1, -1);
    makePerson(36, "Mary", F, 20, 21, 1542, 1587)
      .alias = "Queen of Scots 1542-1567";
    makePerson(37, "Francis II", M, U, U, 1544, 1560)
      .alias = "of France";
    makeMarriage(36, 37);
    makePerson(38, "Henry Stewart", M, 23, 24, 1545, 1567)
      .alias = "Lord Darnley";
    makeMarriage(36, 38);
    makePerson(39, "James Hepburn", M, U, U, 1536, 1578)
      .alias = "Earl of Bothwell";
    makeMarriage(36, 39);
    makePerson(40, "Charles", M, 23, 24, 1556, 1576)
      .alias = "Earl of Lennox";
    makePerson(41, "Elizabeth Cavendish", F, U, U, -1, 1582);
    makeMarriage(40, 41);
    makePerson(42, "6 died young", Person.UNSEXED, 23, 24, -1, -1);

    makePerson(50, "James I", M, 36, 38, 1566, 1625)
      .alias = "King of England 1603-1625, King of Scotland 1567-1625";
    makePerson(51, "Anne", F, U, U, 1574, 1619)
      .alias = "daughter of Frederick II of Denmark";
    makeMarriage(50, 51);
  }

  protected Person makePerson(int id, String n, int sex,
                              int father, int mother, int birth, int death)
  {
    Person p = getPerson(id);
    if (p != null) {
      p.name = n;
      p.sex = sex;
      p.father = father;
      p.mother = mother;
      if (birth > 1000)
        p.birthdate = new GregorianCalendar(birth, 0, 1);
      if (death > 1000)
        p.deathdate = new GregorianCalendar(death, 0, 1);

      Person f = getPerson(father);
      if (f != null)
        f.childrenlist.add(new Integer(id));

      Person m = getPerson(mother);
      if (m != null)
        m.childrenlist.add(new Integer(id));
    }
    return p;
  }

  protected void makeMarriage(int i1, int i2)
  {
    Person p1 = getPerson(i1);
    if (p1 != null) {
      p1.spouseslist.add(new Integer(i2));
      p1.weddingdateslist.add(null);
      p1.weddingplaceslist.add(null);
    }

    Person p2 = getPerson(i2);
    if (p2 != null) {
      p2.spouseslist.add(new Integer(i1));
      p2.weddingdateslist.add(null);
      p2.weddingplaceslist.add(null);
    }
  }


  // using getPerson allows for "forward" references to people that
  // don't exist yet in the database--it makes sure there's a "Person"
  // for the given ID
  protected Person getPerson(int id)
  {
    if (id == Person.UNKNOWN)
      return null;
    Person p = findPerson(id);
    if (p == null) {
      p = new Person();
      p.number = id;
      myIdMap.put(new Integer(id), p);
    }
    return p;
  }


  // these are the public methods (read-only)

  public Person findPerson(int id)
  {
    return (Person)myIdMap.get(new Integer(id));
  }

  public Iterator getIterator()
  {
    return myIdMap.values().iterator();
  }


  // State
  protected TreeMap myIdMap = new TreeMap();
}
