/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.family;

import java.awt.*;
import java.util.Calendar;
import com.nwoods.jgo.*;


/**
 * PersonNode inherits much of the functionality of the JGoTextNode example class.
 * <p>
 * The primary change is how the ports are used and where they are positioned.
 * The TopPort is used for the connection to the parents' marriage link.
 * The RightPort is used for marriage links.
 */
public class PersonNode extends JGoTextNode
{
  public PersonNode() {
    super();
    initCommon();
  }
  
  public PersonNode(String s) {
    super(s);
    initCommon();
  }
  
  private void initCommon()
  {
    // assume marriage links can both come into and go out of
    // the RightPort from any direction
    if (getRightPort() != null) {
      getRightPort().setFromSpot(NoSpot);
      getRightPort().setToSpot(NoSpot);
    }
  }
  
  public void layoutChildren(JGoObject childchanged)
  {
    if (isInitializing()) return;
    if (childchanged == null || childchanged == getLabel()) {
      super.layoutChildren(childchanged);
      if (getRightPort() != null) {
        getRightPort().setSpotLocation(JGoObject.Center, getBackground(), JGoObject.Center);
      }
    }
  }

  // make sure all of the parts of this node have the right appearance
  public void resetColors()
  {
    if (getPerson() == null) return;
    if (getPerson().isMale()) {
      getLabel().setTextColor(myMaleTextColor);
      setPen(myMaleRectPen);
      setBrush(myMaleRectBrush);
    } else if (getPerson().isFemale()) {
      getLabel().setTextColor(myFemaleTextColor);
      setPen(myFemaleRectPen);
      setBrush(myFemaleRectBrush);
    } else {
      getLabel().setTextColor(Color.black);
      setPen(JGoPen.black);
      setBrush(JGoBrush.white);
    }
  }

  // Present additional information not present in the node's text.
  public String getToolTipText()
  {
    Person p = getPerson();
    if (p != null) {
      String msg = "";
      if (p.birthdate == null)
        msg += "?";
      else
        msg += p.birthdate.get(Calendar.YEAR);
      msg += " - ";
      if (p.deathdate == null)
        msg += "?";
      else
        msg += p.deathdate.get(Calendar.YEAR);
      if (p.alias != null) {
        msg += ", ";
        msg += p.alias;
      }
      return msg;
    }
    return null;
  }

  // This is a convenience method for getting the JGoTextNode's Object as a Person.
  public Person getPerson() { return myPerson; }

  // Make sure the JGoTextNode's Object is always a Person, and initialize
  // the parts of this node appropriately for that Person.
  public void setPerson(Person x)
  {
    myPerson = x;
    if (x.name == null)
      setText("<unknown>");
    else
      setText(x.name);
    resetColors();
  }


  // These appearance parameters are shared by all PersonNodes

  public static Color myFemaleTextColor = Color.black;
  public static JGoPen myFemaleRectPen = JGoPen.Null;
  public static JGoBrush myFemaleRectBrush =
      JGoBrush.makeStockBrush(new Color(250, 200, 200));

  public static Color myMaleTextColor = Color.black;
  public static JGoPen myMaleRectPen = JGoPen.Null;
  public static JGoBrush myMaleRectBrush =
      JGoBrush.makeStockBrush(new Color(200, 200, 250));

  // PersonNode state
  private Person myPerson = null;
}

