/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.flower;

import com.nwoods.jgo.*;
import java.awt.*;
import java.net.URL;
import javax.swing.JOptionPane;
import com.nwoods.jgo.examples.*;


// Activities are implemented as SimpleNodes.
//
// Properties include Text, ActivityType, and ID.
// The Text property is actually just the SimpleNode's Label's Text.
//
// This class also supports a standard set of icons (as JGoImages)
// and the notion of a standard size for the node.
public class ActivityNode extends SimpleNode
{
  public ActivityNode()
  {
    super();
  }

  public JGoObject copyObject(JGoCopyEnvironment env)
  {
    ActivityNode newobj = (ActivityNode)super.copyObject(env);
    if (newobj != null) {
      // the ID should *not* be copied blindly--leave as -1 so
      // that we can fix it up later in ownerChange
      // the JGoObjects that are part of this area are copied
      // separately by copyChildren()
      newobj.myActivityType = myActivityType;
      newobj.myTaskDuration = myTaskDuration;
      // no other fields to copy--the Text is actually on a sub-object
    }
    return newobj;
  }

  public void initialize(int acttype, int id)
  {
    super.initialize(myStdPoint, getStdSize(), getImage(acttype), getLabelString(acttype), false, false);

    myID = id;
    myActivityType = acttype;

    resetEditability();

    // create an input port and an output port, each instances of SimpleNodePort
    if (acttype != Start) {
      myTaskDuration = 0;
      myInputPort = new SimpleNodePort(true,this);
    }
    if (acttype != Finish) {
      myTaskDuration = 0;
      myOutputPort = new SimpleNodePort(false,this);
    }
    if (acttype == Activity) {
      myTaskDuration = 5;
    }

    setSize(getStdSize());

    layoutChildren(null);
  }

  public ProcessDocument getDoc() { return (ProcessDocument)getDocument(); }

  // make sure that whenever an ActivityNode is added to a ProcessDocument,
  // it has a unique ID
  protected void ownerChange(JGoObjectCollection oldOwner, JGoObjectCollection newOwner, JGoObject mainObject)
  {
    if (oldOwner == null &&
        newOwner != null &&
        newOwner instanceof ProcessDocument) {
      if (getID() == -1) {
        ProcessDocument doc = (ProcessDocument)newOwner;
        myID = doc.getNextNodeID();
      }
    }
  }

  // Start and Finish nodes do not have editable labels
  public void resetEditability()
  {
    if (getLabel() != null)
      getLabel().setEditable(getActivityType() != Start &&
                             getActivityType() != Finish);
  }

  // You'll probably want to replace this with somewhat more interesting information
  public String getToolTipText()
  {
    String msg = Integer.toString(getTaskDuration());
    return msg;
  }

  // Events

  public boolean doMouseDblClick(int modifiers, Point dc, Point vc, JGoView view)
  {
    ProcessView processView = (ProcessView)view;
    processView.editActivity(this);
    return true;
  }

  // Properties

  public int getID() { return myID; }

  public int getActivityType() { return myActivityType; }

  public String getText()
  {
    if (getLabel() != null)
      return getLabel().getText();
    else
      return "";
  }

  public void setText(String s)
  {
    if (getLabel() != null)
      getLabel().setText(s);
  }

  public int getTaskDuration()
  {
    return myTaskDuration;
  }

  public void setTaskDuration(int taskDuration)
  {
    myTaskDuration = taskDuration;
    updateDownstreamLinks();
  }

  protected void updateDownstreamLinks()
  {
    // Compute the elapsed duration coming into this node by finding the max
    // value of any incoming link.
    // Then add myTaskDuration and set that value on all outgoing links.
    // Finally call updateDownsreamLinks() on all outgoing nodes to force
    // recursive updates of all downstream links.
    int nMax = 0;
    JGoPort inputPort = getInputPort();
    if (inputPort != null) {
      for (JGoListPosition pos = inputPort.getFirstLinkPos(); pos != null; pos = inputPort.getNextLinkPos(pos)) {
        FlowLink inputLink = (FlowLink)inputPort.getLinkAtPos(pos);
        int nLinkVal = Integer.parseInt(inputLink.getText());
        if (nLinkVal > nMax) {
          nMax = nLinkVal;
        }
      }
    }
    Integer intElapsedDurationIncludingThisNode = new Integer(nMax + getTaskDuration());
    String sElapsedDurationIncludingThisNode = intElapsedDurationIncludingThisNode.toString();
    JGoPort outputPort = getOutputPort();
    if (outputPort != null) {
      // Set the label on all outgoing links
      for (JGoListPosition pos = outputPort.getFirstLinkPos(); pos != null; pos = outputPort.getNextLinkPos(pos)) {
        FlowLink outputLink = (FlowLink)outputPort.getLinkAtPos(pos);
        outputLink.setText(sElapsedDurationIncludingThisNode);
      }
      // recurse
      for (JGoListPosition pos = outputPort.getFirstLinkPos(); pos != null; pos = outputPort.getNextLinkPos(pos)) {
        FlowLink outputLink = (FlowLink)outputPort.getLinkAtPos(pos);
        ActivityNode toNode = (ActivityNode)outputLink.getToNode();
        toNode.updateDownstreamLinks();
      }
    }
   }

  public boolean downstreamNodeContainsFlag(int nFlag)
  {
    // recurse looking for a node containing nFlag
    if ((getFlags() & nFlag) != 0)
      return true;
    JGoPort outputPort = getOutputPort();
    if (outputPort == null)
      return false;
    for (JGoListPosition pos = outputPort.getFirstLinkPos(); pos != null;
         pos = outputPort.getNextLinkPos(pos)) {
      FlowLink outputLink = (FlowLink) outputPort.getLinkAtPos(pos);
      ActivityNode toNode = (ActivityNode) outputLink.getToNode();
      if (toNode.downstreamNodeContainsFlag(nFlag))
        return true;
    }
    return false;
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
    // Add <FlowerActivityNode> element
    if (svgDoc.JGoXMLOutputEnabled()) {
      DomElement activityNode = svgDoc.createJGoClassElement(
          "com.nwoods.jgo.examples.flower.ActivityNode", jGoElementGroup);
      activityNode.setAttribute("actid", Integer.toString(myID));
      activityNode.setAttribute("acttype", Integer.toString(myActivityType));
      activityNode.setAttribute("actduration", Integer.toString(myTaskDuration));
    }

    // Have superclass add to the JGoObject group
    super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a <FlowerActivityNode> element
      myID = Integer.parseInt(jGoChildElement.getAttribute("actid"));
      myActivityType = Integer.parseInt(jGoChildElement.getAttribute("acttype"));
      myTaskDuration = Integer.parseInt(jGoChildElement.getAttribute("actduration"));
      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }


  // Static methods

  public static JGoImage getImage(int acttype)
  {
    JGoImage image = null;
    switch (acttype) {
      default:
      case Activity:
        image = new JGoImage(myStdPoint, getStdSize());
        image.loadImage(ActivityNode.class.getResource("images/doc.gif"), true);
        break;
      case Start:
        image = new JGoImage(myStdPoint, getStdSize());
        image.loadImage(ActivityNode.class.getResource("images/star.gif"), true);
        break;
      case Finish:
        image = new JGoImage(myStdPoint, getStdSize());
        image.loadImage(ActivityNode.class.getResource("images/star.gif"), true);
        break;
    }
    return image;
  }

  public static String getLabelString(int acttype)
  {
    switch (acttype) {
      default:
      case Activity: return "Activity";
      case Start:    return "Start";
      case Finish:   return "Finish";
    }
  }


  public static Dimension getStdSize()
  {
    return myStdSize;
  }

  public static void setStdSize(int w, int h)
  {
    myStdSize.width = w;
    myStdSize.height = h;
  }


  // Icon/Activity types
  public static final int Start = 1;
  public static final int Finish = 2;
  public static final int Activity = 3;

  private static Dimension myStdSize = new Dimension(50, 50);
  private static Point myStdPoint = new Point();  // don't care


  // State

  private int myID = -1;
  private int myTaskDuration = 0;
  private int myActivityType = 0;
}
