/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.flower;

import java.awt.*;
import java.awt.event.*;
import java.awt.print.PageFormat;
import java.awt.print.PrinterJob;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import java.util.HashMap;
import java.net.URL;
import com.nwoods.jgo.*;
import com.nwoods.jgo.layout.*;
import com.nwoods.jgo.examples.*;


/**
 * This example app is a simple workflow editor
 */
public class Flower extends JApplet implements Runnable
{
  public Flower()
  {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

      mainMenuBar = new JMenuBar();
      toolBar = new JToolBar();
      filemenu = new JMenu();
      editmenu = new JMenu();
      viewmenu = new JMenu();
      insertmenu = new JMenu();
      layoutmenu = new JMenu();
      helpmenu = new JMenu();

      initMenus();
      initToolbar();

      myDesktop = new JDesktopPane();
      myPalette = new JGoPalette();
      myPalette.setPreferredSize(new Dimension(100, 300));
      myPalette.setMinimumSize(new Dimension(100, 100));

      JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
      splitPane.setContinuousLayout(true);
      splitPane.setLeftComponent(getPalette());
      splitPane.setRightComponent(getDesktop());
      splitPane.setDividerLocation(100);

      Container contentPane = getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(toolBar, BorderLayout.NORTH);
      contentPane.add(splitPane, BorderLayout.CENTER);
      contentPane.validate();
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
    }
  }


  //==============================================================
  // Define all the command actions
  //==============================================================

    static private boolean imagesErrorDisplayed = false;
    static private Icon iconImage(String resourceName)
    {
      URL url = Flower.class.getResource(resourceName);
      if (url == null) {
        if (!imagesErrorDisplayed) {
          JOptionPane.showMessageDialog(null, "Image files not found.\nThe Flower.images directory must be placed in the class path.", "alert", JOptionPane.ERROR_MESSAGE);
          imagesErrorDisplayed = true;
        }
      }
      else
        return new ImageIcon(url);
      return null;
    }

    AppAction FileNewAction = null;
    {
      Icon icon = iconImage("images/new.gif");
      FileNewAction = new AppAction("New", icon, this) {
        public void actionPerformed(ActionEvent e) { newProcess(); }
        public boolean canAct() { return true; } };  // doesn't depend on a view
    }

    AppAction FileOpenAction = null;
    {
      Icon icon = iconImage("images/open.gif");
      FileOpenAction = new AppAction("Open", icon, this) {
        public void actionPerformed(ActionEvent e) { openProcess(); }
        public boolean canAct() { return true; } };  // doesn't depend on a view
    }

    AppAction FileCloseAction = new AppAction("Close", this) {
      public void actionPerformed(ActionEvent e) { closeProcess(); } };

    AppAction FileSaveAction = null;
    {
      Icon icon = iconImage("images/save.gif");
      FileSaveAction = new AppAction("Save", icon, this) {
        public void actionPerformed(ActionEvent e) { saveProcess(); }
        public boolean canAct() { return super.canAct() && getView().getDoc().isLocationModifiable(); } };  // doesn't depend on a view
    }

    AppAction FileSaveAsAction = new AppAction("Save As", this) {
      public void actionPerformed(ActionEvent e) { saveAsProcess(); } };

    AppAction FilePropertiesAction = new AppAction("Process Properties", this) {
      public void actionPerformed(ActionEvent e) { editProcessProperties(); } };

    AppAction PrintPreviewAction = null;
    {
      PrintPreviewAction = new AppAction("Print Preview", null, this) {
        public void actionPerformed(ActionEvent e) {
          PrinterJob pj = PrinterJob.getPrinterJob();
          PageFormat defaultpf = pj.validatePage(pj.defaultPage());
          // Ask the user what page size they're using, what the margins are,
          // and whether they are printing in landscape or portrait mode.
          PageFormat pf = pj.pageDialog(defaultpf);
          if (pf != defaultpf)
            getView().printPreview("Print Preview", pf);
        }
      };
    }

    AppAction PrintAction = null;
    {
      Icon icon = iconImage("images/print.gif");
      PrintAction = new AppAction("Print", icon, this) {
        public void actionPerformed(ActionEvent e) { getView().print(); } };
    }

    AppAction CutAction = null;
    {
      Icon icon = iconImage("images/remove.gif");
      CutAction = new AppAction("Cut", icon, this) {
        public void actionPerformed(ActionEvent e) { getView().cut(); }
        public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty() && getView().getDoc().isModifiable(); } };  // doesn't depend on a view
    }

    AppAction CopyAction = null;
    {
      Icon icon = iconImage("images/copy.gif");
      CopyAction = new AppAction("Copy", icon, this) {
        public void actionPerformed(ActionEvent e) { getView().copy(); }
        public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };  // doesn't depend on a view
    }

    AppAction PasteAction = null;
    {
      Icon icon = iconImage("images/paste.gif");
      PasteAction = new AppAction("Paste", icon, this) {
        public void actionPerformed(ActionEvent e) { getView().paste(); }
        public boolean canAct() { return super.canAct() && getView().getDoc().isModifiable(); } };  // doesn't depend on a view
    }

    AppAction DeleteAction = new AppAction("Delete", this) {
      public void actionPerformed(ActionEvent e) { getView().deleteSelection(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty() && getView().getDoc().isModifiable(); } };

    AppAction SelectAllAction = new AppAction("Select All", this) {
      public void actionPerformed(ActionEvent e) { getView().selectAll(); } };

  JMenuItem UndoMenuItem = null;

    AppAction UndoAction = new AppAction("Undo", this) {
      public void actionPerformed(ActionEvent e) { getView().getDocument().undo(); AppAction.updateAllActions(); }
      public boolean canAct() { return super.canAct() && (getView().getDocument().canUndo()); }
      public void updateEnabled()
      {
        super.updateEnabled();
        if (UndoMenuItem != null && getView() != null)
          UndoMenuItem.setText(getView().getDocument().getUndoManager().getUndoPresentationName());
      } };

  JMenuItem RedoMenuItem = null;

    AppAction RedoAction = new AppAction("Redo", this) {
      public void actionPerformed(ActionEvent e) { getView().getDocument().redo(); AppAction.updateAllActions(); }
      public boolean canAct() { return super.canAct() && (getView().getDocument().canRedo()); }
      public void updateEnabled()
      {
        super.updateEnabled();
        if (RedoMenuItem != null && getView() != null)
          RedoMenuItem.setText(getView().getDocument().getUndoManager().getRedoPresentationName());
      } };

    AppAction ObjectPropertiesAction = new AppAction("Object Properties", this) {
      public void actionPerformed(ActionEvent e) { getView().editObjectProperties(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    AppAction ZoomNormalAction = new AppAction("Normal Zoom", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomNormal(); } };

    AppAction ZoomInAction = new AppAction("Zoom In", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomIn(); }
      public boolean canAct() { return super.canAct() && (getView().getScale() < 8.0f); } };

    AppAction ZoomOutAction = new AppAction("Zoom Out", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomOut(); }
      public boolean canAct() { return super.canAct() && (getView().getScale() > 0.13f); } };

    AppAction ZoomToFitAction = new AppAction("Zoom To Fit", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomToFit(); } };

    AppAction OverviewAction = new AppAction("Overview", this) {
      public void actionPerformed(ActionEvent e) { overviewAction(); } };

    AppAction GridAction = new AppAction("Toggle Grid", this) {
      public void actionPerformed(ActionEvent e) { getView().showGrid(); } };

    AppAction OrthoAction = new AppAction("Toggle Orthogonal Flows", this) {
      public void actionPerformed(ActionEvent e) { getView().getDoc().toggleOrthogonalFlows(); }
      public boolean canAct() { return super.canAct() && getView().getDoc().isModifiable(); } };

    AppAction InsertInputAction = new AppAction("Start", this) {
      public void actionPerformed(ActionEvent e) { getView().insertInput(); }
      public boolean canAct() { return super.canAct() && getView().getDoc().isModifiable(); } };

    AppAction InsertOutputAction = new AppAction("Finish", this) {
      public void actionPerformed(ActionEvent e) { getView().insertOutput(); }
      public boolean canAct() { return super.canAct() && getView().getDoc().isModifiable(); } };

    AppAction InsertActivityAction = new AppAction("Activity", this) {
      public void actionPerformed(ActionEvent e) { getView().insertActivity(); }
      public boolean canAct() { return super.canAct() && getView().getDoc().isModifiable(); } };

    AppAction RandomLayoutAction = new AppAction("Random Layout", this) {
      public void actionPerformed(ActionEvent e) { randomAction(); } };

    AppAction LayerLayoutAction = null;
    {
      Icon icon = iconImage("images/layout.gif");
      LayerLayoutAction = new AppAction("Layered Digraph Layout", icon, this) {
        public void actionPerformed(ActionEvent e) { layerAction(); } };
    }

    AppAction AboutAction = null;
    {
      Icon icon = iconImage("images/tb_07_u.gif");
      AboutAction = new AppAction("About", icon, this) {
        public void actionPerformed(ActionEvent e) { showAbout(); }
        public boolean canAct() { return true; } };  // doesn't depend on a view
    }

    AppAction PropertiesAction = new AppAction("Properties", this) {
      public void actionPerformed(ActionEvent e) { getView().editObjectProperties(); } };


  void initMenus()
  {
    //==============================================================
    // Define all the command actions and setup the menus
    //==============================================================
    JMenuItem item = null;

    filemenu.setText("File");
    filemenu.setMnemonic('F');

    item = filemenu.add(FileNewAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N,Event.CTRL_MASK));
    item.setMnemonic('N');
    item.setIcon(null);  //choose not to use icon in menu

    item = filemenu.add(FileOpenAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O,Event.CTRL_MASK));
    item.setMnemonic('O');
    item.setIcon(null);  //choose not to use icon in menu

    item = filemenu.add(FileCloseAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4,Event.CTRL_MASK));
    item.setMnemonic('C');

    item = filemenu.add(FileSaveAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,Event.CTRL_MASK));
    item.setMnemonic('S');
    item.setIcon(null);  //choose not to use icon in menu

    item = filemenu.add(FileSaveAsAction);
    item.setMnemonic('A');

    filemenu.addSeparator();

//    item = filemenu.add(FilePropertiesAction);
//    item.setMnemonic('r');

    filemenu.addSeparator();

    item = filemenu.add(PrintPreviewAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R,Event.CTRL_MASK));
    item.setMnemonic('R');
    item.setIcon(null);  //choose not to use icon in menu

    item = filemenu.add(PrintAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,Event.CTRL_MASK));
    item.setMnemonic('P');
    item.setIcon(null);  //choose not to use icon in menu

    mainMenuBar.add(filemenu);


    editmenu.setText("Edit");
    editmenu.setMnemonic('E');

    item = editmenu.add(CutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,Event.CTRL_MASK));
    item.setMnemonic('t');
    item.setIcon(null);  //choose not to use icon in menu

    item = editmenu.add(CopyAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,Event.CTRL_MASK));
    item.setMnemonic('C');
    item.setIcon(null);  //choose not to use icon in menu

    item = editmenu.add(PasteAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V,Event.CTRL_MASK));
    item.setMnemonic('P');
    item.setIcon(null);  //choose not to use icon in menu

    item = editmenu.add(DeleteAction);
    item.setMnemonic('D');

    item = editmenu.add(SelectAllAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A,Event.CTRL_MASK));
    item.setMnemonic('l');

    editmenu.addSeparator();

    UndoMenuItem = editmenu.add(UndoAction);
    UndoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z,Event.CTRL_MASK));
    UndoMenuItem.setMnemonic('U');

    RedoMenuItem = editmenu.add(RedoAction);
    RedoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y,Event.CTRL_MASK));
    RedoMenuItem.setMnemonic('R');

    editmenu.addSeparator();

    item = editmenu.add(ObjectPropertiesAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,Event.CTRL_MASK));
    item.setMnemonic('o');

    mainMenuBar.add(editmenu);


    viewmenu.setText("View");
    viewmenu.setMnemonic('V');

    item = viewmenu.add(ZoomNormalAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6,Event.CTRL_MASK | Event.SHIFT_MASK));
    item.setMnemonic('N');

    item = viewmenu.add(ZoomInAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6,Event.CTRL_MASK));
    item.setMnemonic('I');

    item = viewmenu.add(ZoomOutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6,Event.SHIFT_MASK));
    item.setMnemonic('O');

    item = viewmenu.add(ZoomToFitAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0));
    item.setMnemonic('Z');

    viewmenu.addSeparator();

    viewmenu.add(OverviewAction).setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W,Event.CTRL_MASK));

    viewmenu.addSeparator();

    item = viewmenu.add(GridAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G,Event.CTRL_MASK));
    item.setMnemonic('G');

    item = viewmenu.add(OrthoAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T,Event.CTRL_MASK));
    item.setMnemonic('O');

    mainMenuBar.add(viewmenu);


    insertmenu.setText("Insert");
    insertmenu.setMnemonic('I');

    item = insertmenu.add(InsertInputAction);
    item.setMnemonic('S');

    item = insertmenu.add(InsertOutputAction);
    item.setMnemonic('F');

    item = insertmenu.add(InsertActivityAction);
    item.setMnemonic('A');

    mainMenuBar.add(insertmenu);


    layoutmenu.setText("Layout");
    layoutmenu.setMnemonic('L');

    item = layoutmenu.add(RandomLayoutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R,Event.CTRL_MASK));
    item.setMnemonic('R');

    item = layoutmenu.add(LayerLayoutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L,Event.CTRL_MASK));
    item.setMnemonic('L');
    item.setIcon(null);  //choose not to use icon in menu

    mainMenuBar.add(layoutmenu);


    helpmenu.setText("Help");
    helpmenu.setMnemonic('H');

    item = helpmenu.add(AboutAction);
    item.setMnemonic('A');
    item.setIcon(null);  //choose not to use icon in menu

    mainMenuBar.add(helpmenu);

    setJMenuBar(mainMenuBar);


    // initialize popup menus
    JPopupMenu popup = ProcessView.getPopupMenu();
    popup.add(PropertiesAction);
    popup.addSeparator();
    popup.add(CopyAction);
    popup.add(CutAction);
  }

  protected JToolBar initToolbar()
  {
    JButton button = null;
    button = toolBar.add(FileNewAction);
    button.setToolTipText("Create a new work-flow window");
    button = toolBar.add(FileOpenAction);
    button.setToolTipText("Open a previously saved work-flow window");
    button = toolBar.add(FileSaveAction);
    button.setToolTipText("Save a work-flow window");
    toolBar.addSeparator();
    button = toolBar.add(CutAction);
    button.setToolTipText("Cut to clipboard");
    button = toolBar.add(CopyAction);
    button.setToolTipText("Copy to clipboard");
    button = toolBar.add(PasteAction);
    button.setToolTipText("Paste to clipboard");
    toolBar.addSeparator();
    button = toolBar.add(PrintAction);
    button.setToolTipText("Print selected work-flow window");
    toolBar.addSeparator();
    button = toolBar.add(LayerLayoutAction);
    button.setToolTipText("Automatically layout diagram");
    toolBar.addSeparator();
    button = toolBar.add(AboutAction);
    button.setToolTipText("Display help about this application");
    return toolBar;
  }

  public void addExitCommand()
  {
    filemenu.addSeparator();

    AppAction ExitAction = new AppAction("Exit", this) {
      public void actionPerformed(ActionEvent e) { exit(); }
      public boolean canAct() { return true; } };  // doesn't depend on a view

    JMenuItem item = filemenu.add(ExitAction);
    item.setMnemonic('x');
  }

  public void init()
  {
    JGoImage.setDefaultBase(getCodeBase());
  }

  void initPalette()
  {
    getPalette().setBorder(new TitledBorder("Activities"));

    ActivityNode snode;
    JGoDocument doc = getPalette().getDocument();

    // make sure each label is not editable
    snode = new ActivityNode();
    snode.initialize(ActivityNode.Start, -1);
    snode.getLabel().setEditable(false);
    doc.addObjectAtTail(snode);

    snode = new ActivityNode();
    snode.initialize(ActivityNode.Finish, -1);
    snode.getLabel().setEditable(false);
    doc.addObjectAtTail(snode);

    snode = new ActivityNode();
    snode.initialize(ActivityNode.Activity, -1);
    snode.getLabel().setEditable(false);
    doc.addObjectAtTail(snode);
  }

  public void start()
  {
    // enable drag-and-drop from separate thread
    new Thread(this).start();

    initPalette();
  }

  public void run() {
    getPalette().initializeDragDropHandling();

    if (getDesktop().getAllFrames().length == 0) {
      newProcess();
    }

    AppAction.updateAllActions();
  }

  public void destroy() {
    JInternalFrame[] frames = getDesktop().getAllFrames();
    for (int i = 0; i < frames.length; i++) {
      JInternalFrame f = frames[i];
      try {
        f.setClosed(true);
      } catch (Exception x) {
      }
    }
  }

  static public void main(String args[])
  {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

      final Flower app = new Flower();
      mainFrame = new JFrame();

      // close the application when the main window closes
      mainFrame.addWindowListener(new WindowAdapter() {
        public void windowActivated(WindowEvent evt) {
          if (app.getCurrentView() != null) {
            app.getCurrentView().getDoc().updateLocationModifiable();
          }
        }
        public void windowClosing(WindowEvent event) {
          Object object = event.getSource();
          if (object == mainFrame)
            app.exit();
        }
      });

      app.addExitCommand();

      mainFrame.setTitle("Workflow Editor");
      Dimension screensize = Toolkit.getDefaultToolkit().getScreenSize();
      mainFrame.setBounds(0, 0, screensize.width, screensize.height);

      Container contentPane = mainFrame.getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(app, BorderLayout.CENTER);
      contentPane.validate();

      mainFrame.setVisible(true);

      app.start();
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }
  }


  void exit()
  {
    destroy();
    System.exit(0);
  }

  void showAbout()
  {
    HelpDlg helpDlg = new HelpDlg(mainFrame, "About", true);
    helpDlg.setVisible(true);
  }


  void editProcessProperties()
  {
    ProcessView v = getCurrentView();
    if (v != null) {
      v.getDoc().startTransaction();
      new ProcessDialog(v.getFrame(), v.getDoc()).setVisible(true);
      v.getDoc().endTransaction("Process Properties");
    }
  }


  public void createFrame(ProcessDocument doc)
  {
    final ProcessView view = new ProcessView(doc);
    final JInternalFrame frame = new JInternalFrame(doc.getName(), true, true, true);
    frame.setDefaultCloseOperation(JInternalFrame.DISPOSE_ON_CLOSE);
    view.initialize(this, frame);

    // keep track of the "current" view, even if it doesn't have focus
    // try to give focus to a view when it becomes activated
    // enable/disable all the command actions appropriately for the view
    frame.addInternalFrameListener(new InternalFrameListener()
      {
        public void internalFrameActivated(InternalFrameEvent e)
          {
            myCurrentView = view;
            view.requestFocus();
            if (myOverview != null)
              myOverview.setObserved(view);
            view.getDoc().updateLocationModifiable();
            AppAction.updateAllActions();
          }
        public void internalFrameDeactivated(InternalFrameEvent e) {}
        public void internalFrameOpened(InternalFrameEvent e) {}
        public void internalFrameClosing(InternalFrameEvent e) {}
        public void internalFrameClosed(InternalFrameEvent e)
          {
            myCurrentView = null;
            AppAction.updateAllActions();
          }
        public void internalFrameIconified(InternalFrameEvent e) {}
        public void internalFrameDeiconified(InternalFrameEvent e) {}
      }
    );

    frame.addVetoableChangeListener(new java.beans.VetoableChangeListener()
      {
        public void vetoableChange(java.beans.PropertyChangeEvent evt)
               throws java.beans.PropertyVetoException {
          if (evt.getPropertyName().equals(JInternalFrame.IS_CLOSED_PROPERTY) &&
              evt.getOldValue() == Boolean.FALSE &&
              evt.getNewValue() == Boolean.TRUE) {
            if (view.getDoc().isModified()) {
              String msg = "Save changes to ";
              if (view.getDoc().getName().equals(""))
                msg += "modified document?";
              else
                msg += view.getDoc().getName();
              msg += "\n  (";
              if (view.getDoc().getLocation().equals(""))
                msg += "<no location>";
              else
                msg += view.getDoc().getLocation();
              msg += ")";
              int answer = JOptionPane.showConfirmDialog(view.getFrame(),
                                msg, "Closing modified document",
                                JOptionPane.YES_NO_CANCEL_OPTION,
                                JOptionPane.QUESTION_MESSAGE);
              if (answer == JOptionPane.NO_OPTION ||
                  answer == JOptionPane.YES_OPTION) {
                if (answer == JOptionPane.YES_OPTION) {
                  if (view.getDoc().isLocationModifiable())
                    view.getDoc().save();
                  else
                    view.getDoc().saveAs(".wfl");
                }
                // then allow the internal frame to close
                getDesktop().remove(frame);
                getDesktop().repaint();
              } else {
                // CANCEL_OPTION--don't close
                throw new java.beans.PropertyVetoException("", evt);
              }
            }
          }
        }
      }
    );

    Container contentPane = frame.getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(view);

    frame.setSize(500, 400);
    getDesktop().add(frame);
    frame.show();
    view.initializeDragDropHandling();
  }

  void newProcess()
  {
    ProcessDocument doc = new ProcessDocument();
    String t = "Untitled" + Integer.toString(myDocCount++);
    doc.setName(t);
    createFrame(doc);
    doc.setModified(false);
    doc.discardAllEdits();
  }

  void openProcess()
  {
    String defaultLoc = null;
    ProcessView view = getCurrentView();
    if (view != null) {
      ProcessDocument doc = view.getDoc();
      defaultLoc = doc.getLocation();
    }
    ProcessDocument doc = ProcessDocument.open(this, defaultLoc);
    if (doc != null)
      createFrame(doc);
  }

  void closeProcess()
  {
    if (getCurrentView() != null) {
      JInternalFrame frame = getCurrentView().getInternalFrame();
      if (frame != null) {
        try {
          frame.setClosed(true);
        } catch (Exception x) {
        }
      }
    }
  }

  void saveProcess()
  {
    if (getCurrentView() != null) {
      ProcessDocument doc = getCurrentView().getDoc();
      doc.save();
    }
  }

  void saveAsProcess()
  {
    if (getCurrentView() != null) {
      ProcessDocument doc = getCurrentView().getDoc();
      doc.saveAs(".wfl");
    }
  }

  // make the overview window visible
  void overviewAction()
  {
    if (myOverview == null) {
      myOverview = new JGoOverview();
      myOverview.setObserved(getCurrentView());
      myOverviewDialog = new JDialog(mainFrame, "Overview", false);
      myOverviewDialog.getContentPane().setLayout(new BorderLayout());
      myOverviewDialog.getContentPane().add(myOverview, BorderLayout.CENTER);
    }
    myOverviewDialog.pack();
    myOverviewDialog.setVisible(true);
  }

  void randomAction()
  {
    ProcessDocument doc = getCurrentView().getDoc();
    doc.startTransaction();
    Rectangle r = getCurrentView().getViewRect();
    JGoRandomAutoLayout rand = new JGoRandomAutoLayout(doc, (r.x + 100), ((r.width - r.x) - 100), (r.y + 100), ((r.height - r.y) - 100));
    rand.performLayout();
    doc.endTransaction("Random Layout");
  }

  void layerAction()
  {
    ProcessDocument doc = getCurrentView().getDoc();
    doc.startTransaction();
    JGoLayeredDigraphAutoLayout l = new JGoLayeredDigraphAutoLayout(doc,
        30, 30,  JGoLayeredDigraphAutoLayout.LD_DIRECTION_RIGHT,
        JGoLayeredDigraphAutoLayout.LD_CYCLEREMOVE_DFS,
        JGoLayeredDigraphAutoLayout.LD_LAYERING_OPTIMALLINKLENGTH,
        JGoLayeredDigraphAutoLayout.LD_INITIALIZE_DFSOUT, 4,
        JGoLayeredDigraphAutoLayout.LD_AGGRESSIVE_FALSE);
    l.performLayout();
    doc.endTransaction("LayeredDigraph Layout");
  }


  ProcessDocument findProcessDocument(String path)
  {
    Object val = myMap.get(path);
    if (val != null && val instanceof ProcessDocument)
      return (ProcessDocument)val;
    else
      return null;
  }


  ProcessView getCurrentView()
  {
    return myCurrentView;
  }


  JDesktopPane getDesktop() { return myDesktop; }
  JGoPalette getPalette() { return myPalette; }


  // State
  static JFrame mainFrame;
  protected HashMap myMap = new HashMap();
  protected ProcessView myCurrentView;
  protected JDesktopPane myDesktop;
  protected JGoPalette myPalette;

  protected JMenuBar mainMenuBar;
  protected JToolBar toolBar;
  protected JMenu filemenu;
  protected JMenu editmenu;
  protected JMenu viewmenu;
  protected JMenu insertmenu;
  protected JMenu layoutmenu;
  protected JMenu helpmenu;

  protected JDialog myOverviewDialog;
  protected JGoOverview myOverview;

  private int myDocCount = 1;
}
