/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.flower;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.examples.*;
import com.nwoods.jgo.examples.flower.*;

// There is no "standard" Java API for implementing support for reading and
// writing XML.  Several alternatives are mentioned at
// http://java.sun.com/xml/
// including the "archiver" package for JavaBeans persistence
// JAXP for SAX and DOM parsers
// XML Data Binding for compiling XML Schemas into Java classes
//
// There are other sources for similar technology, such as JDOM.

// This class provides an example of using Sun's JAXP to implement custom XML
// support for your application.

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import org.xml.sax.*;
import org.w3c.dom.*;

// This class also provides an example of using the com.nwoods.jgo.svg package
// to save and restore JGoDocuments in the Scalable Vector Graphics XML
// document type (SVG).

// This class also provides an example of using the com.nwoods.jgo.svg package
// to save JGoDocuments in Scalable Vector Graphics format format (SVG) with or
// without JGo XML extensions that allow the exported document to be imported.

import com.nwoods.jgo.svg.DefaultDocument;

// Finally, this class also provides an example of using the org.apache.batik
// package to export a very precise and stand-alone representation of a
// JGoDocument in the Scalable Vector Graphics document type (SVG).
// While this SVG generation technique produces more accurate results when
// viewed with an SVG viewer, it is write-only.  Use the jgo.svg package
// described above for read/write SVG serialization or if you need to add
// your own extensions to the generated SVG.


// Implement the Process model data structure container.
// This class is responsible for loading and storing documents to files,
// and for creating activities (nodes) and flows (links).
//
// ProcessDocument, for this example app, has just a few properties:
// Name, Location, Orthogonal Flows and Link Pen.
// The latter two may appear to belong to a view instead of being
// part of a document, but here these attributes can be conveniently
// stored persistently.
public class ProcessDocument extends JGoDocument
{
  public ProcessDocument()
  {
    setUndoManager(new JGoUndoManager());
  }

  // Basic properties: name and location (pathname)

  public String getName()
  {
    return myName;
  }

  public void setName(String newname)
  {
    String oldName = myName;
    if (!oldName.equals(newname)) {
      myName = newname;
      fireUpdate(NAME_CHANGED, 0, null, 0, oldName);
    }
  }


  public String getLocation()
  {
    return myLocation;
  }

  public void setLocation(String newloc)
  {
    String oldLocation = myLocation;
    if (!oldLocation.equals(newloc)) {
      myLocation = newloc;
      fireUpdate(LOCATION_CHANGED, 0, null, 0, oldLocation);

      updateLocationModifiable();
    }
  }


  // read-only property--can the file be written?
  public boolean isLocationModifiable()
  {
    return myIsLocationModifiable;  // just return cached value
  }

  // There's no setLocationModifiable, because that's controlled externally
  // in the file system.  But because we're caching the writableness,
  // we need a method to update the cache.

  public void updateLocationModifiable()
  {
    boolean canwrite = true;
    if (!getLocation().equals("")) {
      File file = new File(getLocation());
      if (file.exists() && !file.canWrite())
        canwrite = false;
    }
    if (isLocationModifiable() != canwrite) {
      boolean oldIsModifiable = isModifiable();
      myIsLocationModifiable = canwrite;
      if (oldIsModifiable != isModifiable())
        fireUpdate(JGoDocumentEvent.MODIFIABLE_CHANGED, 0, null, (oldIsModifiable ? 1 : 0), null);
    }
  }

  // override to include whether the file can be written
  public boolean isModifiable()
  {
    return super.isModifiable() && isLocationModifiable();
  }


  public void updatePaperColor()
  {
    if (isModifiable())
      setPaperColor(Color.white);
    else
      setPaperColor(new Color(0xDD, 0xDD, 0xDD));
  }


  // new property--has the document been changed?
  public boolean isModified()
  {
    return myIsModified;
  }

  public void setModified(boolean b)
  {
    if (myIsModified != b) {
      myIsModified = b;
      // don't need to notify document listeners
    }
  }


  // Some, but not all, changes to the document should make it "modified"
  public void fireUpdate(int hint, int flags, Object object, int prevInt, Object prevVal)
  {
    // changing the read-only-ness isn't considered modifying the document
    if (hint == JGoDocumentEvent.MODIFIABLE_CHANGED) {
      updatePaperColor();
    } else if (hint != JGoDocumentEvent.PAPER_COLOR_CHANGED) {
      // don't consider the paper color as part of the document, either
      setModified(true);
    }
    super.fireUpdate(hint, flags, object, prevInt, prevVal);
  }


  // creating a new activity
  public ActivityNode newNode(int acttype)
  {
    ActivityNode snode = new ActivityNode();
    snode.initialize(acttype, getNextNodeID());
    addObjectAtTail(snode);
    return snode;
  }

  public int getNextNodeID()
  {
    return ++myLastNodeID;
  }

  public ActivityNode findNodeByID(int id)
  {
    // for larger documents, it would be more efficient to keep a
    // hash table mapping id to ActivityNode
    // for this example, we won't bother with the hash table
    JGoListPosition pos = getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = getObjectAtPos(pos);
      // only consider top-level objects
      pos = getNextObjectPosAtTop(pos);

      if (obj instanceof ActivityNode) {
        ActivityNode node = (ActivityNode)obj;
        if (node.getID() == id) {
          return node;
        }
      }
    }
    return null;
  }


  // creating a new flow between activities
  public FlowLink newLink(JGoPort from, JGoPort to)
  {
    FlowLink ll = new FlowLink(from, to);
    ll.initialize(" ");
    ll.setPen(getLinkPen());
    ll.setOrthogonal(isOrthogonalFlows());
    addObjectAtTail(ll);

    ActivityNode fromNode = (ActivityNode)from.getParent();
    fromNode.updateDownstreamLinks();

    return ll;
  }

  public FlowLink findLinkByNodes(ActivityNode from, ActivityNode to)
  {
    if (from == null || to == null)
      return null;

    JGoPort fromPort = from.getOutputPort();
    if (fromPort == null)
      return null;

    JGoPort toPort = to.getInputPort();
    if (toPort == null)
      return null;

    JGoListPosition pos = toPort.getFirstLinkPos();
    while (pos != null) {
      JGoLink l = toPort.getLinkAtPos(pos);
      pos = toPort.getNextLinkPos(pos);

      JGoPort src = l.getFromPort();
      if (src == fromPort) {
        return (FlowLink)l;
      }
    }
    return null;
  }


  public JGoPen getLinkPen()
  {
    return myPen;
  }

  public void setLinkPen(JGoPen p)
  {
    if (!myPen.equals(p)) {
      myPen = p;
      // now update all links
      JGoListPosition pos = getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = getObjectAtPos(pos);
        // only consider top-level objects
        pos = getNextObjectPosAtTop(pos);
        if (obj instanceof JGoLink) {
          JGoLink link = (JGoLink)obj;
          link.setPen(p);
        }
      }
    }
  }


  // toggle the routing style of the links
  void toggleOrthogonalFlows()
  {
    startTransaction();
    setOrthogonalFlows(!isOrthogonalFlows());
    endTransaction("Orthogonal Flows");
  }

  public boolean isOrthogonalFlows()
  {
    return myOrthoLinks;
  }

  void setOrthogonalFlows(boolean b)
  {
    if (myOrthoLinks != b) {
      myOrthoLinks = b;
      // now update all links
      JGoListPosition pos = getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = getObjectAtPos(pos);
        // only consider top-level objects
        pos = getNextObjectPosAtTop(pos);
        if (obj instanceof JGoLink) {
          JGoLink link = (JGoLink)obj;
          link.setOrthogonal(b);
        }
      }
    }
  }


  // For this sample application, just read and write process documents
  // as files using the default serialization or as a simple XML document.

  public static ProcessDocument open(Component parent, String defaultLocation)
  {
    JFileChooser chooser = new JFileChooser();
    if ((defaultLocation != null) && (!defaultLocation.equals(""))) {
      File currentFile = new File(defaultLocation);
      chooser.setCurrentDirectory(currentFile);
    }
    else
      chooser.setCurrentDirectory(null);
    chooser.setAcceptAllFileFilterUsed(false);
    Filter wflInputFilter = new Filter(".wfl", "Flower temporary serialization format (*.wfl)");
    Filter xmlInputFilter = new Filter(".xml", "Extensible Markup Language (*.xml)");
    Filter svgInputFilter = new Filter(".svg", "JGo SVG with XML extensions (*.svg)");
    chooser.addChoosableFileFilter(wflInputFilter);
    chooser.addChoosableFileFilter(xmlInputFilter);
    chooser.addChoosableFileFilter(svgInputFilter);
    chooser.setFileFilter(wflInputFilter);
    int returnVal = chooser.showOpenDialog(parent);
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      String name = chooser.getSelectedFile().getName();
      String loc = chooser.getSelectedFile().getAbsolutePath();
      loc.toLowerCase();
      FileInputStream fstream = null;
      try {
        fstream = new FileInputStream(loc);
        ProcessDocument doc = null;
        if (loc.endsWith(".xml")) {
          doc = loadXML(fstream);
          if (doc == null) {
            fstream = new FileInputStream(loc);
            doc = loadSVG1(fstream);
          }
        }
        else if (loc.endsWith(".svg"))
          doc = loadSVG1(fstream);
        else
          doc = loadObjects(fstream);
        if (doc == null)
          return null;
        doc.setName(name);
        doc.updateLocationModifiable();
        doc.updatePaperColor();
        doc.setModified(false);
        // the UndoManager is transient and must be setup again when
        // created from serialization
        // but we also need to ignore all changes up to now anyway,
        // so we'll just throw away the old manager and create a new one
        doc.setUndoManager(new JGoUndoManager());
        return doc;
      } catch (IOException x) {
        JOptionPane.showMessageDialog(null,
          x,
          "Open Document Error",
          javax.swing.JOptionPane.ERROR_MESSAGE);
        return null;
      } catch (Exception x) {
        JOptionPane.showMessageDialog(null,
          x,
          "Loading Document Exception",
          javax.swing.JOptionPane.ERROR_MESSAGE);
        return null;
      } finally {
        try { if (fstream != null) fstream.close(); } catch (Exception x) {}
      }
    } else {
      return null;
    }
  }

  public void save()
  {
    if (getLocation().equals("")) {
      saveAs(".wfl");
    } else {
      store();
    }
  }

  public void store()
  {
    if (!getLocation().equals("")) {
      FileOutputStream fstream = null;
      try {
        fstream = new FileOutputStream(getLocation());
        if (myDescription.equals(sFlowerXML))
          storeXML(fstream);
        else if (myDescription.equals(sJGoSVGXML))
          storeSVG1(fstream, true, true);
        else if (myDescription.equals(sJGoSVG))
          storeSVG1(fstream, false, true);
        else if (myDescription.equals(sJGoXML))
          storeSVG1(fstream, true, false);
        else if (myDescription.equals(sBatikSVG))
          storeSVG2(fstream);
        else
          storeObjects(fstream);
      } catch (Exception x) {
        JOptionPane.showMessageDialog(null,
          x,
          "Save Document Error",
          javax.swing.JOptionPane.ERROR_MESSAGE);
      } finally {
        try { if (fstream != null) fstream.close(); } catch (Exception x) {}
        setModified(false);
      }
    }
  }

  public void saveAs(String fileType)
  {
    JFileChooser chooser = new JFileChooser();
    String loc = getLocation();
    File currentFile = new File(loc);
    chooser.setCurrentDirectory(currentFile);
    chooser.setAcceptAllFileFilterUsed(false);
    Filter wflFilter = new Filter(".wfl", sFlowerWFL);
    Filter svgFilter1 = new Filter(".svg", sJGoSVGXML);
    Filter svgFilter2 = new Filter(".svg", sBatikSVG);
    Filter svgFilter3 = new Filter(".svg", sJGoSVG);
    Filter xmlFilter = new Filter(".xml", sFlowerXML);
    Filter xmlFilter2 = new Filter(".xml", sJGoXML);
    chooser.addChoosableFileFilter(wflFilter);
    chooser.addChoosableFileFilter(svgFilter1);
    chooser.addChoosableFileFilter(svgFilter3);
    chooser.addChoosableFileFilter(svgFilter2);
    chooser.addChoosableFileFilter(xmlFilter2);
    chooser.addChoosableFileFilter(xmlFilter);
    if (fileType.equalsIgnoreCase(".svg"))
      chooser.setFileFilter(svgFilter1);
    else if (fileType.equalsIgnoreCase(".xml"))
      chooser.setFileFilter(xmlFilter);
    else
      chooser.setFileFilter(wflFilter);
    int returnVal = chooser.showSaveDialog(null);
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      String ext = ".wfl";
      Filter FileFilter = (Filter)(chooser.getFileFilter());
      myDescription = FileFilter.getDescription();
      ext = FileFilter.getExtension();
      String name = chooser.getSelectedFile().getName();
      setName(name);
      loc = chooser.getSelectedFile().getAbsolutePath();
      String loc2 = loc.toLowerCase();
      if (loc2.indexOf(".") == -1) {
        loc += ext;
      }

      setLocation(loc);
      store();
    }
  }


  static public ProcessDocument loadObjects(InputStream ins)
    throws IOException, ClassNotFoundException
  {
    ObjectInputStream istream = new ObjectInputStream(ins);
    Object newObj = istream.readObject();
    if (newObj instanceof ProcessDocument) {
      ProcessDocument doc = (ProcessDocument)newObj;
      return doc;
    } else {
      return null;
    }
  }

  public void storeObjects(OutputStream outs)
    throws IOException
  {
    ObjectOutputStream ostream = new ObjectOutputStream(outs);
    ostream.writeObject(this);
    ostream.flush();
  }

  static public ProcessDocument loadSVG1(InputStream ins)
  {
    ProcessDocument doc = new ProcessDocument();
    try {
      DefaultDocument svgDomDoc = new DefaultDocument();
      svgDomDoc.SVGReadDoc(ins, doc);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return doc;
  }

  static public ProcessDocument loadXML(InputStream ins)
    throws IOException, UnsupportedOperationException
  {
    ProcessDocument doc = new ProcessDocument();
    Document document = null;
    try {
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      DocumentBuilder builder = factory.newDocumentBuilder();
      document = builder.parse(ins);

      Element process = document.getDocumentElement();
      if (process != null) {
        // If the root element isn't what we expect, just leave
        if (!process.getTagName().equals("Process"))
          return null;
        // first handle the ProcessDocument attributes
        doc.setName(process.getAttribute("name"));
        doc.setLocation(process.getAttribute("location"));
        String v = process.getAttribute("lastnodeid");
        doc.myLastNodeID = Integer.parseInt(v);
        doc.setOrthogonalFlows(process.getAttribute("ortholinks").equals("1"));

        // then create the ActivityNodes and FlowLinks
        NodeList elts = process.getElementsByTagName("*");
        for (int i = 0; i < elts.getLength(); i++) {
          Node item = elts.item(i);
          if (item.getNodeType() == Node.ELEMENT_NODE) {
            Element elt = (Element)item;
            if (elt.getTagName().equals(activityTag)) {
              ActivityNode act = new ActivityNode();
              int actid = Integer.parseInt(elt.getAttribute("id"));
              int acttype = Integer.parseInt(elt.getAttribute("type"));
              int x = Integer.parseInt(elt.getAttribute("x"));
              int y = Integer.parseInt(elt.getAttribute("y"));
              String text = elt.getAttribute("text");
              act.initialize(acttype, actid);
              act.setTopLeft(x, y);
              act.setText(text);
              doc.addObjectAtTail(act);
            } else if (elt.getTagName().equals(flowTag)) {
              int fromid = Integer.parseInt(elt.getAttribute("from"));
              int toid = Integer.parseInt(elt.getAttribute("to"));
              ActivityNode from = doc.findNodeByID(fromid);
              ActivityNode to = doc.findNodeByID(toid);
              if (from != null && to != null) {
                FlowLink flow = new FlowLink(from.getOutputPort(), to.getInputPort());
                String text = elt.getAttribute("text");
                flow.setText(text);
                doc.addObjectAtTail(flow);
              }
            }
          }
        }
      }
    } catch (SAXParseException spe) {
      // Error generated by the parser
      System.err.println ("\n** Parsing error"
         + ", line " + spe.getLineNumber ()
         + ", uri " + spe.getSystemId ());
      System.err.println("   " + spe.getMessage() );

      // Use the contained exception, if any
      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    } catch (SAXException sxe) {
      // Error generated by this application
      // (or a parser-initialization error)
      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    } catch (ParserConfigurationException pce) {
      // Parser with specified options can't be built
      pce.printStackTrace();
    } catch (IOException ioe) {
      // I/O error
      ioe.printStackTrace();
    } catch (Exception x) {
      x.printStackTrace();
    }
    return doc;
  }

  public void storeXML(OutputStream outs)
  {
    Document document = null;
    try {
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      DocumentBuilder builder = factory.newDocumentBuilder();
      document = builder.newDocument();

      Element process = (Element)document.createElement(processTag);
      process.setAttribute("name", getName());
      process.setAttribute("location", getLocation());
      process.setAttribute("lastnodeid", Integer.toString(myLastNodeID));
      process.setAttribute("ortholinks", isOrthogonalFlows() ? "1" : "0");
      document.appendChild(process);

      // first produce all of the nodes
      JGoListPosition pos = getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = getObjectAtPos(pos);
        pos = getNextObjectPosAtTop(pos);

        if (obj instanceof ActivityNode) {
          ActivityNode node = (ActivityNode)obj;
          Element act = document.createElement(activityTag);
          act.setAttribute("id", Integer.toString(node.getID()));
          act.setAttribute("type", Integer.toString(node.getActivityType()));
          act.setAttribute("x", Integer.toString(node.getLeft()));
          act.setAttribute("y", Integer.toString(node.getTop()));
          act.setAttribute("text", node.getText());
          process.appendChild(act);
        }
      }

      // then produce all of the links
      pos = getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = getObjectAtPos(pos);
        pos = getNextObjectPosAtTop(pos);

        if (obj instanceof FlowLink) {
          FlowLink link = (FlowLink)obj;
          Element flow = document.createElement(flowTag);
          flow.setAttribute("from", Integer.toString(link.getFromNode().getID()));
          flow.setAttribute("to", Integer.toString(link.getToNode().getID()));
          flow.setAttribute("text", link.getText());
          process.appendChild(flow);
        }
      }
    } catch (ParserConfigurationException pce) {
      // Parser with specified options can't be built
      pce.printStackTrace();
    }

    if (document != null) {
      try {
        TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer serializer = transformerFactory.newTransformer();
        serializer.setOutputProperty(OutputKeys.METHOD, "xml");
        serializer.setOutputProperty(OutputKeys.INDENT, "yes");
        serializer.transform(new DOMSource(document), new StreamResult(outs));
      } catch (Exception x) {
        x.printStackTrace();
      }
    }
  }


  public void storeSVG1(OutputStream outs, boolean genXMLExtensions, boolean genSVG)
  {
  DefaultDocument svgDomDoc = new DefaultDocument();
  svgDomDoc.setGenerateJGoXML(genXMLExtensions);
  svgDomDoc.setGenerateSVG(genSVG);
  svgDomDoc.SVGWriteDoc(outs, this);
  }

  public void storeSVG2(OutputStream outs)
    throws IOException, UnsupportedOperationException
  {
  SVGGoView svgView = new SVGGoView();
  svgView.setDocument(this);
  svgView.generateSVG(outs);
  }

  public void copyNewValueForRedo(JGoDocumentChangedEdit e)
  {
    switch (e.getHint()) {
      case NAME_CHANGED:
        e.setNewValue(getName());
        return;
      case LOCATION_CHANGED:
        e.setNewValue(getLocation());
        return;
      default:
        super.copyNewValueForRedo(e);
        return;
    }
  }

  public void changeValue(JGoDocumentChangedEdit e, boolean undo)
  {
    switch (e.getHint()) {
      case NAME_CHANGED:
        setName((String)e.getValue(undo));
        return;
      case LOCATION_CHANGED:
        setLocation((String)e.getValue(undo));
        return;
      default:
        super.changeValue(e, undo);
        return;
    }
  }

  public void endTransaction(String pname)
  {
    super.endTransaction(pname);
    AppAction.updateAllActions();
  }

  public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
  {
  // Add <ProcessDocument> element
  if (svgDoc.JGoXMLOutputEnabled()) {
    DomElement processDocElement = svgDoc.createJGoClassElement(
        "com.nwoods.jgo.examples.flower.ProcessDocument", jGoElementGroup);
    processDocElement.setAttribute("isOrtho", myOrthoLinks ? "1" : "0");
    processDocElement.setAttribute("name", myName);
    processDocElement.setAttribute("location", myLocation);
    processDocElement.setAttribute("lastid", Integer.toString(myLastNodeID));
    if (!svgDoc.isRegisteredReference(myPen)) {
      processDocElement.setAttribute("embeddedlinkpen", "true");
      DomElement subGroup = svgDoc.createElement("g");
      processDocElement.appendChild(subGroup);
      myPen.SVGWriteObject(svgDoc, subGroup);
      svgDoc.registerReferencingNode(processDocElement, "linkpen", myPen);
    }
  }
  // Have superclass add to the JGoObject group
  super.SVGWriteObject(svgDoc, jGoElementGroup);
  }

  public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
  {
    if (jGoChildElement != null) {
      // This is a <ProcessDocument> element
      myOrthoLinks = jGoChildElement.getAttribute("isOrtho").equals("1");
      myName = jGoChildElement.getAttribute("name");
      myLocation = jGoChildElement.getAttribute("location");
      myLastNodeID = Integer.parseInt(jGoChildElement.getAttribute("lastid"));
      if (jGoChildElement.getAttribute("embeddedlinkpen").equals("true")) {
        svgDoc.SVGTraverseChildren(jGoDoc, jGoChildElement, null, false);
      }
      String pen = jGoChildElement.getAttribute("linkpen");
      svgDoc.registerReferencingObject(this, "linkpen", pen);
      super.SVGReadObject(svgDoc, jGoDoc, svgElement,
                          jGoChildElement.getNextSiblingJGoClassElement());

      super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
    }
    return svgElement.getNextSibling();
  }


  // Constants
  private static final String processTag = "Process";
  private static final String activityTag = "Activity";
  private static final String flowTag = "Flow";

  // Event hints
  public static final int NAME_CHANGED = JGoDocumentEvent.LAST + 1;
  public static final int LOCATION_CHANGED = JGoDocumentEvent.LAST + 2;


  // State
  private JGoPen myPen = JGoPen.make(JGoPen.SOLID, 2, Color.black);
  private boolean myOrthoLinks = false;

  private String myName = "";
  private String myLocation = "";
  private String myDescription = "";

  private int myLastNodeID = -1;

  private transient boolean myIsLocationModifiable = true;
  private transient boolean myIsModified = false;

  static final String sFlowerWFL = "Flower temporary serialization format, read/write (*.wfl)";
  static final String sJGoSVGXML = "JGo SVG with XML extensions, read/write (*.svg)";
  static final String sBatikSVG = "Batik SVG, export only (*.svg)";
  static final String sJGoSVG = "JGo SVG, export only (*.svg)";
  static final String sFlowerXML = "Flower Custom XML, read/write (*.xml)";
  static final String sJGoXML = "JGo XML, read/write (*.xml)";
}

