/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.flower;

import java.awt.*;
import java.awt.event.*;
import java.awt.print.*;
import java.awt.dnd.*;
import java.awt.datatransfer.*;
import java.awt.geom.*;
import javax.swing.*;
import java.util.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.examples.*;

// Provide a view of a ProcessDocument
// Implement various command handlers
public class ProcessView extends JGoView implements JGoViewListener
{
  public ProcessView()
  {
    super();
  }

  public ProcessView(JGoDocument doc)
  {
    super(doc);
  }

  public void initialize(Flower app, JInternalFrame frame)
  {
    myApp = app;
    myInternalFrame = frame;
    addViewListener(this);
    setGridWidth(10);
    setGridHeight(10);
    updateTitle();
  }

  // creating a ProcessView without specifying a document in the constructor
  // needs to create a ProcessDocument rather than a generic JGoDocument
  public JGoDocument createDefaultModel()
  {
    return new ProcessDocument();
  }

  // convenience method--the return value is a ProcessDocument instead
  // of a JGoDocument
  ProcessDocument getDoc()
  {
    return (ProcessDocument)getDocument();
  }

  Flower getApp() { return myApp; }
  JInternalFrame getInternalFrame() { return myInternalFrame; }


  // handle DELETE, HOME, and arrow keys as well as the page up/down keys
  public void onKeyEvent(KeyEvent evt)
  {
    int t = evt.getKeyCode();
    if (t == KeyEvent.VK_DELETE) {
      if (getDoc().isModifiable()) {
        deleteSelection();
      }
    } else if (t == KeyEvent.VK_HOME) {
      setViewPosition(0, 0);
    } else if (t == KeyEvent.VK_RIGHT) {
      if (getDoc().isModifiable()) {
        doMoveSelection(0, getGridWidth(), 0, EventMouseUp);
      }
    } else if (t == KeyEvent.VK_LEFT) {
      if (getDoc().isModifiable()) {
        doMoveSelection(0, -getGridWidth(), 0, EventMouseUp);
      }
    } else if (t == KeyEvent.VK_DOWN) {
      if (getDoc().isModifiable()) {
        doMoveSelection(0, 0, getGridHeight(), EventMouseUp);
      }
    } else if (t == KeyEvent.VK_UP) {
      if (getDoc().isModifiable()) {
        doMoveSelection(0, 0, -getGridHeight(), EventMouseUp);
      }
    } else if (t == KeyEvent.VK_ESCAPE) {
      getSelection().clearSelection();
    } else if (Character.isLetter(evt.getKeyChar())) {
      if (!selectNextNode(evt.getKeyChar()))
        Toolkit.getDefaultToolkit().beep();
    } else {
      super.onKeyEvent(evt);
    }
  }


  // an example of how to implement popup menus
  public boolean doMouseDown(int modifiers, Point dc, Point vc)
  {
    JGoObject obj = pickDocObject(dc, true);
    if (obj != null &&
        getCurrentMouseEvent() != null &&
        getCurrentMouseEvent().isPopupTrigger()) {
      selectObject(obj);
      return doPopupMenu(modifiers, dc, vc);
    }
    // otherwise implement the default behavior
    return super.doMouseDown(modifiers, dc, vc);
  }

  public boolean doMouseUp(int modifiers, Point dc, Point vc)
  {
    JGoObject obj = pickDocObject(dc, true);
    if (obj != null &&
        getCurrentMouseEvent() != null &&
        getCurrentMouseEvent().isPopupTrigger()) {
      selectObject(obj);
      return doPopupMenu(modifiers, dc, vc);
    }
    // otherwise implement the default behavior
    return super.doMouseUp(modifiers, dc, vc);
  }

  public boolean doMouseDblClick(int modifiers, Point dc, Point vc)
  {
    JGoObject obj = pickDocObject(dc,false);
    if (obj == null) {
      JOptionPane.showInternalMessageDialog(this,
        "The diagram backround has no editable properties",
        "information", JOptionPane.INFORMATION_MESSAGE);
    }
    boolean bReturn = super.doMouseDblClick(modifiers, dc, vc);
    return bReturn;
  }

  public boolean doPopupMenu(int modifiers, Point dc, Point vc)
  {
    JPopupMenu popup = getPopupMenu();
    popup.show(this, vc.x, vc.y);
    return true;
  }

  // there's just a single JPopupMenu, shared by all instances of this class,
  // that gets initialized in the app initialization
  static public JPopupMenu getPopupMenu() { return myPopupMenu; }


  // implement JGoViewListener
  // just need to keep the actions enabled appropriately
  // depending on the selection
  public void viewChanged(JGoViewEvent e)
  {
    // if the selection changed, maybe some commands need to
    // be disabled or re-enabled
    switch(e.getHint()) {
      case JGoViewEvent.UPDATE_ALL:
      case JGoViewEvent.SELECTION_GAINED:
      case JGoViewEvent.SELECTION_LOST:
      case JGoViewEvent.SCALE_CHANGED:
        AppAction.updateAllActions();
        break;
    }
  }

  // implement JGoDocumentListener
  // here we just need to keep the title bar up-to-date
  public void documentChanged(JGoDocumentEvent evt)
  {
    if ((evt.getHint() == ProcessDocument.NAME_CHANGED) ||
        (evt.getHint() == JGoDocumentEvent.MODIFIABLE_CHANGED)) {
      updateTitle();
    }
    super.documentChanged(evt);
  }

  // have the title bar for the internal frame include the name
  // of the document and whether it's read-only
  public void updateTitle()
  {
    if (getInternalFrame() != null) {
      String title = getDoc().getName();
      if (!getDocument().isModifiable())
        title += " (read-only)";
      getInternalFrame().setTitle(title);
      getInternalFrame().repaint();
    }
  }


  // override newLink to force the creation of a FlowLink
  // instead of JGoLink
  // let ProcessDocument do the work
  public void newLink(JGoPort from, JGoPort to)
  {
    JGoLink l = getDoc().newLink(from, to);
    fireUpdate(JGoViewEvent.LINK_CREATED, 0, l);
    getDoc().endTransaction("new FlowLink");
  }

  // implement commands for creating activities in the process
  // let ProcessDocument do the work

  public void insertActivity()
  {
    getDoc().startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 70);
    ActivityNode snode = getDoc().newNode(ActivityNode.Activity);
    snode.setTopLeft(loc);
    getDoc().endTransaction("new Activity");
  }

  public void insertInput()
  {
    getDoc().startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 70);
    ActivityNode snode = getDoc().newNode(ActivityNode.Start);
    snode.setTopLeft(loc);
    getDoc().endTransaction("new Output");
  }

  public void insertOutput()
  {
    getDoc().startTransaction();
    Point loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 70);
    ActivityNode snode = getDoc().newNode(ActivityNode.Finish);
    snode.setTopLeft(loc);
    getDoc().endTransaction("new Input");
  }


  // the default place to put stuff if not dragged there
  public Point getDefaultLocation()
  {
    // to avoid constantly putting things in the same place,
    // keep shifting the default location
    if (myDefaultLocation != null) {
      myDefaultLocation.x += 10;
      myDefaultLocation.y += 10;
    }
    return myDefaultLocation;
  }

  public void setDefaultLocation(Point loc)
  {
    myDefaultLocation = loc;
  }


  // although we use the standard drag-and-drop behavior, we do
  // want to modify the objects that are dropped
  public void drop(DropTargetDropEvent e)
  {
    // try standard drop action
    JGoCopyEnvironment map = getDocument().createDefaultCopyEnvironment();
    getDocument().startTransaction();
    if (getDocument().isModifiable() &&
        doDrop(e, map)) {
      // for this demo application, we need to modify the copied objects
      // to make them a standard size and draggable
      Iterator i = map.values().iterator();
      while (i.hasNext()) {
        Object o = i.next();
        if (o instanceof ActivityNode) {
          ActivityNode anode = (ActivityNode)o;
          anode.setSize(ActivityNode.getStdSize());
          anode.resetEditability();
          snapObject(anode);
        }
      }
      fireUpdate(JGoViewEvent.EXTERNAL_OBJECTS_DROPPED, 0, null);
      getDocument().endTransaction("Drop");
    } else {
      e.rejectDrop();
      getDocument().endTransaction(false);
    }
  }


  // bring up the appropriate dialog, based on the current selection
  void editObjectProperties()
  {
    JGoSelection sel = getSelection();
    if (sel.isEmpty()) {
      getApp().editProcessProperties();
      return;
    }

    getDocument().startTransaction();

    JGoListPosition pos = sel.getFirstObjectPos();
    while (pos != null) {
      JGoObject obj = sel.getObjectAtPos(pos);
      pos = sel.getNextObjectPos(pos);

      if (!obj.isTopLevel())
        continue;
      if (obj instanceof ActivityNode) {
        editActivity((ActivityNode)obj);
      } else if (obj instanceof FlowLink) {
        editFlow((FlowLink)obj);
      }
    }

    getDocument().endTransaction("Object Properties");
  }


  void editActivity(ActivityNode snode)
  {
    switch(snode.getActivityType()) {
      case ActivityNode.Start:
          JOptionPane.showInternalMessageDialog(this,
            "A Start node has no editable properties",
            "information", JOptionPane.INFORMATION_MESSAGE);
        break;
      case ActivityNode.Finish:
          JOptionPane.showInternalMessageDialog(this,
            "A Finish node has no editable properties",
            "information", JOptionPane.INFORMATION_MESSAGE);
        break;
      case ActivityNode.Activity:
        ActivityDialog activityDialog = new ActivityDialog(getFrame(), snode);
        activityDialog.setLocationRelativeTo(null);
        activityDialog.setVisible(true);
        break;
      default:
        break;
    }
  }

  void editFlow(FlowLink link)
  {
    FlowDialog flowDialog = new FlowDialog(getFrame(), link);
    flowDialog.setLocationRelativeTo(null);
    flowDialog.setVisible(true);
  }

  // toggle the grid appearance
  void showGrid()
  {
    int style = getGridStyle();
    if (style == JGoView.GridInvisible) {
      style = JGoView.GridDot;
      setGridPen(JGoPen.black);
      setSnapMove(JGoView.SnapJump);
    } else {
      style = JGoView.GridInvisible;
      setSnapMove(JGoView.NoSnap);
    }
    setGridStyle(style);
  }


  // printing support
  public Rectangle2D.Double getPrintPageRect(Graphics2D g2, PageFormat pf)
  {
    // leave some space at the bottom for a footer
    return new Rectangle2D.Double(pf.getImageableX(), pf.getImageableY(),
                                  pf.getImageableWidth(), pf.getImageableHeight() - 20);
  }

  public void printDecoration(Graphics2D g2, PageFormat pf, int hpnum, int vpnum)
  {
    // draw corners around the getPrintPageRect area
    super.printDecoration(g2, pf, hpnum, vpnum);

    // print the n,m page number in the footer
    String msg = Integer.toString(hpnum);
    msg += ", ";
    msg += Integer.toString(vpnum);

    Paint oldpaint = g2.getPaint();
    g2.setPaint(Color.black);
    Font oldfont = g2.getFont();
    g2.setFont(new Font(JGoText.getDefaultFontFaceName(), Font.PLAIN, 10));
    g2.drawString(msg, (int)(pf.getImageableX() + pf.getImageableWidth()/2),
                       (int)(pf.getImageableY() + pf.getImageableHeight() - 10));
    g2.setPaint(oldpaint);
    g2.setFont(oldfont);
  }

  public double getPrintScale(Graphics2D g2, PageFormat pf)
  {
    return getScale();
  }


  void zoomIn()
  {
    double newscale = Math.rint(getScale() / 0.9f * 100f) / 100f;
    setScale(newscale);
  }

  void zoomOut()
  {
    double newscale = Math.rint(getScale() * 0.9f * 100f) / 100f;
    setScale(newscale);
  }

  void zoomNormal()
  {
    setScale(1.0d);
  }

  void zoomToFit()
  {
    double newscale = 1;
    if (!getDocument().isEmpty()){
      double extentWidth = getExtentSize().width;
      double printWidth = getPrintDocumentSize().width;
      double extentHeight = getExtentSize().height;
      double printHeight = getPrintDocumentSize().height;
      newscale = Math.min((extentWidth / printWidth),(extentHeight / printHeight));
    }
    if (newscale > 2)
      newscale = 1;
    newscale *= getScale();
    setScale(newscale);
    setViewPosition(0, 0);
  }


  public boolean matchesNodeLabel(ActivityNode node, char c)
  {
    if (node == null) return false;
    String name = node.getText();
    return (name.length() > 0 &&
            Character.toUpperCase(name.charAt(0)) == c);
  }

  public boolean selectNextNode(char c)
  {
    c = Character.toUpperCase(c);

    JGoDocument doc = getDocument();

    ActivityNode startnode = null;
    JGoObject obj = getSelection().getPrimarySelection();
    if (obj != null && obj instanceof ActivityNode)
      startnode = (ActivityNode)obj;

    JGoListPosition startpos = null;
    if (startnode != null)
      startpos = doc.findObject(startnode);

    JGoListPosition pos = startpos;
    if (pos != null)
      pos = doc.getNextObjectPosAtTop(pos);

    while (pos != null) {
      obj = doc.getObjectAtPos(pos);
      pos = doc.getNextObjectPosAtTop(pos);

      if (obj instanceof ActivityNode) {
        ActivityNode pn = (ActivityNode)obj;
        if (matchesNodeLabel(pn, c)) {
          selectObject(pn);
          scrollRectToVisible(pn.getBoundingRect());
          return true;
        }
      }
    }
    pos = doc.getFirstObjectPos();
    while (pos != null && pos != startpos) {
      obj = doc.getObjectAtPos(pos);
      pos = doc.getNextObjectPosAtTop(pos);

      if (obj instanceof ActivityNode) {
        ActivityNode pn = (ActivityNode)obj;
        if (matchesNodeLabel(pn, c)) {
          selectObject(pn);
          scrollRectToVisible(pn.getBoundingRect());
          return true;
        }
      }
    }
    return false;
  }
  public boolean validLink(JGoPort from, JGoPort to)
  {
    // Set a flag in the node being linked from
    boolean bReturn = true;
    ActivityNode fromNode = null;
    JGoObject fromObj = from.getParentNode();
    if (fromObj instanceof ActivityNode) {
      fromNode = (ActivityNode)fromObj;
      int nFlags = fromNode.getFlags();
      fromNode.setFlags(fromNode.getFlags() | 0x10000);
    }
    // Recursively traverse nodes starting from node being linked to
    // looking for the flag.  If found, we have a circular path.
    JGoObject toObj = to.getParentNode();
    if (toObj instanceof ActivityNode) {
      ActivityNode node = (ActivityNode)toObj;
      bReturn =  !node.downstreamNodeContainsFlag(0x10000);
    }
    fromNode.setFlags(fromNode.getFlags() & ~(0x10000));
    return bReturn && super.validLink(from, to);
  }

  // State
  protected Point myDefaultLocation = new Point(10, 10);
  protected Flower myApp = null;
  protected JInternalFrame myInternalFrame = null;
  static protected JPopupMenu myPopupMenu = new JPopupMenu();
}
