/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import javax.swing.*;
import java.util.Random;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  /**
   * Summary description for Clock.
   */
  public class Clock extends MultipleIndicatorMeter {
    public Clock() {}

    public void initialize() {
      super.initialize();
      setLabelRelativeSpot(TopMiddle);
      myHourHand = createHourHand();
      myMinuteHand = createMinuteHand();
      mySecondHand = createSecondHand();
      addIndicator(myHourHand);
      addIndicator(myMinuteHand);
      addIndicator(mySecondHand);
      updateDigital();
    }

    /**
     * The indicator that displays the seconds.
     */
    public ClockHand getSecondHand() { return mySecondHand; }
    public void setSecondHand(ClockHand value) {
      ClockHand old = mySecondHand;
      if (old != value) {
        removeObject(mySecondHand);
        mySecondHand = value;
        addObjectAtTail(mySecondHand);
        mySecondHand.setScale(getScale());
        update(ChangedSecondHand, 0, old);
      }
    }

    /**
     * The indicator that displays the minutes.
     */
    public ClockHand getMinuteHand() { return myMinuteHand; }
    public void setMinuteHand(ClockHand value) {
      ClockHand old = myMinuteHand;
      if (old != value) {
        removeObject(myMinuteHand);
        myMinuteHand = value;
        addObjectAtTail(myMinuteHand);
        myMinuteHand.setScale(getScale());
        update(ChangedMinuteHand, 0, old);
      }
    }

    /**
     * The indicator that displays the hours.
     */
    public HourClockHand getHourHand() { return myHourHand; }
    public void setHourHand(HourClockHand value) {
      HourClockHand old = myHourHand;
      if (old != value) {
        removeObject(myHourHand);
        myHourHand = value;
        addObjectAtTail(myHourHand);
        myHourHand.setScale(getScale());
        update(ChangedHourHand, 0, old);
      }
    }

    protected JGoObject createBackground() {
      JGoRectangle rect = new JGoRectangle();
      rect.setSelectable(false);
      rect.setBrush(JGoBrush.lightGray);
      return rect;
    }

    protected JGoText createLabel() {
      JGoText text = new JGoText();
      text.setSelectable(false);
      text.setTransparent(true);
      return text;
    }

    protected AbstractGraduatedScale createScale() {
      GraduatedScaleElliptical scale = new GraduatedScaleElliptical();
      scale.setMinimum(0);
      scale.setMaximum(60);
      scale.setTickLabels(false);
      scale.setStartAngle(270);
      scale.setSweepAngle(360);
      scale.setVisible(false);

      return scale;
    }

    public ClockHand createSecondHand() {
      ClockHand sec = new ClockHand();
      sec.setSelectable(false);
      sec.setScale(getScale());
      sec.setPen(JGoPen.red);
      sec.setStyle(IndicatorNeedle.StyleLine);
      sec.setConstantLength(true);
      sec.setActionEnabled(false);
      sec.setValue(0);

      return sec;
    }

    public ClockHand createMinuteHand() {
      ClockHand min = new ClockHand();
      min.setSelectable(false);
      min.setScale(getScale());
      min.setPen(JGoPen.black);
      min.setBrush(JGoBrush.black);
      min.setStyle(IndicatorNeedle.StyleKite);
      min.setThickness(15);
      min.setConstantLength(true);
      min.setActionEnabled(false);
      min.setValue(0);

      return min;
    }

    public HourClockHand createHourHand() {
      HourClockHand hour = new HourClockHand();
      hour.setSelectable(false);
      hour.setScale(getScale());
      hour.setPen(JGoPen.black);
      hour.setBrush(JGoBrush.black);
      hour.setStyle(IndicatorNeedle.StyleKite);
      hour.setThickness(15);
      hour.setConstantLength(true);
      hour.setActionEnabled(false);
      hour.setValue(0);

      return hour;
    }


    protected void copyChildren(JGoArea newgroup, JGoCopyEnvironment env) {
      super.copyChildren(newgroup, env);
      Clock newobj = (Clock)newgroup;
      newobj.mySecondHand = (ClockHand)env.get(mySecondHand);
      newobj.myMinuteHand = (ClockHand)env.get(myMinuteHand);
      newobj.myHourHand = (HourClockHand)env.get(myHourHand);

      if (newobj.mySecondHand != null)
        newobj.mySecondHand.setScale(newobj.getScale());
      if (newobj.myMinuteHand != null)
        newobj.myMinuteHand.setScale(newobj.getScale());
      if (newobj.myHourHand != null)
        newobj.myHourHand.setScale(newobj.getScale());
    }

    /**
     * If any part is removed from this group,
     * be sure to remove any references in local fields.
     */
    public JGoObject removeObjectAtPos(JGoListPosition pos) {
      JGoObject obj = super.removeObjectAtPos(pos);
      if (obj == mySecondHand)
        mySecondHand = null;
      else if (obj == myMinuteHand)
        myMinuteHand = null;
      else if (obj == myHourHand)
        myHourHand = null;
      return obj;
    }
    
    public void layoutChildren(JGoObject childchanged) { 
      if (isInitializing()) return;
      super.layoutChildren(childchanged);
      layoutIndicators(childchanged);
    }


    public void layoutIndicators(JGoObject childchanged) {
      JGoObject scale = getScale();
      Point c = new Point(scale.getLeft() + scale.getWidth()/2, scale.getTop() + scale.getHeight()/2);
      if (mySecondHand != null) {
        getSecondHand().setBoundingRect(getScale().getBoundingRect());
        getSecondHand().setPivotPoint(c);
        getSecondHand().setThickness(Math.min(getScale().getWidth(), getScale().getHeight())/20);
        getSecondHand().setLength(Math.min(getScale().getWidth(), getScale().getHeight())*3/8);
      }

      if (myMinuteHand != null) {
        getMinuteHand().setBoundingRect(getScale().getBoundingRect());
        getMinuteHand().setPivotPoint(c);
        getMinuteHand().setThickness(Math.min(getScale().getWidth(), getScale().getHeight())/20);
        getMinuteHand().setLength(Math.min(getScale().getWidth(), getScale().getHeight())*3/8);
      }

      if (myHourHand != null) {
        getHourHand().setBoundingRect(getScale().getBoundingRect());
        getHourHand().setPivotPoint(c);
        getHourHand().setThickness(Math.min(getScale().getWidth(), getScale().getHeight())/20);
        getHourHand().setLength(Math.min(getScale().getWidth(), getScale().getHeight())*2/8);
      }
    }

    public double getSecond() { return Math.floor(getSecondHand().getValue()); }

    public int getMinute() { return (int)Math.floor(getMinuteHand().getValue()); }

    public int getHour() { return (int)Math.floor(getHourHand().getValue()/5); } // (24 hours)/(120 = twice scale range) = 1/5


    /**
     * Set the time that the clock displays.
     */
    public void setTime(int hour, int minute, double second) {
      getSecondHand().setValue(second);
      getMinuteHand().setValue(minute + second/60.0);
      getHourHand().setValue(5*(hour + minute/60.0 + second/(60.0*60.0)));

      updateDigital();
    }


    public void incrementTime() {
      getSecondHand().setValue(getSecondHand().getValue() + 1);
      getMinuteHand().setValue(getMinuteHand().getValue() + 1.0/60.0);
      getHourHand().setValue(getHourHand().getValue() + 5*1.0/(60.0*60.0));

      updateDigital();
    }


    /**
     * Keep the text label display up-to-date.
     */
    public void updateDigital() {
      if (getLabel() == null)
        return;

      String str;

      str = Integer.toString((int)(getHour()%12));
      if (str == "0")
        str = "12";

      str += ":";

      if (getMinute() < 10)
        str += "0";  
      str += Integer.toString(getMinute()) + ":";

      if (getSecond() < 10)
        str += "0";  
      str += Double.toString(getSecond());

      if (getHour() < 12)
        str += "am";
      else
        str += "pm";

      getLabel().setText(str);
    }


    public void copyNewValueForRedo(JGoDocumentChangedEdit e) {
      switch (e.getFlags()) {
        case ChangedBackground:
          e.setNewValue(getBackground());
          return;
        case ChangedScale:
          e.setNewValue(getScale());
          return;
        case ChangedSecondHand:
          e.setNewValue(getSecondHand());
          return;
        case ChangedMinuteHand:
          e.setNewValue(getMinuteHand());
          return;
        case ChangedHourHand:
          e.setNewValue(getHourHand());
          return;
        default:
          super.copyNewValueForRedo(e);
          return;
      }
    }

    /**
     * Handle undo and redo changes.
     */
    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
      switch (e.getFlags()) {
        case ChangedBackground:
          setBackground((JGoObject)e.getValue(undo));
          return;
        case ChangedScale:
          setScale((GraduatedScaleElliptical)e.getValue(undo));
          return;
        case ChangedSecondHand:
          setSecondHand((ClockHand)e.getValue(undo));
          return;
        case ChangedMinuteHand:
          setMinuteHand((ClockHand)e.getValue(undo));
          return;
        case ChangedHourHand:
          setHourHand((HourClockHand)e.getValue(undo));
          return;
        default:
          super.changeValue(e, undo);
          return;
      }
    }

    /**
     * This is a JGoObject update subhint.
     */
    public static final int ChangedSecondHand = 14203;
    /**
     * This is a JGoObject update subhint.
     */
    public static final int ChangedMinuteHand = 14204;
    /**
     * This is a JGoObject update subhint.
     */
    public static final int ChangedHourHand = 14205;

    // Clock state
    private ClockHand mySecondHand;
    private ClockHand myMinuteHand;
    private HourClockHand myHourHand;
  }
