/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import javax.swing.*;
import java.util.Random;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  /**
   * A version of a Meter designed to also maintain
   * a constant IndicatorBar with multiple phases, used to 
   * show regional areas along its AbstractGraduatedScale
   * (e.g. a highlighted red "danger" region on a pressure gauge).
   */
  public class GaugeMeter extends Meter {
    public GaugeMeter() {}

    public void initialize() {
      super.initialize();
      int bot = 30;
      if (getLabel() != null)
        bot += getLabel().getHeight();
      setInsets(new Insets(30, 30, bot, 30));
      myGauge = createGauge();
      JGoListPosition pos = findObject(getScale());
      if (pos != null)
        insertObjectBefore(pos, myGauge);
      else
        addObjectAtTail(myGauge);
      layoutChildren(null);
    }

    protected AbstractGraduatedScale createScale() {
      GraduatedScaleElliptical scale = new GraduatedScaleElliptical();
      scale.setStartAngle(110);
      scale.setSweepAngle(320);
      scale.setTickUnit(2);
      scale.setTickMajorFrequency(10);
      scale.setTickLengthLeft(0.1f);
      scale.setTickLengthRight(10);
      scale.setLabelDistance(10);
      return scale;
    }

    protected AbstractIndicator createIndicator() {
      IndicatorNeedle needle = new IndicatorNeedle();
      needle.setActionEnabled(true);
      needle.setLength(getScale().getHeight()/2);
      needle.setStyle(IndicatorNeedle.StyleKite);
      needle.setThickness(20);
      needle.setBrush(JGoBrush.red);
      return needle;
    }

    /**
     * Method for creating an instance of the meter's standard indicator.
     */
    protected IndicatorBar createGauge() {
      IndicatorBarElliptical gauge = new IndicatorBarElliptical();
      gauge.setThickness(20);

      gauge.addPhase(new Phase(0, 75, Color.green));
      gauge.addPhase(new Phase(75, 90, Color.yellow));

      gauge.setScale(getScale());
      gauge.setValue(gauge.getScale().getMaximum());

      return gauge;
    }

    /**
     * The IndicatorBar which is used to display regions on the scale.
     */
    public IndicatorBar getGauge() { return myGauge; }
    public void setGauge(IndicatorBar value) {
      IndicatorBar old = myGauge;
      if (old != value) {
        if (old != null)
          removeObject(old);
        myGauge = value;
        if (value != null) {
          JGoListPosition pos = findObject(getScale());
          if (pos != null)
            insertObjectBefore(pos, value);
          else
            addObjectAtHead(value);
        }
        update(ChangedGauge, 0, old);
      }
    }

    public void layoutChildren(JGoObject childchanged) { 
      if (isInitializing()) return;
      super.layoutChildren(childchanged);
      if (getScale() != null) {
        if (getGauge() != null) {
          Rectangle r = new Rectangle(getScale().getBoundingRect());
          //r.Inflate(-this.Gauge.Thickness, -this.Gauge.Thickness);
          getGauge().setBoundingRect(r);
        }
        IndicatorNeedle ineedle = null;
        if (getIndicator() instanceof IndicatorNeedle) ineedle = (IndicatorNeedle)getIndicator();
        if (ineedle != null) {
          JGoObject scale = getScale();
          ineedle.setPivotPoint(new Point(scale.getLeft() + scale.getWidth()/2, scale.getTop() + scale.getHeight()/2));
          if (((IndicatorNeedle)getIndicator()).isConstantLength())
            ineedle.setLength(Math.min(getScale().getWidth()/2, getScale().getHeight()/2));
        }
      }
    }

    protected void copyChildren(JGoArea newgroup, JGoCopyEnvironment env) {
      super.copyChildren(newgroup, env);
      GaugeMeter newobj = (GaugeMeter)newgroup;
      newobj.myGauge = (IndicatorBar)env.get(myGauge);
      newobj.myGauge.setScale(newobj.getScale());
    }

    /**
     * If any part is removed from this group,
     * be sure to remove any references in local fields.
     */
    public JGoObject removeObjectAtPos(JGoListPosition pos) {
      JGoObject obj = super.removeObjectAtPos(pos);
      if (obj == myGauge)
        myGauge = null;
      return obj;
    }
    
    public void copyNewValueForRedo(JGoDocumentChangedEdit e) {
      switch (e.getFlags()) {
        case ChangedGauge:
          e.setNewValue(getGauge());
          return;
        default:
          super.copyNewValueForRedo(e);
          return;
      }
    }

    /**
     * Handle undo and redo changes.
     */
    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
      switch (e.getFlags()) {
        case ChangedGauge:
          setGauge((IndicatorBar)e.getValue(undo));
          return;
        default:
          super.changeValue(e, undo);
          return;
      }
    }

    /**
     * This is a JGoObject update subhint.
     */
    public static final int ChangedGauge = 13800;

    private IndicatorBar myGauge;
  }
