/*
*  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
*  Reserved.
*
*  Restricted Rights: Use, duplication, or disclosure by the U.S.
*  Government is subject to restrictions as set forth in subparagraph
*  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
*  52.227-14 Alt. III, as applicable.
*/

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import java.awt.geom.*;
import javax.swing.*;
import java.text.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  /**
   * A 2-D representation of a cylinder.
   */
  public class JGoCylinder extends JGoDrawable {
    /**
     * The constructor produces a cylinder with a standard JGoPen.black
     * outline, no JGoBrush fill, and a default radius dimension of 10.
     */
    public JGoCylinder() {}

    public JGoObject copyObject(JGoCopyEnvironment env) {
      JGoCylinder newobj = (JGoCylinder)super.copyObject(env);
      newobj.myMinorRadius = myMinorRadius;
      newobj.myPerspective = myPerspective;
      newobj.myPath = null;
      return newobj;
    }

    /**
     * Gets the length of minor radius of the cylinder's ellipse.
     * <p>
     * This defaults to 10.
     */
    public int getMinorRadius() { return myMinorRadius; }

    /**
     * Sets the length of minor radius of the cylinder's ellipse.
     * <p>
     * If given a negative value, it uses zero instead.
     */
    public void setMinorRadius(int value) {
      int old = myMinorRadius;
      if (value < 0)
        value = 0;
      if (old != value) {
        myMinorRadius = value;
        resetPath();
        update(ChangedMinorRadius, old, null);
      }
    }

    /**
     * Gets how to draw the cylinder, whether the full ellipse
     * is drawn on the top (JGoObject.TopCenter) or bottom (JGoObject.BottomCenter)
     * or left (JGoObject.LeftCenter) or right (JGoObject.RightCenter).
     * <p>
     * This defaults to JGoObject.TopCenter.
     */
    public int getPerspective() { return myPerspective; }

    /**
     * Gets how to draw the cylinder, whether the full ellipse
     * is drawn on the top (JGoObject.TopCenter) or bottom (JGoObject.BottomCenter)
     * or left (JGoObject.LeftCenter) or right (JGoObject.RightCenter).
     *
     * @value must be one of TopCenter, BottomCenter, LeftCenter, or RightCenter
     */
    public void setPerspective(int value) {
      int old = myPerspective;
      if (old != value) {
        myPerspective = value;
        resetPath();
        update(ChangedPerspective, old, null);
      }
    }

    // whenever this is moved or resized, rebuild the gradient brush
    protected void geometryChange(Rectangle old) {
      super.geometryChange(old);
      resetPath();
    }

    public void paint(Graphics2D g, JGoView view) {
      GeneralPath poly = getPath(view);
      drawPath(g, getPen(), getBrush(), poly);
    }

    GeneralPath getPath(JGoView view) {
      if (myPath == null) {
        myPath = new GeneralPath(GeneralPath.WIND_NON_ZERO, 2*4);
      }
      if (myPath.getCurrentPoint() == null) {
        makePath(myPath, view);
      }
      return myPath;
    }

    void resetPath() {
      if (myPath != null)
        myPath.reset();
    }

    void makePath(GeneralPath path, JGoView view) {
      Rectangle rect = getBoundingRect();
      int x = rect.x;
      int y = rect.y;
      int w = rect.width;
      int h = rect.height;
      int minrad = getMinorRadius();

      if (minrad == 0) {
        int x1 = x;
        int x2 = x + w;
        int y1 = y;
        int y2 = y + h;
        path.moveTo(x1, y1);
        path.lineTo(x2, y1);
        path.lineTo(x2, y2);
        path.lineTo(x1, y2);
        path.lineTo(x1, y1);
      } else {
        if (getPerspective() == JGoObject.TopCenter || getPerspective() == JGoObject.BottomCenter) {
          if (minrad > h/2)
            minrad = h/2;
          int m2 = minrad*2;
          int x1 = x;
          int x2 = x + w;
          int y1 = y;
          int y2 = y + minrad;
          int y4 = y + h - m2;
          int y5 = y + h - minrad;
          if (getPerspective() == JGoObject.TopCenter) {
            path.moveTo(x2, y2);
            path.lineTo(x2, y5);
            path.append(new Arc2D.Float(x1, y4, w, m2, 0, -180, Arc2D.OPEN), true);
            path.lineTo(x1, y2);
            path.append(new Ellipse2D.Float(x1, y1, w, m2), false);
          } else {
            path.moveTo(x1, y5);
            path.lineTo(x1, y2);
            path.append(new Arc2D.Float(x1, y1, w, m2, 180, -180, Arc2D.OPEN), true);
            path.lineTo(x2, y5);
            path.append(new Ellipse2D.Float(x1, y4, w, m2), false);
          }
        } else {  // LeftCenter or RightCenter
          if (minrad > w/2)
            minrad = w/2;
          int m2 = minrad*2;
          int x1 = x;
          int x2 = x + minrad;
          int x4 = x + w - m2;
          int x5 = x + w - minrad;
          int y1 = y;
          int y2 = y + h;
          if (getPerspective() == JGoObject.LeftCenter) {
            path.moveTo(x2, y1);
            path.lineTo(x5, y1);
            path.append(new Arc2D.Float(x4, y1, m2, h, 90, -180, Arc2D.OPEN), true);
            path.lineTo(x2, y2);
            path.append(new Ellipse2D.Float(x1, y1, m2, h), false);
          } else {
            path.moveTo(x5, y2);
            path.lineTo(x2, y2);
            path.append(new Arc2D.Float(x1, y1, m2, h, 270, -180, Arc2D.OPEN), true);
            path.lineTo(x5, y1);
            path.append(new Ellipse2D.Float(x4, y1, m2, h), false);
          }
        }
      }
    }

    /**
     * Consider the actual shape of the cylinder to determine
     * if a given point is inside.
     * <p>
     * Currently this does not take the pen width into account.
     *
     * @param pnt a Point in document coordinates
     */
    public boolean isPointInObj(Point pnt) {
      // do the easy rectangular case first
      if (!super.isPointInObj(pnt))
        return false;
      GeneralPath poly = getPath(null);
      return poly.contains(pnt.x, pnt.y);
    }

    /**
     * Find the point on this cylinder that intersects with a given infinite line from P1 to P2
     * that is closest to point P1.
     *
     * @param p1x point P1's X coordinate we are trying to be closest to, on a line with P2
     * @param p1y point P1's Y coordinate we are trying to be closest to, on a line with P2
     * @param p2x point P2's X coordinate
     * @param p2y point P2's Y coordinate
     * @param result a Point that is modified to hold the intersection point, if there is any intersection
     * @return true if there is an intersection (held in the RESULT parameter) of the infinite line P1-P2 with this cylinder
     */
    public boolean getNearestIntersectionPoint(int p1x, int p1y, int p2x, int p2y, Point result) {
      int pw2 = (getPen() != null ? (getPen().getWidth()+1) / 2 : 1);
      Rectangle rect = getBoundingRect();
      int x = rect.x - pw2;
      int y = rect.y - pw2;
      int w = rect.width + 2*pw2;
      int h = rect.height + 2*pw2;
      int minrad = getMinorRadius();

      int a1x, a1y, a2x, a2y, b1x, b1y, b2x, b2y;  // two straight line endpoints
      int arcx, arcy, arcw, arch, arca, arcs;      // arc rectangle and start/sweep angles
      int ellx, elly, ellw, ellh, ella, ells;      // ellipse rectangle and start/sweep angles for outside
      if (getPerspective() == JGoObject.TopCenter || getPerspective() == JGoObject.BottomCenter) {
        if (minrad > h/2)
          minrad = h/2;
        int m2 = minrad*2;
        int x1 = x;
        int x2 = x + w;
        int y1 = y;
        int y2 = y + minrad;
        int y4 = y + h - m2;
        int y5 = y + h - minrad;
        a1x = x2; a1y = y2;
        a2x = x2; a2y = y5;
        b1x = x1; b1y = y5;
        b2x = x1; b2y = y2;
        if (getPerspective() == JGoObject.TopCenter) {
          arcx = x1; arcy = y4; arcw = w; arch = m2; arca = 0; arcs = -180;
          ellx = x1; elly = y1; ellw = w; ellh = m2; ella = 180; ells = -180;
        } else {
          arcx = x1; arcy = y1; arcw = w; arch = m2; arca = 180; arcs = -180;
          ellx = x1; elly = y4; ellw = w; ellh = m2; ella = 0; ells = -180;
        }
      } else {  // LeftCenter or RightCenter
        if (minrad > w/2)
          minrad = w/2;
        int m2 = minrad*2;
        int x1 = x;
        int x2 = x + minrad;
        int x4 = x + w - m2;
        int x5 = x + w - minrad;
        int y1 = y;
        int y2 = y + h;
        a1x = x2; a1y = y1;
        a2x = x5; a2y = y1;
        b1x = x5; b1y = y2;
        b2x = x2; b2y = y2;
        if (getPerspective() == JGoObject.LeftCenter) {
          arcx = x4; arcy = y1; arcw = m2; arch = h; arca = 270; arcs = -180;
          ellx = x1; elly = y1; ellw = m2; ellh = h; ella = 90; ells = -180;
        } else {
          arcx = x1; arcy = y1; arcw = m2; arch = h; arca = 90; arcs = -180;
          ellx = x4; elly = y1; ellw = m2; ellh = h; ella = 270; ells = -180;
        }
      }

      Point P = new Point();
      float closestdist = 10e20f;
      int closestx = 0;
      int closesty = 0;

      // check two straight line segments
      if (JGoStroke.getNearestIntersectionOnLine(a1x, a1y, a2x, a2y, p1x, p1y, p2x, p2y, P)) {
        float dist = (P.x-p1x) * (P.x-p1x) + (P.y-p1y) * (P.y-p1y);
        if (dist < closestdist) {
          closestdist = dist;
          closestx = P.x; closesty = P.y;
        }
      }
      if (JGoStroke.getNearestIntersectionOnLine(b1x, b1y, b2x, b2y, p1x, p1y, p2x, p2y, P)) {
        float dist = (P.x-p1x) * (P.x-p1x) + (P.y-p1y) * (P.y-p1y);
        if (dist < closestdist) {
          closestdist = dist;
          closestx = P.x; closesty = P.y;
        }
      }

      // check arc
      if (getNearestIntersectionOnArc(arcx, arcy, arcw, arch, p1x, p1y, p2x, p2y, P, arca, -arcs)) {
        // calculate the (non-square-rooted) distance from C to P
        float dist = (P.x-p1x) * (P.x-p1x) + (P.y-p1y) * (P.y-p1y);
        if (dist < closestdist) {
          closestdist = dist;
          closestx = P.x; closesty = P.y;
        }
      }

      // check ellipse
      if (getNearestIntersectionOnArc(ellx, elly, ellw, ellh, p1x, p1y, p2x, p2y, P, ella, -ells)) {
        // calculate the (non-square-rooted) distance from C to P
        float dist = (P.x-p1x) * (P.x-p1x) + (P.y-p1y) * (P.y-p1y);
        if (dist < closestdist) {
          closestdist = dist;
          closestx = P.x; closesty = P.y;
        }
      }

      result.x = closestx;
      result.y = closesty;
      return (closestdist < 10e20f);
    }

    public void copyNewValueForRedo(JGoDocumentChangedEdit e) {
      switch (e.getFlags()) {
        case ChangedMinorRadius:
          e.setNewValueInt(getMinorRadius());
          return;
        case ChangedPerspective:
          e.setNewValueInt(getPerspective());
          return;
        default:
          super.copyNewValueForRedo(e);
          return;
      }
    }

    /**
     * Handle this class's property changes for undo and redo
     */
    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
      switch (e.getFlags()) {
        case ChangedGeometry:
          super.changeValue(e, undo);
          resetPath();
          return;
        case ChangedMinorRadius:
          setMinorRadius(e.getValueInt(undo));
          return;
        case ChangedPerspective:
          setPerspective(e.getValueInt(undo));
          return;
        default:
          super.changeValue(e, undo);
          resetPath();
          return;
      }
    }

    public static boolean getNearestIntersectionOnArc(int rectx, int recty, int rectw, int recth, int p1x, int p1y, int p2x, int p2y, Point result, float startAngle, float sweepAngle) {
      float Rx = rectw/2;
      float Ry = recth/2;

      float Cx = rectx + Rx;
      float Cy = recty + Ry;

      float sa;
      float sw;

      if (sweepAngle < 0) {
        sa = startAngle + sweepAngle;
        sw = -sweepAngle;
      } else {
        sa = startAngle;
        sw = sweepAngle;
      }

      if (p1x != p2x) {

        float m;
        if (p1x > p2x)
          m = (p1y - p2y)/(float)(p1x - p2x);
        else
          m = (p2y - p1y)/(float)(p2x - p1x);

        float b = (p1y - Cy) - m*(p1x - Cx);

        float sqrt = (float)Math.sqrt( (Rx*Rx)*(m*m) + (Ry*Ry) - (b*b) );

        float xplus = ((-((Rx*Rx) * m * b) + (Rx*Ry*sqrt)) / ((Ry*Ry) + ((Rx*Rx) * (m*m)))) + Cx;
        float xminus = ((-((Rx*Rx) * m * b) - (Rx*Ry*sqrt)) / ((Ry*Ry) + ((Rx*Rx) * (m*m)))) + Cx;

        float yplus = m*(xplus - Cx) + b + Cy;
        float yminus = m*(xminus - Cx) + b + Cy;

        float angleplus = GetAngle(xplus - Cx, yplus - Cy);
        float angleminus = GetAngle(xminus - Cx, yminus - Cy);

        if (angleplus < sa)
          angleplus += 360;
        if (angleminus < sa)
          angleminus += 360;
        if (angleplus > sa + sw)
          angleplus -= 360;
        if (angleminus > sa + sw)
          angleminus -= 360;

        boolean bplus = ((angleplus >= sa) && (angleplus <= sa + sw));
        boolean bminus = ((angleminus >= sa) && (angleminus <= sa + sw));

        if (bplus && bminus) {

          float distplus = Math.abs((p1x - xplus)*(p1x - xplus)) + Math.abs((p1y - yplus)*(p1y - yplus));
          float distminus = Math.abs((p1x - xminus)*(p1x - xminus)) + Math.abs((p1y - yminus)*(p1y - yminus));

          if (distplus < distminus) {
            result.x = (int)xplus; result.y = (int)yplus;
          } else {
            result.x = (int)xminus; result.y = (int)yminus;
          }

          return true;

        } else if (bplus && !bminus) {
          result.x = (int)xplus; result.y = (int)yplus;
          return true;

        } else if (!bplus && bminus) {
          result.x = (int)xminus; result.y = (int)yminus;
          return true;

        } else { // (!bplus && !bminus)
          result.x = 0; result.y = 0;
        }
        return false;

      } else {

        float sqrt = (float)Math.sqrt( (Ry*Ry) - ((Ry*Ry)/(Rx*Rx))*((p1x-Cx)*(p1x-Cx)) );

        float yplus = Cy + sqrt;
        float yminus = Cy - sqrt;

        float angleplus = GetAngle(p1x - Cx, yplus - Cy);
        float angleminus = GetAngle(p1x - Cx, yminus - Cy);

        if (angleplus < sa)
          angleplus += 360;
        if (angleminus < sa)
          angleminus += 360;
        if (angleplus > sa + sw)
          angleplus -= 360;
        if (angleminus > sa + sw)
          angleminus -= 360;

        boolean bplus = ((angleplus >= sa) && (angleplus <= sa + sw));
        boolean bminus = ((angleminus >= sa) && (angleminus <= sa + sw));

        if (bplus && bminus) {

          float distplus = Math.abs(yplus - p1y);
          float distminus = Math.abs(yminus - p1y);

          if (distplus < distminus) {
            result.x = (int)p1x; result.y = (int)yplus;
          } else {
            result.x = (int)p1x; result.y = (int)yminus;
          }
          return true;

        } else if (bplus && !bminus) {
          result.x = (int)p1x; result.y = (int)yplus;
          return true;

        } else if (!bplus && bminus) {
          result.x = (int)p1x; result.y = (int)yminus;
          return true;

        } else { //(!bplus && !bminus)
          result.x = 0; result.y = 0;
        }
        return false;
      }
    }

    static float GetAngle(float x, float y) {
      float A;
      if (x == 0) {
        if (y > 0)
          A = 90;
        else
          A = 270;
      } else if (y == 0) {
        if (x > 0)
          A = 0;
        else
          A = 180;
      } else {
        A = (float)(Math.atan(Math.abs(y/x))*180/Math.PI);
        if (x < 0) {
          if (y < 0)
            A += 180;
          else
            A = 180-A;
        } else if (y < 0) {
          A = 360-A;
        }
      }
      return A;
    }

    public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
    {
      // Add <JGoCylinder> element
      if (svgDoc.JGoXMLOutputEnabled()) {
        DomElement JGoCylinder = svgDoc.createJGoClassElement(
            "com.nwoods.jgo.examples.instrumentdemo.JGoCylinder", jGoElementGroup);
          JGoCylinder.setAttribute("minorradius", Integer.toString(myMinorRadius));
          JGoCylinder.setAttribute("perspective", Integer.toString(myPerspective));
      }
      // Have superclass add to the JGoObject group
      super.SVGWriteObject(svgDoc, jGoElementGroup);
    }

    public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
    {
      if (jGoChildElement != null) {
        // This is a <JGoCylinder> element
        myMinorRadius = Integer.parseInt(jGoChildElement.getAttribute("minorradius"));
        myPerspective = Integer.parseInt(jGoChildElement.getAttribute("perspective"));
        super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
      }
      return svgElement.getNextSibling();
    }

    // GoCylinder constants

    /**
     * This is a JGoObject update subhint.
     */
    public static final int ChangedMinorRadius = 1481;
    /**
     * This is a JGoObject update subhint.
     */
    public static final int ChangedPerspective = 1483;

    //GoCylinder state
    private int myMinorRadius = 10;
    private int myPerspective = JGoObject.TopCenter;
    transient private GeneralPath myPath = null;
  }
