/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import javax.swing.*;
import java.text.*;
import com.nwoods.jgo.*;

  /**
   * A node with a resizable icon, a label, and a port on each side.
   * <p>
   * Setting the <see cref="GoNode.Location"/>, <see cref="GoNode.Resizable"/>,
   * <see cref="GoNode.Reshapable"/> and <see cref="GoNode.Shadowed"/>
   * properties actually set the same properties on the
   * <see cref="SelectionObject"/>, which is the <see cref="Icon"/>.
   */
  public class JGoSimpleNode extends JGoNode {
    /**
     * Create an empty JGoSimpleNode--call the
     * <see cref="Initialize(ResourceManager, String, String)"/> method(s)
     * to create an icon, a label, and ports.
     */
    public JGoSimpleNode() {}

    /**
     * Initialize an empty JGoSimpleNode to have an icon, a label, and two ports.
     * @param res">
     * Provides the <c>ResourceManager</c> holding an <c>Image</c> resource named by
     * <paramref name="iconname"/>.  If this parameter is null,
     * <see cref="GoImage.DefaultResourceManager"/> is used instead.
     * </param>
     * @param iconname">
     * The name of the <c>Image</c> resource in the <c>ResourceManager</c>
     * given by <paramref name="res"/>, or else a file name if no resource manager
     * can be used (i.e., when both <paramref name="res"/> is null and
     * <see cref="GoImage.DefaultResourceManager"/> is null).
     * </param>
     * @param name">
     * The initial string value for the <see cref="Label"/>.
     * If this value is null, no label is created for this node.
     * </param>
     */
    public void initialize(String iconname, String name) {
      setInitializing(true);
      myIcon = createIcon(iconname);
      addObjectAtTail(myIcon);
      initializeCommon(name);
    }

    public void initialize(JGoObject icon, String name) {
      setInitializing(true);
      myIcon = icon;
      addObjectAtTail(icon);
      initializeCommon(name);
    }

    private void initializeCommon(String name) {
      myText = name;
      myLabel = createLabel(name);
      addObjectAtTail(myLabel);
      if (myLabel != null)
        myLabel.setUpdatePartner(true);
      myInPort = createPort(true);
      addObjectAtTail(myInPort);
      myOutPort = createPort(false);
      addObjectAtTail(myOutPort);

      setInitializing(false);
      layoutChildren(null);
    }
    
    /**
     * Create and initialize an image to act as the node's icon.
     * @param iconname
     * @return
     */
    protected JGoObject createIcon(String iconname) {
      if (iconname != null) {
        JGoImage ni = new JGoImage();
        ni.loadImage(iconname, false);
        return ni;
      } else {
        JGoRectangle rect = new JGoRectangle();
        rect.setSelectable(false);
        rect.setSize(20, 20);
        return rect;
      }
    }

    /**
     * Create and initialize a text label.
     * @param name
     * @return a non-editable, non-selectable, middle-aligned, non-rescaling <see cref="JGoText"/> object</returns>
     */
    protected JGoText createLabel(String name) {
      JGoText l = null;
      if (name != null) {
        l = new JGoText();
        l.setText(name);
        l.setSelectable(false);
        l.setAlignment(JGoText.ALIGN_CENTER);
      }
      return l;
    }

    /**
     * Create and initialize a port to be either on the left or on the right of this node.
     * @param input
     * @return a <see cref="JGoPort"/></returns>
     * <p>
     * You may want to this to return null when you don't want a port
     * on a particular side.
     * </remarks>
     */
    protected JGoPort createPort(boolean input) {
      JGoPort p = new JGoPort();
      p.setSize(6, 6);
      p.setValidSource(!input);
      p.setValidDestination(input);
      return p;
    }

    /**
     * Make copies of the icon, label and two ports.
     * @param newgroup
     * @param env
     * <p>
     * Remember to this to copy any objects you add to this class.
     * </remarks>
     */
    protected void copyChildren(JGoArea newgroup, JGoCopyEnvironment env) {
      super.copyChildren(newgroup, env);
      JGoSimpleNode newnode = (JGoSimpleNode)newgroup;
      newnode.myText = myText;
      newnode.myIcon = (JGoObject)env.get(myIcon);
      newnode.myLabel = (JGoText)env.get(myLabel);
      newnode.myInPort = (JGoPort)env.get(myInPort);
      newnode.myOutPort = (JGoPort)env.get(myOutPort);
      newnode.myOrientation = myOrientation;
    }
    
    /**
     * Position the label and two ports relative to the icon.
     * @param childchanged
     * <p>
     * When <see cref="JGoObject.Initializing"/> is true, this method does nothing.
     * This method also does nothing if there is no <see cref="Icon"/>.
     * </remarks>
     */
    public void layoutChildren(JGoObject childchanged) {
      if (isInitializing()) return;

      JGoObject icon = getIcon();
      if (icon == null) return;

      if (getOrientation() == OrientationHorizontal) {
        if (getLabel() != null)
          getLabel().setSpotLocation(TopCenter, icon, BottomCenter);
        if (getInPort() != null)
          getInPort().setSpotLocation(RightCenter, icon, LeftCenter);
        if (getOutPort() != null)
          getOutPort().setSpotLocation(LeftCenter, icon, RightCenter);
      } else {
        if (getLabel() != null)
          getLabel().setSpotLocation(LeftCenter, icon, RightCenter);
        if (getInPort() != null)
          getInPort().setSpotLocation(BottomCenter, icon, TopCenter);
        if (getOutPort() != null)
          getOutPort().setSpotLocation(TopCenter, icon, BottomCenter);
      }
    }

    /**
     * If any part is removed from this group,
     * be sure to remove any references in local fields.
     */
    public JGoObject removeObjectAtPos(JGoListPosition pos) {
      JGoObject obj = super.removeObjectAtPos(pos);
      if (obj == myIcon)
        myIcon = null;
      else if (obj == myLabel) {
        myLabel = null;
      } else if (obj == myInPort)
        myInPort = null;
      else if (obj == myOutPort)
        myOutPort = null;
      return obj;
    }
      
    
    /**
     * Gets the <see cref="JGoObject"/> acting as the icon for this node.
     * <p>
     * You might want to the <see cref="createIcon(ResourceManager, String)"/> method(s)
     * if you want to create a different kind of <see cref="GoImage"/> when
     * constructing this kind of node.
     * The new shape will have its Center location,
     * Selectable, Resizable, Reshapable, ResizesRealtime, and Shadowed
     * properties copied from the old shape.
     * </remarks>
     */
    public JGoObject getIcon() { return myIcon; }
    
    public void setIcon(JGoObject value) {
      JGoObject old = myIcon;
      if (old != value) {
        if (old != null)
          removeObject(old);
        myIcon = value;
        if (value != null)
          addObjectAtHead(value);
        update(ChangedIcon, 0, old);
      }
    }

    /**
     * Gets this node's icon, assuming it is a <see cref="GoImage"/>, as it usually is.
     */
    public JGoImage getImage() {
      if (getIcon() instanceof JGoImage)
        return (JGoImage)getIcon();
      else
        return null;
    }

    
    /**
     * When this node's <see cref="Text"/> string changes,
     * be sure to update the label's text.
     * <p>
     * This has a Text string that is normally the same as the Label's text string,
     * but the Label might not exist or might show a different string.
     * </remarks>
     */
    public String getText() { return myText; }
    
    public void setText(String value) {
      String old = myText;
      if (old != value) {
        myText = value;
        update(ChangedText, 0, old);
        if (getLabel() != null)
          getLabel().setText(value);
      }
    }
    
    /**
     * Gets or sets the <see cref="JGoText"/> label.
     * <p>
     * You might want to the <see cref="createLabel"/> method
     * if you want to create a different kind of <see cref="JGoText"/> when
     * constructing this kind of node.
     * </remarks>
     */
    public JGoText getLabel() { return myLabel; }
    
    public void setLabel(JGoText value) {
      JGoText old = myLabel;
      if (old != value) {
        if (old != null) {
          removeObject(old);
        }
        myLabel = value;
        if (value != null) {
          addObjectAtTail(value);
          value.setUpdatePartner(true);
        }
        update(ChangedLabel, 0, old);
      }
    }
    
    /**
     * Gets or sets the port on the left side of this node, normally acting as the "input".
     * <p>
     * You might want to the <see cref="createPort"/> method
     * if you want to create a different kind of <see cref="JGoPort"/> when
     * constructing this kind of node.
     * </remarks>
     */
    public JGoPort getInPort() { return myInPort; }
    public JGoPort getInputPort() { return myInPort; }

    public void setInPort(JGoPort value) {
      JGoPort old = myInPort;
      if (old != value) {
        if (old != null)
          removeObject(old);
        myInPort = value;
        if (value != null)
          addObjectAtTail(value);
        update(ChangedInPort, 0, old);
      }
    }
    
    /**
     * Gets the port on the right side of this node, normally acting as the "output".
     * <p>
     * You might want to the <see cref="createPort"/> method
     * if you want to create a different kind of <see cref="JGoPort"/> when
     * constructing this kind of node.
     * </remarks>
     */
    public JGoPort getOutPort() { return myOutPort; }
    public JGoPort getOutputPort() { return myOutPort; }
    
    public void setOutPort(JGoPort value) {
      JGoPort old = myOutPort;
      if (old != value) {
        if (old != null)
          removeObject(old);
        myOutPort = value;
        if (value != null)
          addObjectAtTail(value);
        update(ChangedOutPort, 0, old);
      }
    }
    
    /**
     * Gets or sets the general orientation of the node
     * <value>
     * This defaults to <c>OrientationHorizontal</c>
     * </value>
     */
    public int getOrientation() { return myOrientation; }
    
    public void setOrientation(int value) {
      setOrientation(value, false);
    }

    private void setOrientation(int o, boolean undoing) {
      int old = myOrientation;
      if (old != o) {
        myOrientation = o;
        update(ChangedOrientation, (int)old, null);
        if (!undoing) {
          OnOrientationUpdate(old);
        }
      }
    }

    /**
     * Determine how to change the whole node when the <see cref="Orientation"/> changes.
     * @param old">the former <see cref="Orientation"/> value</param>
     * <p>
     * A horizontal orientation has the ports positioned on the left and right of the
     * node, with the source links coming into the <see cref="InPort"/> on the left and
     * the destination links going out from the <see cref="OutPort"/> on the right.
     * A vertical orientation has the <see cref="InPort"/> on top, so links come in
     * at the top, and the <see cref="OutPort"/> is on the bottom of the node, so links
     * go out from the bottom.
     * </remarks>
     */
    public void OnOrientationUpdate(int old) {
      if (getOrientation() == OrientationVertical) {
        if (getInPort() != null) {
          getInPort().setToSpot(TopCenter);
          getInPort().setFromSpot(TopCenter);
        }
        if (getOutPort() != null) {
          getOutPort().setToSpot(BottomCenter);
          getOutPort().setFromSpot(BottomCenter);
        }
      } else {
        if (getInPort() != null) {
          getInPort().setToSpot(LeftCenter);
          getInPort().setFromSpot(LeftCenter);
        }
        if (getOutPort() != null) {
          getOutPort().setToSpot(RightCenter);
          getOutPort().setFromSpot(RightCenter);
        }
      }
      layoutChildren(null);
    }


    /**
     * When the label changes its text string, change this node's <see cref="Text"/> property too.
     * @param partner
     * @param hint
     * @param prevInt
     * @param prevVal
     */
    protected void partnerUpdate(JGoObject partner, int hint, int prevInt, Object prevVal) {
      super.partnerUpdate(partner, hint, prevInt, prevVal);
      if (hint == JGoText.ChangedText && partner == getLabel()) {
        setText(getLabel().getText());
      }
    }

    public void copyNewValueForRedo(JGoDocumentChangedEdit e) {
      switch (e.getFlags()) {
        case ChangedText:
          e.setNewValue(getText());
          return;
        case ChangedIcon:
          e.setNewValue(getIcon());
          return;
        case ChangedLabel:
          e.setNewValue(getLabel());
          return;
        case ChangedInPort:
          e.setNewValue(getInPort());
          return;
        case ChangedOutPort:
          e.setNewValue(getOutPort());
          return;
        case ChangedOrientation:
          e.setNewValueInt(getOrientation());
          return;
        default:
          super.copyNewValueForRedo(e);
          return;
      }
    }

    /**
     * Handle this class's property changes for undo and redo
     * @param e
     * @param undo
     */
    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
      switch (e.getFlags()) {
        case ChangedText:
          setText((String)e.getValue(undo));
          return;
        case ChangedIcon:
          setIcon((JGoObject)e.getValue(undo));
          return;
        case ChangedLabel:
          setLabel((JGoText)e.getValue(undo));
          return;
        case ChangedInPort:
          setInPort((JGoPort)e.getValue(undo));
          return;
        case ChangedOutPort:
          setOutPort((JGoPort)e.getValue(undo));
          return;
        case ChangedOrientation:
          setOrientation(e.getValueInt(undo));
          return;
        default:
          super.changeValue(e, undo);
          return;
      }
    }


    // JGoSimpleNode constants
    /**
     * A value for the Orientation property.
     */
    public static final int OrientationHorizontal = 0;

    /**
     * A value for the Orientation property.
     */
    public static final int OrientationVertical = 1;

    /**
     * This is a <see cref="JGoObject.Changed"/> subhint.
     */
    public static final int ChangedText = 2601;
    /**
     * This is a <see cref="JGoObject.Changed"/> subhint.
     */
    public static final int ChangedIcon = 2602;
    /**
     * This is a <see cref="JGoObject.Changed"/> subhint.
     */
    public static final int ChangedLabel = 2603;
    /**
     * This is a <see cref="JGoObject.Changed"/> subhint.
     */
    public static final int ChangedInPort = 2604;
    /**
     * This is a <see cref="JGoObject.Changed"/> subhint.
     */
    public static final int ChangedOutPort = 2605;
    /**
     * This is a <see cref="JGoObject.Changed"/> subhint.
     */
    public static final int ChangedOrientation = 2606;


    // JGoSimpleNode state
    private String myText = "";
    private JGoObject myIcon = null;
    private JGoText myLabel = null;
    private JGoPort myInPort = null;
    private JGoPort myOutPort = null;
    private int myOrientation = OrientationHorizontal;
  }
