/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import java.awt.geom.*;
import javax.swing.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  /**
   * This meter uses a knob that is kept square with a gradient highlight.
   */
  public class KnobMeter extends Meter {
    public KnobMeter() {}

    public void initialize() {
      super.initialize();
      int bot = 30;
      if (getLabel() != null)
        bot += getLabel().getHeight();
      setInsets(new Insets(30, 30, bot, 30));
    }

    public JGoObject copyObject(JGoCopyEnvironment env) {
      KnobMeter newobj = (KnobMeter)super.copyObject(env);
      newobj.UpdateBrush();
      return newobj;
    }

    protected AbstractGraduatedScale createScale() {
      GraduatedScaleElliptical scale = new GraduatedScaleElliptical();
      scale.setStartAngle(90);
      scale.setSweepAngle(315);
      scale.setTickUnit(2);
      scale.setTickMajorFrequency(10);
      return scale;
    }

    protected AbstractIndicator createIndicator() {
      IndicatorKnob knob = new IndicatorKnob();
      knob.setActionEnabled(true);
      knob.setQuantized(true);
      knob.setQuantizeUnit(2);
      return knob;
    }

    private void UpdateBrush() {
      if (getIndicator() != null) {
        Rectangle r = getIndicator().getBoundingRect();
        if (r.width > 0 && r.height > 0) {
          RoundGradientPaint rgp = new RoundGradientPaint(r.x+r.width/3, r.y+r.height/3, Color.white,
            new Point2D.Double(r.width/3, r.height/3), Color.blue);
          getIndicator().setBrush(new JGoBrush(rgp));
        } else {
          getIndicator().setBrush(JGoBrush.make(JGoBrush.SOLID, new Color(127, 127, 220)));
        }
      }
    }

    // whenever this is moved or resized, rebuild the gradient brush
    protected void geometryChange(Rectangle old) {
      super.geometryChange(old);
      UpdateBrush();
    }

    // whenever the children are moved, rebuild the gradient brush
    // make sure the scale and indicator remain circular
    public void layoutChildren(JGoObject childchanged) {
      if (isInitializing()) return;
      if (getBackground() != null) {
        if (childchanged == null) {
          getBackground().setBoundingRect(getBoundingRect());
        }
        if (childchanged == null || childchanged == getBackground()) {
          Insets insets = getInsets();
          Rectangle r = new Rectangle(getBackground().getBoundingRect());
          if (r.width > insets.left + insets.right) {
            r.x += insets.left;
            r.width -= insets.left + insets.right;
          }
          if (r.height > insets.top + insets.bottom) {
            r.y += insets.top;
            r.height -= insets.top + insets.bottom;
          }
          if (r.width != r.height) {
            r.width = Math.min(r.width, r.height);
            r.height = r.width;
            r.x = getBackground().getLeft() + getBackground().getWidth()/2 - r.width/2;
            r.y = getBackground().getTop() + getBackground().getHeight()/2 - r.height/2;
          }
          if (getScale() != null) {
            getScale().setBoundingRect(r);
          }
          if (getIndicator() != null) {
            getIndicator().setBoundingRect(r);
          }
          if (getLabel() != null) {
            getLabel().setSpotLocation(getLabelRelativeSpot(), getBackground(), getLabelSpot());
          }
        }
      }
      UpdateBrush();
    }

    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
      switch (e.getFlags()) {
        case ChangedGeometry:
          super.changeValue(e, undo);
          UpdateBrush();
          break;
        default:
          super.changeValue (e, undo);
          break;
      }
    }

    public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup)
    {
      // Add <KnobMeter> element
      if (svgDoc.JGoXMLOutputEnabled()) {
        DomElement KnobMeter = svgDoc.createJGoClassElement(
            "com.nwoods.jgo.examples.instrumentdemo.KnobMeter", jGoElementGroup);
      }
      // Have superclass add to the JGoObject group
      super.SVGWriteObject(svgDoc, jGoElementGroup);
    }

    public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement)
    {
      if (jGoChildElement != null) {
        // This is a <KnobMeter> element
        super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
      }
      return svgElement.getNextSibling();
    }
  }
