/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import javax.swing.*;
import java.text.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  /**
   * A simulation of a water pipe, that connects the OutputPort
   * of one Tank with the InputPort of another Tank.
   */
  public class Pipe extends JGoLabeledLink {
    public Pipe() {
      setOrthogonal(true);
      setRoundedCorners(true);
      setPen(JGoPen.make(JGoPen.SOLID, 4, Color.gray));
      setHighlight(JGoPen.make(JGoPen.SOLID, 6, Color.black));
      // create the MidLabel GoText, showing the Rate
      JGoLinkLabel lab = new JGoLinkLabel();
      lab.setSelectable(false);
      lab.setBkColor(Color.white);
      lab.setTransparent(false);
      lab.setEditable(true);
      lab.setEditOnSingleClick(true);
      lab.setText(Double.toString(getRate()));
      lab.setAlignment(JGoText.ALIGN_CENTER);
      setMidLabel(lab);
      lab.setUpdatePartner(true);  // notice when the label is changed, so we can update the Rate
    }

    public JGoObject copyObject(JGoCopyEnvironment env) {
      Pipe newobj = (Pipe)super.copyObject(env);
      newobj.myRate = myRate;
      return newobj;
    }

    // when the Rate is modified, also update the Label
    public double getRate() { return myRate; }
    public void setRate(double value) {
      double old = myRate;
      if (old != value) {
        myRate = value;
        update(ChangedRate, 0, new Double(old));
        JGoText lab = null;
        if (getMidLabel() instanceof JGoText) lab = (JGoText)getMidLabel();
        if (lab != null)
          lab.setText(Double.toString(value));
      }
    }

    // when the label is modified, also update the Rate
    protected void partnerUpdate(JGoObject partner, int hint, int prevInt, Object prevVal) {
      if (hint == JGoText.ChangedText && partner instanceof JGoText && partner == getMidLabel()) {
        JGoText lab = (JGoText)partner;
        String s = lab.getText();
        try {
          double v = myFormatter.parse(s).doubleValue();
          setRate(v);
        } catch (ParseException ex) {
          lab.setText((String)prevVal);
        }
      }
    }

    // compute the maximum amount that could pass in one unit of time;
    // this is the least of the Rate the amount available at the source,
    // and the amount of room at the destination
    public double MaxFlow() {
      Tank src = null;
      if (getFromPort() != null && getFromPort().getParentNode() instanceof Tank)
        src = (Tank)getFromPort().getParentNode();
      Tank dst = null;
      if (getToPort() != null && getToPort().getParentNode() instanceof Tank)
        dst = (Tank)getToPort().getParentNode();
      if (src != null && dst != null) {
        return Math.min(getRate(), Math.min(src.getValue()-src.getMinimum(), dst.getMaximum()-dst.getValue()));
      } else {
        return 0;
      }
    }

    // transfer an amount from the source tank to the destination tank;
    // this does not check that the value is not more than the Rate
    public void Flow(double v) {
      Tank src = null;
      if (getFromPort() != null && getFromPort().getParentNode() instanceof Tank)
        src = (Tank)getFromPort().getParentNode();
      Tank dst = null;
      if (getToPort() != null && getToPort().getParentNode() instanceof Tank)
        dst = (Tank)getToPort().getParentNode();
      if (src != null && dst != null) {
        src.setValue(src.getValue() - v);
        dst.setValue(dst.getValue() + v);
      }
    }

    public void copyNewValueForRedo(JGoDocumentChangedEdit e) {
      switch (e.getFlags()) {
        case ChangedRate:
          e.setNewValue(new Double(getRate()));
          return;
        default:
          super.copyNewValueForRedo(e);
          return;
      }
    }

    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
      switch (e.getFlags()) {
        case ChangedRate:
          setRate(e.getValueDouble(undo));
          return;
        default:
          super.changeValue(e, undo);
          return;
      }
    }

    public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup) {
      // Add <Pipe> element
      if (svgDoc.JGoXMLOutputEnabled()) {
        DomElement Pipe = svgDoc.createJGoClassElement(
            "com.nwoods.jgo.examples.instrumentdemo.Pipe", jGoElementGroup);
      }
      // Have superclass add to the JGoObject group
      super.SVGWriteObject(svgDoc, jGoElementGroup);
    }

    public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc,
                                 DomElement svgElement,
                                 DomElement jGoChildElement) {
      if (jGoChildElement != null) {
        // This is a <Pipe> element
        super.SVGReadObject(svgDoc, jGoDoc, svgElement,
                            jGoChildElement.getNextSiblingJGoClassElement());
      }
      return svgElement.getNextSibling();
    }

    public static final int ChangedRate = 12450;

    private NumberFormat myFormatter = NumberFormat.getNumberInstance();

    private double myRate = 1;
  }
