/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import javax.swing.*;
import java.text.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  /**
   * A simulation of a water tank, with one inlet and one outlet that can be connected
   * to other tanks using Pipes.
   */
  public class Tank extends JGoSimpleNode {
    public Tank() {}

    public void initialize() {
      Meter m = new Meter();
      m.setSelectable(false);
      m.setLabel(null);
      m.setInsets(new Insets(0, 0, 0, 0));

      JGoCylinder cyl = new LinearGradientCylinder();
      cyl.setSelectable(false);
      cyl.setMinorRadius(4);
      cyl.setSize(new Dimension(40, 80));
      m.setBackground(cyl);

      GraduatedScaleLinear scale = new GraduatedScaleLinear();
      scale.setMaximum(1000);
      scale.setTickMajorFrequency(10);
      scale.setTickUnit(100);
      scale.setTickLabels(false);
      m.setScale(scale);

      IndicatorBar bar = new IndicatorBar();
      bar.setActionEnabled(true);
      bar.setBrush(JGoBrush.blue);
      bar.setScale(scale);
      bar.setThickness(10);
      bar.setValue(500);
      m.setIndicator(bar);
      bar.setPartner(this);
      bar.setUpdatePartner(true);

      m.setSize(30, 100);
      initialize(m, "Tank");
      setOrientation(JGoSimpleNode.OrientationVertical);
      getLabel().setEditable(true);
      getLabel().setBkColor(Color.white);
      getLabel().setTransparent(false);
      getLabel().setUpdatePartner(true);
    }

    public JGoObject copyObject(JGoCopyEnvironment env) {
      Tank newobj = (Tank)super.copyObject(env);
      Meter newmeter = newobj.getMeter();
      if (newmeter != null) {
        AbstractIndicator ind = newmeter.getIndicator();
        if (ind != null) {
          ind.setPartner(newobj);
          ind.setUpdatePartner(true);
        }
      }
      return newobj;
    }

    /**
     * Method for creating a standard instance of the Tank's TankPorts.
     */
    protected JGoPort createPort(boolean input) {
      JGoPort port = new JGoPort();
      port.setStyle(JGoPort.StyleRectangle);
      port.setPen(JGoPen.black);
      port.setBrush(JGoBrush.gray);
      port.setAutoRescale(false);
      port.setSize(6, 6);
      port.setValidSource(!input);
      port.setValidDestination(input);
      return port;
    }

    public Meter getMeter() {
      if (getIcon() instanceof Meter)
        return (Meter)getIcon();
      else
        return null;
    }

    public void layoutChildren(JGoObject childchanged) {
      if (isInitializing()) return;
      super.layoutChildren(childchanged);
      if (getMeter() == null) return;
      if (getMeter().getScale() == null) return;
      if (getMeter().getScale() == childchanged) return;
      GraduatedScaleLinear gsl = null;
      if (getMeter().getScale() instanceof GraduatedScaleLinear)
        gsl = (GraduatedScaleLinear)getMeter().getScale();
      if (gsl != null && getMeter().getBackground() != null) {
        Rectangle r = new Rectangle(getMeter().getBackground().getBoundingRect());
        gsl.setStartPoint(new Point(r.x + r.width/2, r.y + 19*r.height/20));
        gsl.setEndPoint(new Point(r.x + r.width/2, r.y + r.height/20));
        if (getMeter().getIndicator() != null)
          getMeter().getIndicator().setBoundingRect(gsl.getBoundingRect());
      }
    }

    // Keep the indicator value and the label in sync
    protected void partnerUpdate(JGoObject partner, int hint, int prevInt, Object prevVal) {
      if (hint == AbstractIndicator.ChangedValue && partner == getMeter().getIndicator() && getLabel() != null) {
        myFormatter.setMaximumFractionDigits(1);
        String s = myFormatter.format(getValue());
        getLabel().setText(s);
      } else if (hint == JGoText.ChangedText && partner == getLabel() && getLabel() != null) {
        String s = getLabel().getText();
        try {
          myFormatter.setMaximumFractionDigits(1);
          double v = myFormatter.parse(s).doubleValue();
          setValue(v);
        } catch (ParseException ex) {
          getLabel().setText((String)prevVal);
        }
      }
    }

    /**
     * Gets or sets the current indicated value of the Tank.
     * <p>
     * Directly checks and changes Meter's AbstractIndicator's Value.
     */
    public double getValue() {
      if (getMeter() == null)
        return 0;
      if (getMeter().getIndicator() != null)
        return getMeter().getIndicator().getValue();
      else
        return 0;
    }
    public void setValue(double value) {
      if (getMeter().getIndicator() != null)
        getMeter().getIndicator().setValue(value);
    }

    public double getMinimum() {
      if (getMeter() == null)
        return 0;
      if (getMeter().getScale() != null)
        return getMeter().getScale().getMinimum();
      else
        return 0;
    }
    public void setMinimum(double value) {
      if (getMeter().getScale() != null)
        getMeter().getScale().setMinimum(value);
    }

    public double getMaximum() {
      if (getMeter().getScale() != null)
        return getMeter().getScale().getMaximum();
      else
        return 0;
    }
    public void setMaximum(double value) {
      if (getMeter().getScale() != null)
        getMeter().getScale().setMaximum(value);
    }

    // let as much fluid flow out of this tank through
    // pipes connected at this tank's output port,
    // as can go in one unit of time
    public void drain() {
      if (getValue() <= getMinimum())
        return;
      double total = 0;
      JGoPort port = getOutputPort();
      JGoListPosition pos = port.getFirstLinkPos();
      while (pos != null) {
        JGoLink l = port.getLinkAtPos(pos);
        pos = port.getNextLinkPos(pos);

        Pipe p = null;
        if (l instanceof Pipe) p = (Pipe)l;
        if (p != null) {
          total += p.MaxFlow();
        }
      }
      if (total > 0) {
        double available = Math.min(total, getValue()-getMinimum());
        pos = port.getFirstLinkPos();
        while (pos != null) {
          JGoLink l = port.getLinkAtPos(pos);
          pos = port.getNextLinkPos(pos);

          Pipe p = null;
          if (l instanceof Pipe) p = (Pipe)l;
          if (p != null) {
            double w = p.MaxFlow();
            p.Flow(w);
            available -= w;
          }
        }
      }
    }

    private NumberFormat myFormatter = NumberFormat.getNumberInstance();

    public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup) {
      // Add <Tank> element
      if (svgDoc.JGoXMLOutputEnabled()) {
        DomElement Tank = svgDoc.createJGoClassElement(
            "com.nwoods.jgo.examples.instrumentdemo.Tank", jGoElementGroup);
      }
      // Have superclass add to the JGoObject group
      super.SVGWriteObject(svgDoc, jGoElementGroup);
    }

    public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc,
                                 DomElement svgElement,
                                 DomElement jGoChildElement) {
      if (jGoChildElement != null) {
        // This is a <Tank> element
        super.SVGReadObject(svgDoc, jGoDoc, svgElement,
                            jGoChildElement.getNextSiblingJGoClassElement());
      }
      return svgElement.getNextSibling();
    }
}


