/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 */

package com.nwoods.jgo.examples.instrumentdemo;

import java.awt.*;
import javax.swing.*;
import com.nwoods.jgo.*;
import com.nwoods.jgo.instruments.*;

  public class Thermometer extends Meter {
    public Thermometer() {}

    public void initialize() {
      super.initialize();
      myFScale = createFahrenheitScale();
      JGoListPosition pos = findObject(getIndicator());
      if (pos != null)
        insertObjectBefore(pos, myFScale);
      else
        addObjectAtTail(myFScale);
      myScaleLabel = createScaleLabel();
      pos = findObject(getBackground());
      if (pos != null)
        insertObjectAfter(pos, myScaleLabel);
      else
        addObjectAtTail(myScaleLabel);
      JGoEllipse bulb = new JGoEllipse();
      bulb.setPen(null);
      bulb.setBrush(JGoBrush.red);
      addObjectAtTail(bulb);
      myBulb = bulb;
      layoutChildren(null);
    }

    protected JGoObject createBackground() {
      JGoRectangle rect = new JGoRectangle();
      rect.setSelectable(false);
      rect.setBrush(JGoBrush.gray);
      return rect;
    }

    protected JGoText createScaleLabel() {
      JGoText t = new JGoText();
      t.setSelectable(false);
      t.setTransparent(true);
      t.setBold(true);
      t.setText("F        C");
      return t;
    }

    protected GraduatedScaleLinear createFahrenheitScale() {
      GraduatedScaleLinear scale = new GraduatedScaleLinear();
      scale.setMinimum(-40);
      scale.setMaximum(212);
      scale.setTickLengthRight(0);
      scale.setTickLengthLeft(10);
      scale.setTickUnit(5);
      scale.setTickMajorFrequency(4);
      scale.setLabelStyle(AbstractGraduatedScale.LabelStyleLeft);
      return scale;
    }

    protected AbstractGraduatedScale createScale() {
      GraduatedScaleLinear scale = new GraduatedScaleLinear();
      scale.setMinimum(-40);
      scale.setMaximum(100);
      scale.setTickLengthRight(10);
      scale.setTickLengthLeft(0);
      scale.setTickUnit(5);
      scale.setTickMajorFrequency(4);
      scale.setLabelStyle(AbstractGraduatedScale.LabelStyleRight);
      return scale;
    }

    protected AbstractIndicator createIndicator() {
      IndicatorBar bar = new IndicatorBar();
      bar.setActionEnabled(true);
      bar.setThickness(8);
      bar.setPen(null);
      return bar;
    }


    /**
     * Make sure all references in fields are updated to refer to the copied objects.
     */
    protected void copyChildren(JGoArea newgroup, JGoCopyEnvironment env) {
      super.copyChildren(newgroup, env);
      Thermometer newobj = (Thermometer)newgroup;
      newobj.myFScale = (GraduatedScaleLinear)env.get(myFScale);
      newobj.myScaleLabel = (JGoText)env.get(myScaleLabel);
      newobj.myBulb = (JGoEllipse)env.get(myBulb);
    }

    /**
     * If any part is removed from this group,
     * be sure to remove any references in local fields.
     */
    public JGoObject removeObjectAtPos(JGoListPosition pos) {
      JGoObject obj = super.removeObjectAtPos(pos);
      if (obj == myFScale)
        myFScale = null;
      else if (obj == myScaleLabel)
        myScaleLabel = null;
      else if (obj == myBulb)
        myBulb = null;
      return obj;
    }
    
    public void layoutChildren(JGoObject childchanged) { 
      if (isInitializing()) return;
      super.layoutChildren(childchanged);
      if (getBackground() != null) {
        Rectangle rect = new Rectangle(getBackground().getBoundingRect());

        int scalelabheight = 0;
        if (getScaleLabel() != null) {
          scalelabheight = getScaleLabel().getHeight();
          getScaleLabel().setSpotLocation(TopMiddle, getBackground(), TopMiddle);
        }
        int captionheight = 0;
        if (getLabel() != null) {
          captionheight = getLabel().getHeight();
          if (myBulb != null)
            captionheight += 12;
        }
        Point sp = new Point(rect.x + rect.width/2, rect.y + rect.height - captionheight - 5);
        Point ep = new Point(rect.x + rect.width/2, rect.y + scalelabheight + 5);

        if (getCelsiusScale() != null) {
          getCelsiusScale().setStartPoint(sp);
          getCelsiusScale().setEndPoint(ep);
          if (myBulb != null) {
            myBulb.setSize(14, 14);
            myBulb.setSpotLocation(TopCenter, sp.x, sp.y-1);
          }
        }
        if (getFahrenheitScale() != null) {
          getFahrenheitScale().setStartPoint(sp);
          getFahrenheitScale().setEndPoint(ep);
        }
      }
    }

    public GraduatedScaleLinear getFahrenheitScale() { return (GraduatedScaleLinear)myFScale; }

    public GraduatedScaleLinear getCelsiusScale() { return (GraduatedScaleLinear)getScale(); }

    public JGoText getScaleLabel() { return myScaleLabel; }

    /**
     * Gets the temperature of the thermometer, in degrees Celsius.
     */
    public double getCelsius() { return getIndicator().getValue(); }

    /**
     * Sets the temperature of the thermometer, in degrees Celsius.
     */
    public void setCelsius(double value) { getIndicator().setValue(value); }


    /**
     * Gets the temperature of the thermometer, in degrees Fahrenheit.
     */
    public double getFahrenheit() { return convertCtoF(getIndicator().getValue()); }

    /**
     * Sets the temperature of the thermometer, in degrees Fahrenheit.
     */
    public void setFahrenheit(double value) { getIndicator().setValue(convertFtoC(value)); }


    /**
     * Convert a temperature in degrees Fahrenheit to degrees Celsius.
     */
    public static double convertFtoC(double F){
      return (F-32)*5/9;
    }

    /**
     * Convert a temperature in degrees Celsius to degrees Fahrenheit.
     */
    public static double convertCtoF(double C){
      return (C*9/5)+32;
    }

    // Thermometer state
    private GraduatedScaleLinear myFScale;
    private JGoText myScaleLabel;
    private JGoEllipse myBulb;
  }
