/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.jtreeapp;

import java.applet.*;
import java.awt.*;
import java.awt.dnd.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.tree.*;
import com.nwoods.jgo.*;

/**
 * This is a Simple JGo applet/application that demonstrates the use of GoTreeNodes.
 * <p>
 * The user can do all the standard mouse manipulations (select, move, copy, link)
 * and can delete with the DELETE key,
 * cut/copy/paste (^X/^C/^V) if not an applet, and undo/redo (^Z/^Y).
 */
public class JTreeApp extends JApplet implements Runnable, JGoViewListener, JGoDocumentListener {
  public JTreeApp() {
    myView = new JGoView();
    getContentPane().add(myView, BorderLayout.CENTER);
  }
  
  public void start() {  // Applet starting
    new Thread(this).start();  // enable drag-and-drop from separate thread (JDC bug 4225247)
  }

  public void run() {
    myView.initializeDragDropHandling();
  }

  public static void main(String args[]) {  // run applet as an application
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

      JFrame mainFrame = new JFrame();
      // close the application when the main window closes
      mainFrame.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent event) {
          System.exit(0);
        }
      });
      mainFrame.setTitle("JTree App");
      mainFrame.setSize(700, 400);

      JTreeApp app = new JTreeApp();
      Container contentPane = mainFrame.getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(app, BorderLayout.CENTER);
      contentPane.validate();

      mainFrame.setVisible(true);

      app.init();
      app.start();

      javax.swing.JOptionPane.showMessageDialog(app,
        "Type Ctrl-T to add children to the currently selected JTree treenode;\n" +
        "Type Ctrl-Shift-T to remove the currently selected JTree treenode.\n\n" +
        "Each GoTreeNode supports a different combination of allowing source and/or destination links.\n\n" +
        "Only the rightmost GoTreeNode supports cross-level links.\n\n" +
        "CAVEAT: copying GoTreeNodes is not supported until you implement a policy for copying TreeModels,\n" +
        "    and undo/redo of JTree changes is not supported until you implement undo/redo support for TreeModels.\n",
        "Demo of GoTreeNodes displaying JTrees",
        javax.swing.JOptionPane.INFORMATION_MESSAGE);
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }
  }

  public void init() {  // App initialization--only here do we do anything JGo specific
    // the view has a black background and magenta selection handles
    myView.setPrimarySelectionColor(JGoBrush.ColorMagenta);
    myView.setIncludingNegativeCoords(true);
    myView.setHidingDisabledScrollbars(true);
    myView.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent evt) {
        handleKeyPressed(evt);
      }
    });
    // detect whenever a temporary link is added to the view so we can change its color
    myView.addViewListener(this);

    // initialize the document
    JGoDocument doc = myView.getDocument();
    // detect whenever a link is added to the document so we can modify its appearance
    doc.addDocumentListener(this);
    
    GoTreeNode treenode = new GoTreeNode();
    DefaultMutableTreeNode root = new DefaultMutableTreeNode("root");
    DefaultTreeModel model = new DefaultTreeModel(root);
    root.add(new DefaultMutableTreeNode("first"));
    root.add(new DefaultMutableTreeNode("second"));
    root.add(new DefaultMutableTreeNode("third"));
    root.add(new DefaultMutableTreeNode("fourth"));
    root.add(new DefaultMutableTreeNode("fifth"));
    root.add(new DefaultMutableTreeNode("sixth"));
    root.add(new DefaultMutableTreeNode("seventh"));
    root.add(new DefaultMutableTreeNode("eighth"));
    treenode.initialize(model, true, false, myView);
    treenode.setBoundingRect(new Point(50, 100), new Dimension(100, 150));
    doc.addObjectAtTail(treenode);

    treenode = new GoTreeNode();
    treenode.initialize(null, true, true, myView);
    treenode.setBoundingRect(new Point(250, 100), new Dimension(100, 150));
    doc.addObjectAtTail(treenode);

    treenode = new GoTreeNode();
    treenode.initialize(null, false, true, myView);
    treenode.setBoundingRect(new Point(450, 100), new Dimension(100, 150));
    doc.addObjectAtTail(treenode);
    treenode.setAllowingCrossLevelLinks(true);

    // enable undo & redo
    doc.setUndoManager(new JGoUndoManager());
  }

  // cut/copy/paste only works when the clipboard is available--
  // i.e. not normally when running as an applet
  void handleKeyPressed(KeyEvent evt) {
    int code = evt.getKeyCode();
    if (code == KeyEvent.VK_DELETE) {  // delete the currently selected objects
      myView.deleteSelection();
    } else if (code == KeyEvent.VK_HOME) {  // scroll so that the top-left-most object is at the top-left corner
      Rectangle docbounds = myView.getDocument().computeBounds();
      myView.setViewPosition(docbounds.x, docbounds.y);
    } else if (code == KeyEvent.VK_END) {  // scroll so that the bottom-right-most object is at the bottom-right corner
      Rectangle docbounds = myView.getDocument().computeBounds();
      Dimension viewsize = myView.getExtentSize();
      myView.setViewPosition(Math.max(docbounds.x, docbounds.x+docbounds.width-viewsize.width),
                             Math.max(docbounds.y, docbounds.y+docbounds.height-viewsize.height));
    } else if (evt.isControlDown() && code == KeyEvent.VK_A) {  // select all
      myView.selectAll();
    } else if (evt.isControlDown() && code == KeyEvent.VK_X) {  // cut
      myView.cut();
    } else if (evt.isControlDown() && code == KeyEvent.VK_C) {  // copy
      myView.copy();
    } else if (evt.isControlDown() && code == KeyEvent.VK_V) {  // paste
      myView.paste();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Z) {  // undo
      myView.getDocument().undo();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Y) {  // redo
      myView.getDocument().redo();
    } else if (evt.isControlDown() && code == KeyEvent.VK_T) {  // redo
      modifyTree(evt.getModifiers());
    }
  }

  // JGoViewListener:
  // Modify the temporary link created when drawing a new link
  // (the default black would not appear against the black background)
  public void viewChanged(JGoViewEvent evt) {
    if (evt.getHint() == JGoViewEvent.INSERTED) {
      if (evt.getJGoObject() instanceof JGoLink) {
        JGoLink link = (JGoLink)evt.getJGoObject();
        link.setPen(JGoPen.make(JGoPen.SOLID, 2, JGoBrush.ColorGray));
        link.setBrush(JGoBrush.gray);
        link.setArrowHeads(false, true);
      }
    }
  }

  // JGoDocumentListener:
  // If a link is added to the document, make sure it has a new color, is wider than
  // normal, and has an arrowhead at its "to" end
  public void documentChanged(JGoDocumentEvent evt) {
    if (evt.getHint() == JGoDocumentEvent.INSERTED && (evt.getJGoObject() instanceof JGoLink)) {
      JGoLink link = (JGoLink)evt.getJGoObject();
      link.setPen(JGoPen.make(JGoPen.SOLID, 2, link.getBrush().getColor()));  // pen controls the line and the outline of the arrowhead
      link.setArrowHeads(false, true);  // have an arrowhead at the "to" end
    }
  }

  void modifyTree(int modifiers) {
    JGoObject primsel = myView.getSelection().getPrimarySelection();
    if (primsel == null) return;
    if (!(primsel instanceof GoTreeNode)) return;

    GoTreeNode gonode = (GoTreeNode)primsel;
    JTree tree = gonode.getTree().getJTree(myView);
    DefaultTreeModel model = (DefaultTreeModel)tree.getModel();
    Object selobj = tree.getLastSelectedPathComponent();
    if (selobj != null && selobj instanceof MutableTreeNode) {
      MutableTreeNode tn = (MutableTreeNode)selobj;
      if ((modifiers & InputEvent.SHIFT_MASK) != 0 &&
        selobj != model.getRoot()) {
        // delete treenode, except for the root
        int[] indices = new int[1];
        Object[] objects = new Object[1];
        TreeNode parent = tn.getParent();
        indices[0] = parent.getIndex(tn);
        objects[0] = tn;
        tn.removeFromParent();
        model.nodesWereRemoved(parent, indices, objects);
      } else {
        int numchildren = tn.getChildCount();
        DefaultMutableTreeNode newtn = new DefaultMutableTreeNode(Integer.toString(TNcounter++));
        tn.insert(newtn, numchildren);
        int[] indices = new int[1];
        indices[0] = numchildren;
        model.nodesWereInserted(tn, indices);
      }
    }
  }

  private JGoView myView;
  private int myNodeCounter = 0;
  private int TNcounter = 23;
}
