/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.layoutdemo;

import com.nwoods.jgo.*;
import java.awt.*;

public class BasicLayoutNode extends JGoNode
{
  /**
   * A newly constructed BasicLayoutNode is not usable until you've
   * called initialize().
   */
  public BasicLayoutNode()
  {
    super();
  }

  protected void copyChildren(JGoArea newarea, JGoCopyEnvironment env)
  {
    BasicLayoutNode newobj = (BasicLayoutNode)newarea;

    newobj.numPorts = numPorts;
    newobj.myPorts = new BasicLayoutPort[numPorts];

    super.copyChildren(newarea, env);

    newobj.myEllipse = (JGoEllipse)env.get(myEllipse);
    newobj.myLabel = (JGoText)env.get(myLabel);

    Point c = newobj.myEllipse.getSpotLocation(Center);
    for (int i = 0; i < numPorts; i++) {
      BasicLayoutPort op = myPorts[i];
      BasicLayoutPort np = (BasicLayoutPort)env.get(op);
      newobj.myPorts[i] = np;
      np.myEllipse = newobj.myEllipse;
    }
  }

  /**
   * Keep the parts of a BasicLayoutNode positioned relative to each other
   * by setting their locations using some of the standard spots of
   * a JGoObject.
   * <p>
   * By default the label will be positioned at the bottom of the node,
   * above the ellipse.  To change this to be below the ellipse, at
   * the bottom of the node, change the myLabel.setSpotLocation() call.
   */
  public void layoutChildren()
  {
    if (myEllipse == null) return;
    if (myLabel != null) {
      // put the label above the node
      myLabel.setSpotLocation(TopCenter, myEllipse, BottomCenter);
      // put the label below the node
      //myLabel.setSpotLocation(TopCenter, myEllipse, BottomCenter);
    }
    if (myPorts != null){
      Point c = myEllipse.getSpotLocation(Center);
      for (int i = 0; i < numPorts; i++)
        if(horizontal)
          myPorts[i].setSpotLocation(Center, c.x - (getStdSize() / 4) * (numPorts - 1) + (getStdSize() / 2) * i, c.y);
        else
          myPorts[i].setSpotLocation(Center, c.x, c.y - (getStdSize() / 4) * (numPorts - 1) + (getStdSize() / 2) * i);
    }
  }

  /**
   * Specify the location of the node; the size is constant.
   * If labeltext is null, do not create a label.
   */
  public void initialize(Point loc, String labeltext, int ports, boolean horiz)
  {
    horizontal = horiz;
    numPorts = ports;
    if(numPorts <= 0)
      numPorts = 1;
    if(horizontal)
      setSize((myStdSize / 2) * (numPorts + 1), myStdSize);
    else
      setSize(myStdSize, (myStdSize / 2) * (numPorts + 1));

    // the user can move this node around
    setDraggable(true);
    // the user cannot resize this node
    setResizable(false);

    // create the bigger circle/ellipse around and behind the port
    myEllipse = new JGoEllipse(getTopLeft(), getSize());
    myEllipse.setSelectable(false);
    myEllipse.setDraggable(false);

    // can't setLocation until myEllipse exists
    myEllipse.setSpotLocation(JGoObject.Center, loc);

    // if there is a string, create a label with a transparent
    // background that is centered
    if (labeltext != null) {
      myLabel = new JGoText(labeltext);
      myLabel.setSelectable(false);
      myLabel.setDraggable(false);
      myLabel.setAlignment(JGoText.ALIGN_CENTER);
      myLabel.setTransparent(true);
    }

    // create a Port, which knows how to make sure
    // connected JGoLinks have a reasonable end point
    myPorts = new BasicLayoutPort[numPorts];
    for(int i = 0; i < numPorts; i++){
      myPorts[i] = new BasicLayoutPort();
      myPorts[i].myEllipse = myEllipse;
      myPorts[i].setSize(16, 16);
    }

    // add all the children to the area
    addObjectAtHead(myEllipse);
    if (myLabel != null)
      addObjectAtTail(myLabel);
     for(int i = 0; i < numPorts; i++)
        addObjectAtTail(myPorts[i]);

    // now position the label and port appropriately
    // relative to the ellipse
    layoutChildren();
    setBrush(JGoBrush.white);
    setColor(Color.red);
  }


  // Convenience methods: control the ellipse's pen and brush
  public JGoPen getPen()
  {
    return myEllipse.getPen();
  }

  public void setPen(JGoPen p)
  {
    myEllipse.setPen(p);
  }

  public JGoBrush getBrush()
  {
    return myEllipse.getBrush();
  }

  public void setBrush(JGoBrush b)
  {
    myEllipse.setBrush(b);
  }

  public Color getColor()
  {
    return getPen().getColor();
  }

  public void setColor(Color c)
  {
    setPen(JGoPen.makeStockPen(c));
  }

  public void colorChange()
  {
    Color c = getColor();
    if (c == Color.red)
      setColor(Color.green);
    else if (c == Color.green)
      setColor(Color.blue);
    else if (c == Color.blue)
      setColor(Color.red);
  }

  public boolean isHorizontal()
  {
    return horizontal;
  }

  public int getNumPorts()
  {
    return numPorts;
  }

  static int getStdSize()
  {
    return myStdSize;
  }

  static void setStdSize(int size)
  {
    myStdSize = size;
  }

  static int myStdSize = 40;

  // get access to the parts of the node
  public JGoObject getEllipse() { return myEllipse; }
  public JGoText getLabel() { return myLabel; }
  public BasicLayoutPort[] getPorts() { return myPorts; }


    // State
  protected JGoEllipse myEllipse = null;
  protected JGoText myLabel = null;
  protected BasicLayoutPort[] myPorts = null;
  protected int numPorts = 0;
  protected boolean horizontal = true;
}