/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.layoutdemo;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.*;
import java.util.LinkedList;
import com.nwoods.jgo.*;


// Implement the Demo model data structure container.
// This class is responsible for loading and storing documents to files,
// and for creating activities (nodes) and flows (links).
//
// DemoDocument, for this example app, has just a few properties:
// Name, Location, Orthogonal Flows and Link Pen.
// The latter two may appear to belong to a view instead of being
// part of a document, but here these attributes can be conveniently
// stored persistently.
public class DemoDocument extends JGoDocument
{
  public DemoDocument()
  {
  }

  public String getName()
  {
    return myName;
  }

  public void setName(String newname)
  {
    String oldName = myName;
    if (!oldName.equals(newname)) {
      myName = newname;
      fireUpdate(NameChanged, 0, null, 0, oldName);
    }
  }

  public String getLocation()
  {
    return myLocation;
  }

  public void setLocation(String newloc)
  {
    String oldLocation = myLocation;
    if (!oldLocation.equals(newloc)) {
      myLocation = newloc;
      fireUpdate(LocationChanged, 0, null, 0, oldLocation);
    }
  }

  // creating a new basicLayoutNode.
  public BasicLayoutNode newNode(Point p, String s, int numPorts, boolean horiz)
  {
    BasicLayoutNode snode = new BasicLayoutNode();
    snode.initialize(p, s, numPorts, horiz);
    addObjectAtTail(snode);
    return snode;
  }

  // creating a new link between layout nodes.
  public JGoLink newLink(JGoPort from, JGoPort to)
  {
    if(from.getParent() == to.getParent())
      return null;
    JGoLink ll = new JGoLink(from, to);
    ll.setPen(getLinkPen());
    getLinksLayer().addObjectAtTail(ll);
    ll.setArrowHeads(false, isShowArrows());
    return ll;
  }


  public JGoPen getLinkPen()
  {
    return myPen;
  }

  public void setLinkPen(JGoPen p)
  {
    if (!myPen.equals(p)) {
      myPen = p;
      // now update all links
      JGoListPosition pos = getFirstObjectPos();
      while (pos != null) {
        JGoObject obj = getObjectAtPos(pos);
        // only consider top-level objects
        pos = getNextObjectPosAtTop(pos);
        if (obj instanceof JGoLink) {
          JGoLink link = (JGoLink)obj;
          link.setPen(p);
        }
      }
    }
  }



  // For this sample application, read and write Demo documents
  // as files using an ascii format.  The default serialization is
  // not used, because any changes to the JGoLink, BasicLayoutNode,
  // or BasicLayoutPort would cause previously saved files to become
  // unaccessable.

  public static DemoDocument open(Component parent, String defaultLocation)
  {
    JFileChooser chooser = new JFileChooser();
    if ((defaultLocation != null) && (!defaultLocation.equals(""))) {
      File currentFile = new File(defaultLocation);
      chooser.setCurrentDirectory(currentFile);
    }
    else
      chooser.setCurrentDirectory(null);
    int returnVal = chooser.showOpenDialog(null);
    DemoDocument doc = new DemoDocument();
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      String loc = chooser.getSelectedFile().getAbsolutePath();
      RandomAccessFile fstream = null;
      try {
        fstream = new RandomAccessFile(loc, "r");
        //changing the open procedure...
        int version = fstream.readInt();
        doc.myVersion = version;
        String title = fstream.readLine();
        doc.setName(title);
        doc.setLocation(loc);
        boolean b = fstream.readBoolean();
        if(!b)
          doc.toggleArrows();
        boolean more = fstream.readBoolean();
        while(more) {
          boolean isNode = fstream.readBoolean();
          if(isNode)
            doc.readNode(fstream);
          else
            doc.readLink(fstream);
          more = fstream.readBoolean();
        }
        doc.changed = false;
        return doc;
      } catch (IOException x) {
        JOptionPane.showMessageDialog(null,
          x,
          "Open Document Error",
          javax.swing.JOptionPane.ERROR_MESSAGE);
        return null;
      } catch (Exception x) {
        JOptionPane.showMessageDialog(null,
          x,
          "Loading Document Exception",
          javax.swing.JOptionPane.ERROR_MESSAGE);
        return null;
      } finally {
        try { if (fstream != null) fstream.close(); } catch (Exception x) {}
      }
    } else {
      return null;
    }
  }

  public void save()
  {
    if (getLocation().equals("")) {
      saveAs();
    } else {
      store();
    }
  }

  public void store()
  {
    boolean oldchanged = changed;
    if (!getLocation().equals("")) {
      changed = false;
      RandomAccessFile fstream = null;
      try {
        fstream = new RandomAccessFile(getLocation(), "rw");
        fstream.setLength(0);
        fstream.writeInt(myVersion);
        fstream.writeBytes(getName() + "\r\n");
        fstream.writeBoolean(isShowArrows());
        //now write out all of the nodes and links, starting with the nodes
        JGoListPosition pos = getFirstObjectPos();
        LinkedList links = new LinkedList();
        while (pos != null) {
          JGoObject obj = getObjectAtPos(pos);
          pos = getNextObjectPosAtTop(pos);
          if (obj instanceof BasicLayoutNode)
          {
            BasicLayoutNode anode = (BasicLayoutNode) obj;
            obj = null;
            fstream.writeBoolean(true); //more objects to read
            writeNode(anode, fstream);
          }
          else if (obj instanceof JGoLink)
          {
            //add obj to linked list so that it can be added after the nodes.
            links.addLast(obj);
          }
        }
        while(links.size() > 0)
        {
          JGoLink ll = (JGoLink)links.removeLast();
          //write ll to file
          fstream.writeBoolean(true); //more objects to read
          writeLink(ll, fstream);
        }
        fstream.writeBoolean(false); //no more object to read
      } catch (IOException x) {
        JOptionPane.showMessageDialog(null,
          x,
          "Save Document Error",
          javax.swing.JOptionPane.ERROR_MESSAGE);
        changed = oldchanged;
      } finally {
        try { if (fstream != null) fstream.close(); } catch (Exception x) {changed = oldchanged;}
      }
    }
  }

  public void saveAs()
  {
    JFileChooser chooser = new JFileChooser();
    chooser.setCurrentDirectory(null);
    int returnVal = chooser.showSaveDialog(null);
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      String name = chooser.getSelectedFile().getName();
      this.setName(name);
      String loc = chooser.getSelectedFile().getAbsolutePath();
      setLocation(loc);
      store();
    }
  }

  private void readNode(RandomAccessFile fstream) throws IOException
  {
    if(myVersion == 1)
    {
      boolean hasLabel = fstream.readBoolean();
      String label = null;
      if(hasLabel) {
        label = fstream.readLine();
      }
      char color = fstream.readChar();
      int numPorts = fstream.readInt();
      boolean hor = fstream.readBoolean();
      int x = fstream.readInt();
      int y = fstream.readInt();
      Point c = new Point(x, y);
      BasicLayoutNode anode = newNode(c, label, numPorts, hor);
      if(color == 'g')
        anode.colorChange();
      else if(color == 'b')  {
        anode.colorChange();
        anode.colorChange();
      }
    }
  }

  private void readLink(RandomAccessFile fstream) throws IOException
  {
    if(myVersion == 1)
    {
      int numPoints = fstream.readInt();
      Point[] points = new Point[numPoints];
      for(int i = 0; i < numPoints; i++)
      {
        int x = fstream.readInt();
        int y = fstream.readInt();
        points[i] = new Point(x, y);
      }
      BasicLayoutPort fromPort = pickPort(points[0]);
      BasicLayoutPort toPort = pickPort(points[numPoints - 1]);
      JGoLink ll = newLink(fromPort, toPort);
      if(numPoints > 2)
      {
        points[0] = ll.getStartPoint();
        points[numPoints -1] = ll.getEndPoint();
        ll.removeAllPoints();
        for(int i = 0; i < numPoints; i++)
          ll.addPoint(points[i]);
      }
    }
  }

  private void writeNode(BasicLayoutNode anode, RandomAccessFile fstream) throws IOException
  {
    if(myVersion == 1)
    {
      fstream.writeBoolean(true);  //is a node.
      if(anode.getLabel() != null)
      {
        fstream.writeBoolean(true);  //there is a label
        fstream.writeBytes(anode.getLabel().getText() + "\r\n");
      }
      else fstream.writeBoolean(false); //no label
      if(anode.getColor() == Color.green)
        fstream.writeChar('g');
      else if(anode.getColor() == Color.blue)
        fstream.writeChar('b');
      else
        fstream.writeChar('r');
      fstream.writeInt(anode.getNumPorts());
      fstream.writeBoolean(anode.isHorizontal());
      Point center = anode.getSpotLocation(JGoObject.Center);
      fstream.writeInt(center.x);
      fstream.writeInt(center.y);
    }
  }

  private void writeLink(JGoLink ll, RandomAccessFile fstream) throws IOException
  {
    if(myVersion == 1)
    {
      fstream.writeBoolean(false); //not a node (i.e. is a link)
      int numPoints = ll.getNumPoints();
      fstream.writeInt(numPoints);
      Point p = ll.getFromPort().getSpotLocation(JGoObject.Center);
      fstream.writeInt(p.x);
      fstream.writeInt(p.y);
      for(int i = 1; i < numPoints - 1; i++)
      {
        p = ll.getPoint(i);
        fstream.writeInt(p.x);
        fstream.writeInt(p.y);
      }
      p = ll.getToPort().getSpotLocation(JGoObject.Center);
      fstream.writeInt(p.x);
      fstream.writeInt(p.y);
    }
  }

  public BasicLayoutPort pickPort(Point pointToCheck)
  {
    JGoListPosition pos = this.getLastObjectPos();
    while (pos != null) {
      JGoObject obj = this.getObjectAtPos(pos);
      pos = this.getPrevObjectPos(pos);

      if (obj.isVisible() &&
          obj.isPointInObj(pointToCheck)) {
          if (obj instanceof JGoArea) {
            // handle inside area
            JGoObject child = ((JGoArea)obj).pickObject(pointToCheck, false);
            if (child != null)
              obj = child;
          }
          if (obj instanceof BasicLayoutPort)
            return (BasicLayoutPort)obj;
      }
    }
    return null;
  }

  public boolean isShowArrows()
  {
    return showArrows;
  }

  public void toggleArrows()
  {
    showArrows = !showArrows;
    JGoListPosition pos = getFirstObjectPos();
    while(pos != null) {
      JGoObject obj = getObjectAtPos(pos);
      if(obj instanceof JGoLink)
        ((JGoLink)obj).setArrowHeads(false, isShowArrows());
      pos = getNextObjectPosAtTop(pos);
    }
  }

  public boolean isChanged()
  {
    return changed;
  }

  public void setChanged(boolean b)
  {
    changed = b;
  }

  // Event hints
  public static final int NameChanged = JGoDocumentEvent.LAST + 1;
  public static final int LocationChanged = JGoDocumentEvent.LAST + 2;

  // State
  private JGoPen myPen = JGoPen.make(JGoPen.SOLID, 2, Color.black);

  private String myName = "";
  private String myLocation = "";
  private static boolean showArrows = true;
  private int myVersion = 1;

  private boolean changed = false;
}