/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.layoutdemo;

import java.awt.*;
import java.awt.event.*;
import java.awt.print.*;
import java.awt.dnd.*;
import java.awt.datatransfer.*;
import java.awt.geom.*;
import javax.swing.*;
import java.util.*;
import com.nwoods.jgo.*;

// Provide a view of a DemoDocument
// Implement various command handlers
public class DemoView extends JGoView implements JGoViewListener
{
  public DemoView()
  {
    super();
  }

  public DemoView(JGoDocument doc)
  {
    super(doc);
  }

  public void initialize(LayoutDemo app, JInternalFrame frame)
  {
    myApp = app;
    myInternalFrame = frame;
    addViewListener(this);
    setGridWidth(10);
    setGridHeight(10);
    setSnapMove(JGoView.SnapJump);
    showGrid();
  }

  // creating a DemoView without specifying a document in the constructor
  // needs to create a DemoDocument rather than a generic JGoDocument
  public JGoDocument createDefaultModel()
  {
    return new DemoDocument();
  }

  // convenience method--the return value is a DemoDocument instead
  // of a JGoDocument
  DemoDocument getDoc()
  {
    return (DemoDocument)getDocument();
  }

  LayoutDemo getApp() { return myApp; }
  JInternalFrame getInternalFrame() { return myInternalFrame; }


  // handle DELETE key as well as the page up/down keys
  public void onKeyEvent(KeyEvent evt)
  {
    int t = evt.getKeyCode();
    if (t == KeyEvent.VK_DELETE) {
      deleteSelection();
    } else {
      super.onKeyEvent(evt);
    }
  }


  // implement JGoViewListener
  // just need to keep the actions enabled appropriately
  // depending on the selection
  public void viewChanged(JGoViewEvent e)
  {
    // if the selection changed, maybe some commands need to
    // be disabled or re-enabled
    switch(e.getHint()) {
      case JGoViewEvent.UPDATE_ALL:
      case JGoViewEvent.SELECTION_GAINED:
      case JGoViewEvent.SELECTION_LOST:
      case JGoViewEvent.SCALE_CHANGED:
        AppAction.updateAllActions();
        break;
    }
  }

  // implement JGoDocumentListener
  // here we just need to keep the title bar up-to-date
  public void documentChanged(JGoDocumentEvent evt)
  {
    if (evt.getHint() == DemoDocument.NameChanged &&
        getInternalFrame() != null) {
      updateTitle();
    }
    if (evt.getHint() == JGoDocumentEvent.CHANGED || evt.getHint() == JGoDocumentEvent.ARRANGED ||
        evt.getHint() == JGoDocumentEvent.INSERTED || evt.getHint() == JGoDocumentEvent.REMOVED)
      getDoc().setChanged(true);
    super.documentChanged(evt);
  }
  
  // have the title bar for the internal frame include the name
  // of the document
  public void updateTitle()
  {
    if (getInternalFrame() != null) {
      String title = getDoc().getName();
      getInternalFrame().setTitle(title);
      getInternalFrame().repaint();
    }
  }

  
  // override newLink to allow the document to decide what
  // kind of link to create
  public void newLink(JGoPort from, JGoPort to)
  {
    JGoLink l = getDoc().newLink(from, to);
    fireUpdate(JGoViewEvent.LINK_CREATED, 0, l);
    getDoc().endTransaction("new Link");
  }


  // implement commands for creating activities in the Demo
  // let DemoDocument do the work

  public void insertNode()
  {
    insertNode(null, null, 1, true);
  }
  
  public void insertNode(Point loc)
  {
    insertNode(loc, null, 1, true);
  }
  
  public void insertNode(String s)
  {
    insertNode(null, s, 1, true);
  }
  
  public void insertNode(Point loc, String s)
  {
    insertNode(loc, s, 1, true);
  }
  
  public void insertNode(Point loc, String s, int numPorts, boolean horizontal)
  {
    if (loc == null) loc = getDefaultLocation();
    if (loc == null) loc = new Point(100, 70);
    BasicLayoutNode snode = getDoc().newNode(loc, s, numPorts, horizontal);
  }
  
  // the default place to put stuff if not dragged there
  public Point getDefaultLocation()
  {
    // to avoid constantly putting things in the same place,
    // keep shifting the default location
    if (myDefaultLocation != null) {
      myDefaultLocation.x += 10;
      myDefaultLocation.y += 10;
    }
    return myDefaultLocation;
  }

  public void setDefaultLocation(Point loc)
  {
    myDefaultLocation = loc;
  }


  // toggle the grid appearance
  void showGrid()
  {
    int style = getGridStyle();
    if (style == JGoView.GridInvisible) {
      style = JGoView.GridDot;
      setGridPen(JGoPen.black);
    } else {
      style = JGoView.GridInvisible;
    }
    setGridStyle(style);
  }

  // printing support
  public Rectangle2D.Double getPrintPageRect(Graphics2D g2, PageFormat pf)
  {
    // leave some space at the bottom for a footer
    return new Rectangle2D.Double(pf.getImageableX(), pf.getImageableY(),
                                  pf.getImageableWidth(), pf.getImageableHeight() - 20);
  }

  public void printDecoration(Graphics2D g2, PageFormat pf, int hpnum, int vpnum)
  {
    // draw corners around the getPrintPageRect area
    super.printDecoration(g2, pf, hpnum, vpnum);

    // print the n,m page number in the footer
    String msg = Integer.toString(hpnum);
    msg += ", ";
    msg += Integer.toString(vpnum);

    Paint oldpaint = g2.getPaint();
    g2.setPaint(Color.black);
    Font oldfont = g2.getFont();
    g2.setFont(new Font(JGoText.getDefaultFontFaceName(), Font.PLAIN, 10));
    g2.drawString(msg, (int)(pf.getImageableX() + pf.getImageableWidth()/2),
                       (int)(pf.getImageableY() + pf.getImageableHeight() - 10));
    g2.setPaint(oldpaint);
    g2.setFont(oldfont);
  }

  public double getPrintScale(Graphics2D g2, PageFormat pf)
  {
    return getScale();
  }


  void zoomIn()
  {
    double newscale = Math.rint(getScale() / 0.9f * 100f) / 100f;
    setScale(newscale);
  }

  void zoomOut()
  {
    double newscale = Math.rint(getScale() * 0.9f * 100f) / 100f;
    setScale(newscale);
  }

  void zoomNormal()
  {
    setScale(1.0d);
  }

  void zoomToFit()
  {
    double newscale = 1;
    if (!getDocument().isEmpty()){
      double extentWidth = getExtentSize().width;
      double printWidth = getPrintDocumentSize().width;
      double extentHeight = getExtentSize().height;
      double printHeight = getPrintDocumentSize().height;
      newscale = Math.min((extentWidth / printWidth),(extentHeight / printHeight));
    }
    newscale *= getScale();
    if (newscale > 1)
      newscale = 1;
    setScale(newscale);
    setViewPosition(0, 0);
  }

  
  public boolean doMouseDblClick(int modifiers, Point dc, Point vc)
  {
    JGoObject obj = pickDocObject(dc,false);
    if (obj != null)
    {
      //get top level object
      while(obj.getParent() != null)
        obj = obj.getParent();
      ((BasicLayoutNode)obj).colorChange();
    }
    else insertNode(dc);
    return super.doMouseDblClick(modifiers, dc, vc);
  }
  
  public boolean isChanged()
  {
    return getDoc().isChanged();
  }
  
  // State
  protected Point myDefaultLocation = new Point(80, 80);
  protected LayoutDemo myApp = null;
  protected JInternalFrame myInternalFrame = null;
}
