/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.layoutdemo;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.util.HashMap;
import com.nwoods.jgo.*;
import com.nwoods.jgo.layout.JGoNetwork;
import java.beans.*;

/**
 * This app gives an example of the layout algorithms.
 */
public class LayoutDemo extends JApplet implements Runnable
{
  public LayoutDemo()
  {
    initMenus();

    Container contentPane = getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(getDesktop(), BorderLayout.CENTER);
    contentPane.add(getStatusArea(), BorderLayout.SOUTH);
    initStatusArea();
    contentPane.validate();
  }


  void initMenus()
  {
    //==============================================================
    // Define all the command actions and setup the menus
    //==============================================================
    JMenuItem item = null;

    filemenu.setText("File");
    filemenu.setMnemonic('F');

    AppAction FileNewAction = new AppAction("New", this) {
      public void actionPerformed(ActionEvent e) { newDemo(); }
      public boolean canAct() { return true; } };  // doesn't depend on a view

    item = filemenu.add(FileNewAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N,Event.CTRL_MASK));
    item.setMnemonic('N');

    AppAction FileOpenAction = new AppAction("Open", this) {
      public void actionPerformed(ActionEvent e) { openDemo(); }
      public boolean canAct() { return true; } };  // doesn't depend on a view

    item = filemenu.add(FileOpenAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O,Event.CTRL_MASK));
    item.setMnemonic('O');

    AppAction FileCloseAction = new AppAction("Close", this) {
      public void actionPerformed(ActionEvent e) { closeDemo(); } };

    item = filemenu.add(FileCloseAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4,Event.CTRL_MASK));
    item.setMnemonic('C');

    AppAction FileSaveAction = new AppAction("Save", this) {
      public void actionPerformed(ActionEvent e) { saveDemo(); } };

    item = filemenu.add(FileSaveAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,Event.CTRL_MASK));
    item.setMnemonic('S');

    AppAction FileSaveAsAction = new AppAction("Save As", this) {
      public void actionPerformed(ActionEvent e) { saveAsDemo(); } };

    item = filemenu.add(FileSaveAsAction);
    item.setMnemonic('A');

    filemenu.addSeparator();

    AppAction PrintAction = new AppAction("Print", this) {
      public void actionPerformed(ActionEvent e) { getView().print(); } };

    item = filemenu.add(PrintAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,Event.CTRL_MASK));
    item.setMnemonic('P');

    filemenu.addSeparator();

    AppAction ExitAction = new AppAction("Exit", this) {
      public void actionPerformed(ActionEvent e) { exit(); }
      public boolean canAct() { return true; } };  // doesn't depend on a view

    item = filemenu.add(ExitAction);
    item.setMnemonic('x');

    mainMenuBar.add(filemenu);


    editmenu.setText("Edit");
    editmenu.setMnemonic('E');

    AppAction CutAction = new AppAction("Cut", this) {
      public void actionPerformed(ActionEvent e) { getView().cut(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    item = editmenu.add(CutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,Event.CTRL_MASK));
    item.setMnemonic('t');

    AppAction CopyAction = new AppAction("Copy", this) {
      public void actionPerformed(ActionEvent e) { getView().copy(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    item = editmenu.add(CopyAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,Event.CTRL_MASK));
    item.setMnemonic('C');

    AppAction PasteAction = new AppAction("Paste", this) {
      public void actionPerformed(ActionEvent e) { getView().paste(); } };

    item = editmenu.add(PasteAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V,Event.CTRL_MASK));
    item.setMnemonic('P');

    AppAction DeleteAction = new AppAction("Delete", this) {
      public void actionPerformed(ActionEvent e) { getView().deleteSelection(); }
      public boolean canAct() { return super.canAct() && !getView().getSelection().isEmpty(); } };

    item = editmenu.add(DeleteAction);
    item.setMnemonic('D');

    AppAction SelectAllAction = new AppAction("Select All", this) {
      public void actionPerformed(ActionEvent e) { getView().selectAll(); } };

    item = editmenu.add(SelectAllAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A,Event.CTRL_MASK));
    item.setMnemonic('l');

    mainMenuBar.add(editmenu);


    viewmenu.setText("View");
    viewmenu.setMnemonic('V');

    AppAction ZoomNormalAction = new AppAction("Normal Zoom", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomNormal(); } };

    item = viewmenu.add(ZoomNormalAction);
    item.setMnemonic('N');

    AppAction ZoomInAction = new AppAction("Zoom In", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomIn(); }
      public boolean canAct() { return super.canAct() && (getView().getScale() < 8.0f); } };

    item = viewmenu.add(ZoomInAction);
    item.setMnemonic('I');

    AppAction ZoomOutAction = new AppAction("Zoom Out", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomOut(); }
      public boolean canAct() { return super.canAct() && (getView().getScale() > 0.13f); } };

    item = viewmenu.add(ZoomOutAction);
    item.setMnemonic('O');

    AppAction ZoomToFitAction = new AppAction("Zoom To Fit", this) {
      public void actionPerformed(ActionEvent e) { getView().zoomToFit(); } };

    item = viewmenu.add(ZoomToFitAction);
    item.setMnemonic('Z');

    viewmenu.addSeparator();

    AppAction GridAction = new AppAction("Toggle Grid", this) {
      public void actionPerformed(ActionEvent e) { getView().showGrid(); } };

    item = viewmenu.add(GridAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G,Event.CTRL_MASK));
    item.setMnemonic('G');

    AppAction ToggleAction = new AppAction("Toggle Arrow Heads", this) {
      public void actionPerformed(ActionEvent e) { getView().getDoc().toggleArrows(); } };

    item = viewmenu.add(ToggleAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T,Event.CTRL_MASK));
    item.setMnemonic('T');

    mainMenuBar.add(viewmenu);


    insertmenu.setText("Insert");
    insertmenu.setMnemonic('I');

    AppAction InsertNodeAction = new AppAction("Basic Node", this) {
      public void actionPerformed(ActionEvent e) { nodeAction(); } };

    item = insertmenu.add(InsertNodeAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B,Event.CTRL_MASK));
    item.setMnemonic('B');

    mainMenuBar.add(insertmenu);

    layoutmenu.setText("Layout");
    layoutmenu.setMnemonic('L');

    AppAction RandomLayoutAction = new AppAction("Random Layout", this) {
      public void actionPerformed(ActionEvent e) { randomAction(); } };

    item = layoutmenu.add(RandomLayoutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R,Event.CTRL_MASK));
    item.setMnemonic('R');

    AppAction ForceLayoutAction = new AppAction("Force-Directed Layout", this) {
      public void actionPerformed(ActionEvent e) { forceAction(); } };

    item = layoutmenu.add(ForceLayoutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F,Event.CTRL_MASK));
    item.setMnemonic('F');


    AppAction LayerLayoutAction = new AppAction("Layered Digraph Layout", this) {
      public void actionPerformed(ActionEvent e) { layerAction(); } };

    item = layoutmenu.add(LayerLayoutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L,Event.CTRL_MASK));
    item.setMnemonic('L');

    AppAction TreeLayoutAction = new AppAction("Tree Layout", this) {
      public void actionPerformed(ActionEvent e) { treeAction(); } };

    item = layoutmenu.add(TreeLayoutAction);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T,Event.CTRL_MASK));
    item.setMnemonic('T');

    mainMenuBar.add(layoutmenu);

    helpmenu.setText("Help");
    helpmenu.setMnemonic('H');

    AppAction AboutAction = new AppAction("About", this) {
      public void actionPerformed(ActionEvent e) { showAbout(); }
      public boolean canAct() { return true; } };  // doesn't depend on a view

    item = helpmenu.add(AboutAction);
    item.setMnemonic('A');

    mainMenuBar.add(helpmenu);

    setJMenuBar(mainMenuBar);
  }


  public void init()
  {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }

    AppAction.updateAllActions();
  }

  public void start()
  {
    // enable drag-and-drop from separate thread
    new Thread(this).start();
  }

  public void run() {
    newDemo();
  }

  static public void main(String args[])
  {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

      final JFrame mainFrame = new JFrame();
      final LayoutDemo app = new LayoutDemo();

      // close the application when the main window closes
      mainFrame.addWindowListener(new WindowAdapter() {
        public void windowClosing(java.awt.event.WindowEvent event) {
          Object object = event.getSource();
          if (object == mainFrame)
            app.exit();
        }
      });

      mainFrame.setTitle("Layout Demo");
      Dimension screensize = Toolkit.getDefaultToolkit().getScreenSize();
      mainFrame.setBounds(0, 0, screensize.width, screensize.height);

      Container contentPane = mainFrame.getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(app, BorderLayout.CENTER);
      contentPane.validate();

      mainFrame.setVisible(true);

      app.init();
      app.start();
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }
  }


  void exit()
  {
    System.exit(0);
  }

  void showAbout()
  {
    javax.swing.JOptionPane.showMessageDialog(this,
          "LayoutDemo, an example of the JGo auto layout code.\n\n" +
          "                      Northwoods Software\n                         www.nwoods.com",
          "       About LayoutDemo",
          javax.swing.JOptionPane.INFORMATION_MESSAGE);
  }


  public void createFrame(DemoDocument doc)
  {
    final DemoView view = new DemoView(doc);
    final JInternalFrame frame = new JInternalFrame(doc.getName(), true, true, true);
    view.initialize(this, frame);

    // keep track of the "current" view, even if it doesn't have focus
    // try to give focus to a view when it becomes activated
    // enable/disable all the command actions appropriately for the view
    frame.addVetoableChangeListener(new CloseListener(this, frame));
    frame.addInternalFrameListener(new InternalFrameListener()
      {
        public void internalFrameActivated(InternalFrameEvent e)
          {
            myCurrentView = view;
            view.requestFocus();
            AppAction.updateAllActions();
          }
        public void internalFrameDeactivated(InternalFrameEvent e) {}
        public void internalFrameOpened(InternalFrameEvent e) {view.zoomToFit(); view.showGrid(); view.setSnapMove(JGoView.NoSnap);}
        public void internalFrameClosing(InternalFrameEvent e) {}
        public void internalFrameClosed(InternalFrameEvent e) {}
        public void internalFrameIconified(InternalFrameEvent e) {}
        public void internalFrameDeiconified(InternalFrameEvent e) {}
      }
    );

    Container contentPane = frame.getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(view);

    frame.setSize(700, 500);
    getDesktop().add(frame);
    frame.show();
    view.initializeDragDropHandling();
  }

  void newDemo()
  {
    DemoDocument doc = new DemoDocument();
    String t = "Untitled" + Integer.toString(myDocCount++);
    doc.setName(t);
    createFrame(doc);
  }

  void openDemo()
  {
    String defaultLoc = null;
    DemoView view = getCurrentView();
    if (view != null) {
      DemoDocument doc = view.getDoc();
      defaultLoc = doc.getLocation();
    }
    DemoDocument doc = DemoDocument.open(this, defaultLoc);
    if (doc != null)
      createFrame(doc);
  }

  void closeDemo()
  {
    if (getCurrentView() != null) {
      JInternalFrame frame = getCurrentView().getInternalFrame();
      if (frame != null) {
        if (getCurrentView().isChanged()) {
          String msg = "Save Changes to " + getCurrentView().getDoc().getName();
          int choice = javax.swing.JOptionPane.showConfirmDialog(frame, msg, "Northwoods Software LayoutDemo",
                      javax.swing.JOptionPane.YES_NO_CANCEL_OPTION, javax.swing.JOptionPane.QUESTION_MESSAGE);
          if (choice == javax.swing.JOptionPane.YES_OPTION)
            saveDemo();
          if((myCurrentView.isChanged() && choice == javax.swing.JOptionPane.YES_OPTION) ||
                      choice == javax.swing.JOptionPane.CANCEL_OPTION)
            //save was cancelled, or cancel was chosen.  Don't close.
            return;
        }
        getDesktop().getDesktopManager().closeFrame(frame);
        myCurrentView = null;
        AppAction.updateAllActions();
      }
    }
  }

  void saveDemo()
  {
    if (getCurrentView() != null) {
      DemoDocument doc = getCurrentView().getDoc();
      doc.save();
    }
  }

  void saveAsDemo()
  {
    if (getCurrentView() != null) {
      DemoDocument doc = getCurrentView().getDoc();
      doc.saveAs();
    }
  }

  DemoDocument findDemoDocument(String path)
  {
    Object val = myMap.get(path);
    if (val != null && val instanceof DemoDocument)
      return (DemoDocument)val;
    else
      return null;
  }


  DemoView getCurrentView()
  {
    return myCurrentView;
  }

  void randomAction()
  {
    setStatus("");
    DemoDocument doc = getCurrentView().getDoc();
    Rectangle r = getCurrentView().getViewRect();
    SimpleRAL s = new SimpleRAL(doc, (r.x + 100), ((r.width - r.x) - 100), (r.y + 100), ((r.height - r.y) - 100));
    s.performLayout();
    setStatus("Random Layout done");
  }

  void forceAction()
  {
    setStatus("");
    ForceDialog f = new ForceDialog(getCurrentView().getFrame(), "Force-Directed Settings", true, getCurrentView(), this);
    f.setVisible(true);
  }

  void layerAction()
  {
    setStatus("");
    LayerDialog l = new LayerDialog(getCurrentView().getFrame(), "Layered-Digraph Settings", true, getCurrentView(), this);
    l.setVisible(true);
  }

  void treeAction()
  {
    setStatus("");
    TreeDialog t = new TreeDialog(getCurrentView().getFrame(), "Tree Layout Settings", true, getCurrentView());
    t.setVisible(true);
  }


  void nodeAction()
  {
    NodeDialog n = new NodeDialog(getCurrentView().getFrame(), "Node Settings", true, getCurrentView());
    n.setVisible(true);
  }

  JDesktopPane getDesktop() { return myDesktop; }

  public JPanel getStatusArea() {
    return myStatusArea;
  }

  public void setStatus(String s) {
    if (s.equals(""))
      s = " ";
    myStatusLabel.setText(s);
    myStatusLabel.paintImmediately(0, 0, myStatusLabel.getWidth(), myStatusLabel.getHeight());
  }

  protected void initStatusArea()
  {
    getStatusArea().setMinimumSize(new Dimension(10, 10));
    getStatusArea().setBorder(BorderFactory.createEtchedBorder());
    getStatusArea().setLayout(new BorderLayout());
    getStatusArea().add(myStatusLabel, BorderLayout.CENTER);
    setStatus("Ready");
  }

  // State
  protected HashMap myMap = new HashMap();
  protected DemoView myCurrentView = null;
  protected JDesktopPane myDesktop = new JDesktopPane();

  protected JMenuBar mainMenuBar = new JMenuBar();
  protected JMenu filemenu = new JMenu();
  protected JMenu editmenu = new JMenu();
  protected JMenu viewmenu = new JMenu();
  protected JMenu insertmenu = new JMenu();
  protected JMenu layoutmenu = new JMenu();
  protected JMenu helpmenu = new JMenu();

  protected JPanel myStatusArea = new JPanel();
  protected JLabel myStatusLabel = new JLabel();

  private int myDocCount = 1;


  class CloseListener implements VetoableChangeListener {
    CloseListener(LayoutDemo app, JInternalFrame frame)
    {
      myApp = app;
      myFrame = frame;
    }

    public void vetoableChange(PropertyChangeEvent e) throws PropertyVetoException {
      String name = e.getPropertyName();
      if(name.equals(JInternalFrame.IS_CLOSED_PROPERTY)) {
        Component internalFrame = (Component)e.getSource();
        Boolean oldvalue = (Boolean)e.getOldValue(),
                newvalue = (Boolean)e.getNewValue();
        if(oldvalue == Boolean.FALSE && newvalue == Boolean.TRUE) {
          msg = "Save Changes to " + myApp.getCurrentView().getDoc().getName();
          if (getCurrentView().isChanged()) {
            int choice = javax.swing.JOptionPane.showConfirmDialog(myFrame, msg, "Northwoods Software LayoutDemo",
                javax.swing.JOptionPane.YES_NO_CANCEL_OPTION, javax.swing.JOptionPane.QUESTION_MESSAGE);
            if (choice == javax.swing.JOptionPane.YES_OPTION)
              saveDemo();
            if((myCurrentView.isChanged() && choice == javax.swing.JOptionPane.YES_OPTION) || choice == javax.swing.JOptionPane.CANCEL_OPTION)
            //save was cancelled, or cancel was chosen.  Don't close.
              throw new PropertyVetoException("close cancelled", e);
            else {  //user either saved or chose "no".  either way, it's ok to close window.
              myApp.myCurrentView = null;
              AppAction.updateAllActions();
            }
          }
          else { //no changes.  just close it.
            myApp.myCurrentView = null;
            AppAction.updateAllActions();
          }
        }
      }
    }
    private String msg = "";
    private LayoutDemo myApp = null;
    private JInternalFrame myFrame = null;
  }

}