/*
 *  Copyright (c) Northwoods Software Corporation, 1998-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.svg;

import java.awt.Rectangle;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Font;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.Color;
import javax.swing.JComponent;
import com.nwoods.jgo.*;
import java.net.URL;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.util.Map;
import java.util.WeakHashMap;
import java.lang.Float;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import java.io.Reader;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.FileInputStream;
import org.xml.sax.InputSource;
import javax.xml.parsers.*;

import org.apache.batik.transcoder.TranscoderException;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.TranscodingHints;
import org.apache.batik.transcoder.image.ImageTranscoder;
import org.apache.batik.transcoder.ErrorHandler;

public class SVGImage extends JGoImage {
    /**
     * Load an SVG image given a filename. This
     * method will return only after the image has been
     * completely loaded.
     * <p>
     * For version 4.0, this also depends on the value of
     * getDefaultBase().  If that URL is non-null, it is combined
     * with the given filename to specify a URL from which to
     * load an Image.
     *
     * @param filename the image file
     * @return true if loading was successful
     */
    public boolean loadImage(String filename) {
        return super.loadImage(filename, true);
    }

    /**
     * Load an image given a URL. This
     * method will return only after the image has been
     * completely loaded.
     *
     * @param url the URL for the image
     * @return true if loading was successful
     */
    public boolean loadImage(URL url) {
        return super.loadImage(url, true);
    }

    /**
     * Return an Image for the given filename path.
     */
    public Image getImage(String path) {
      return null;
    }

    /**
     * Return an Image for the given URL.
     */
    public Image getImage(URL url) {
      return null;
    }

    private Image rasterizeImage() {
        Image img = null;
        if (getFilename() != null) {
            try {
                if(!getFilename().equals(strOldFile)) {
                    input = new TranscoderInput("file://" + getFilename());
                    strOldFile = getFilename();
                }
                img = createBufferedImage();
            }
            catch (Exception e) {
              e.printStackTrace();
            }
        }
        if (this.getURL() != null) {
            try {
                if(!getURL().equals(urlOld)) {
                    input = new TranscoderInput(getURL().toString());
                    urlOld = getURL();
                }
                img = createBufferedImage();
            }
            catch (Exception e) {
            }
        }
        return img;
    }

    /**
     * Sets the default dimension of the svg viewport.
     * If a viewport is ot specified by height and width attributes in the
     * <svg> element, use the viewport dimensions defined here.
     * The viewport defines the height and width in pixels of the image to be
     * displayed.
     * <p>
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().

     * @param vp the viewport dimensions
     */
    public void setDefaultViewport(Dimension vp) {
        Dimension oldvp = defaultViewport;
        if (!oldvp.equals(vp)) {
            defaultViewport = vp;
            bUpdateImage = true; // set to update image in paint
            // Signal state change to support undo/redo
            update(ViewportChanged, 0, oldvp);
        }
    }

    /**
     * Returns the image that represents the SVG document.
     */
    public BufferedImage createBufferedImage() throws TranscoderException {
        //    Rasterizer r = new Rasterizer();
        r.setTranscodingHints( (Map) hints);
        r.transcode(input, null);
        return img;
    }

    /**
     * Sets the width of the image to rasterize.
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param width the image width
     */
    public void setImageWidth(float width) {
        TranscodingHints oldHints = hints;
        Float oldWidth = (Float)oldHints.get(ImageTranscoder.KEY_WIDTH);
        if ((oldWidth == null) || (oldWidth.floatValue() != width)) {
            hints.put(ImageTranscoder.KEY_WIDTH, new Float(width));
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // a call to update image in Paint method
        }
    }

    /**
     * Sets the height of the image to rasterize.
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param width the image height
     */
    public void setImageHeight(float height) {
        TranscodingHints oldHints = hints;
        Float oldHeight = (Float)oldHints.get(ImageTranscoder.KEY_HEIGHT);
        if ((oldHeight == null) || (oldHeight.floatValue() != height)) {
            hints.put(ImageTranscoder.KEY_HEIGHT, new Float(height));
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // a call to update image in Paint Method
        }
    }

    /**
     * Sets the preferred language to use. SVG documents can provide text in
     * multiple languages, this method lets you control which language to use
     * if possible. e.g. "en" for english or "fr" for french.
     * <p>
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param language the preferred language to use
     */
    public void setLanguages(String language) {
        TranscodingHints oldHints = hints;
        String oldLanguage = (String)oldHints.get(ImageTranscoder.KEY_LANGUAGE);
        if ((oldLanguage == null) || (!oldLanguage.equals(language))) {
            hints.put(ImageTranscoder.KEY_LANGUAGE, language);
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // signal to redraw Image in Paint
        }
    }

    /**
     * Sets the unit conversion factor to the specified value. This method
     * lets you choose how units such as 'em' are converted. e.g. 0.26458 is
     * 96dpi (the default) or 0.3528 is 72dpi.
     * <p>
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param px2mm the pixel to millimeter convertion factor.
     */
    public void setPixelToMMFactor(float px2mm) {
        TranscodingHints oldHints = hints;
        Float oldpx2mm = (Float)oldHints.get(ImageTranscoder.KEY_PIXEL_TO_MM);
        if ((oldpx2mm == null) || (oldpx2mm.floatValue() != px2mm)) {
            hints.put(ImageTranscoder.KEY_PIXEL_TO_MM, new Float(px2mm));
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // signal to redraw image in Paint
        }
    }

    /**
     * Sets the uri of the user stylesheet. The user stylesheet can be used to
     * override styles.
     *
     * @param uri the uri of the user stylesheet
     */
    public void setUserStyleSheetURI(String uri) {
        TranscodingHints oldHints = hints;
        String olduri = (String)oldHints.get(ImageTranscoder.KEY_USER_STYLESHEET_URI);
        if ((olduri == null) || (!olduri.equals(uri))) {
            hints.put(ImageTranscoder.KEY_USER_STYLESHEET_URI, uri);
            bUpdateImage = true; // set to update image in paint
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
        }
    }

    /**
     * Sets whether or not the XML parser used to parse SVG document should be
     * validating or not, depending on the specified parameter. For futher
     * details about how media work, see the
     * <a href="http://www.w3.org/TR/CSS2/media.html";>Media types in the CSS2
     * specification</a>.
     * <p>
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param b true means the XML parser will validate its input
     */
    public void setXMLParserValidating(boolean b) {
        TranscodingHints oldHints = hints;
        Boolean oldbool = (Boolean)oldHints.get(ImageTranscoder.KEY_XML_PARSER_VALIDATING);
        if ((oldbool == null) || (oldbool.booleanValue() != b)) {
            hints.put(ImageTranscoder.KEY_XML_PARSER_VALIDATING, new Boolean(b));
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // call to update image in paint
        }
    }

    /**
     * Sets the media to rasterize. The medium should be separated by
     * comma. e.g. "screen", "print" or "screen, print"
     * <p>
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param media the media to use
     */
    public void setMedia(String media) {
        TranscodingHints oldHints = hints;
        String oldMedia = (String)oldHints.get(ImageTranscoder.KEY_MEDIA);
        if ((oldMedia == null) || (!oldMedia.equals(media))) {
            hints.put(ImageTranscoder.KEY_MEDIA, media);
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // set to update image in paint
        }
    }

    /**
     * Sets the alternate stylesheet to use. For futher details, you can have
     * a look at the <a href="http://www.w3.org/TR/xml-stylesheet/";>Associating
     * Style Sheets with XML documents</a>.
     * <p>
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     *
     * @param alternateStylesheet the alternate stylesheet to use if possible
     */
    public void setAlternateStylesheet(String alternateStylesheet) {
        TranscodingHints oldHints = hints;
        String oldAlternateStylesheet = (String)oldHints.get(ImageTranscoder.KEY_ALTERNATE_STYLESHEET);
        if ((oldAlternateStylesheet == null) || (!oldAlternateStylesheet.equals(alternateStylesheet))) {
            hints.put(ImageTranscoder.KEY_ALTERNATE_STYLESHEET, alternateStylesheet);
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // set to update image in paint
        }
    }

    /**
     * Sets the Paint to use for the background of the image.
     * Note that setting this property after loadImage() has been called
     * will cause the SVG image to be reloaded (a potentially time-consuming
     * operation).  We recommend setting all SVGImage properties prior to
     * calling loadImage().
     * <p>
     * Note that setBackgroundColor() does not work in Batik versions
     * prior to Batik 1.5 beta 2 (see bug #6707 in the batik database).
     * If a background color is needed using a Batik version prior to this,
     * a JGoArea can be used to combine an SVGImage in the foreground with
     * a JGoRectangle of a specified color in the background.
     *
     * @param p the paint to use for the background
     */
    public void setBackgroundColor(Color p) {
        TranscodingHints oldHints = hints;
        Color oldColor = (Color)oldHints.get(ImageTranscoder.KEY_BACKGROUND_COLOR);
        if ((oldColor == null) || (!oldColor.equals(p))) {
            hints.put(ImageTranscoder.KEY_BACKGROUND_COLOR, p);
            // Signal state change to support undo/redo
            update(HintChanged, 0, oldHints);
            bUpdateImage = true; // set to update image in paint
        }
    }

    public JGoObject copyObject(JGoCopyEnvironment env) {
        SVGImage newobj = (SVGImage)super.copyObject(env);
        newobj.hints = hints;
        newobj.defaultViewport = defaultViewport;
        return newobj;
    }

    public void copyNewValueForRedo(JGoDocumentChangedEdit e) {
        // Copy the current state before doing the undo do it can
        // be reset in a future redo operation
        switch (e.getFlags()) {
            case HintChanged: {
                e.setNewValue(hints);
                return;
            }
            case ViewportChanged: {
                e.setNewValue(defaultViewport);
                return;
            }
            default:
                super.copyNewValueForRedo(e);
                return;
        }
    }

    public void changeValue(JGoDocumentChangedEdit e, boolean undo) {
        // Actually perform the undo or redo operation
        switch (e.getFlags()) {
            case HintChanged:
                hints.put(ImageTranscoder.KEY_ALTERNATE_STYLESHEET, e.getValue(undo));
                return;
            case ViewportChanged: {
                setDefaultViewport((Dimension)(e.getValue(undo)));
            }
            default:
                super.changeValue(e, undo);
                return;
        }
    }

    public void SVGWriteObject(DomDoc svgDoc, DomElement jGoElementGroup) {
        // Add SVGImage element
        if (svgDoc.JGoXMLOutputEnabled()) {
            DomElement jSVGImage = svgDoc.createJGoClassElement(
            "com.nwoods.jgo.examples.svg.SVGImage", jGoElementGroup);
            if (hints.get(ImageTranscoder.KEY_ALTERNATE_STYLESHEET) != null)
                jSVGImage.setAttribute("alternatestylesheet",
                (String) hints.get(ImageTranscoder.
                KEY_ALTERNATE_STYLESHEET));
            Color bgcolor = (Color) hints.get(ImageTranscoder.KEY_BACKGROUND_COLOR);
            if (bgcolor != null)
                jSVGImage.setAttribute("backgroundcolor",
                Integer.toString(bgcolor.getRGB()));
            if (hints.get(ImageTranscoder.KEY_HEIGHT) != null)
                jSVGImage.setAttribute("imageheight",
                ( (Float) (hints.get(ImageTranscoder.KEY_HEIGHT))).
                toString());
            if (hints.get(ImageTranscoder.KEY_WIDTH) != null)
                jSVGImage.setAttribute("imagewidth",
                ( (Float) (hints.get(ImageTranscoder.KEY_WIDTH))).
                toString());
            if (hints.get(ImageTranscoder.KEY_LANGUAGE) != null)
                jSVGImage.setAttribute("languages",
                (String) hints.get(ImageTranscoder.KEY_LANGUAGE));
            if (hints.get(ImageTranscoder.KEY_MEDIA) != null)
                jSVGImage.setAttribute("media",
                (String) hints.get(ImageTranscoder.KEY_MEDIA));
            if (hints.get(ImageTranscoder.KEY_PIXEL_TO_MM) != null)
                jSVGImage.setAttribute("pixeltomm",
                ( (Float) (hints.get(ImageTranscoder.
                KEY_PIXEL_TO_MM))).toString());
            if (hints.get(ImageTranscoder.KEY_USER_STYLESHEET_URI) != null)
                jSVGImage.setAttribute("userstylesheet",
                (String) hints.get(ImageTranscoder.
                KEY_USER_STYLESHEET_URI));
            if (hints.get(ImageTranscoder.KEY_XML_PARSER_VALIDATING) != null)
                jSVGImage.setAttribute("parservalidating",
                ( (Boolean) (hints.get(ImageTranscoder.
                KEY_XML_PARSER_VALIDATING))).toString());
            if (defaultViewport != null) {
                jSVGImage.setAttribute("defviewportheight",
                Integer.toString(defaultViewport.height));
                jSVGImage.setAttribute("defviewportwidth",
                Integer.toString(defaultViewport.width));
            }
        }

        // Have superclass add to the JGoObject group
        super.SVGWriteObject(svgDoc, jGoElementGroup);
    }

    public DomNode SVGReadObject(DomDoc svgDoc, JGoDocument jGoDoc, DomElement svgElement, DomElement jGoChildElement) {
        if (jGoChildElement != null) {
            // This is a SVGImage element
            if (jGoChildElement.getAttribute("alternatestylesheet").length() > 0)
                hints.put(ImageTranscoder.KEY_ALTERNATE_STYLESHEET, jGoChildElement.getAttribute("alternatestylesheet"));
            if (jGoChildElement.getAttribute("backgroundcolor").length() > 0) {
                int rgb = Integer.parseInt(jGoChildElement.getAttribute("backgroundcolor"));
                hints.put(ImageTranscoder.KEY_BACKGROUND_COLOR, new Color(rgb));
            }
            if (jGoChildElement.getAttribute("imageheight").length() > 0)
                hints.put(ImageTranscoder.KEY_HEIGHT, new Float(jGoChildElement.getAttribute("imageheight")));
            if (jGoChildElement.getAttribute("imagewidth").length() > 0)
                hints.put(ImageTranscoder.KEY_WIDTH, new Float(jGoChildElement.getAttribute("imagewidth")));
            if (jGoChildElement.getAttribute("languages").length() > 0)
                hints.put(ImageTranscoder.KEY_LANGUAGE, jGoChildElement.getAttribute("languages"));
            if (jGoChildElement.getAttribute("media").length() > 0)
                hints.put(ImageTranscoder.KEY_MEDIA, jGoChildElement.getAttribute("media"));
            if (jGoChildElement.getAttribute("pixeltomm").length() > 0)
                hints.put(ImageTranscoder.KEY_PIXEL_TO_MM, new Float(jGoChildElement.getAttribute("pixeltomm")));
            if (jGoChildElement.getAttribute("userstylesheet").length() > 0)
                hints.put(ImageTranscoder.KEY_USER_STYLESHEET_URI, jGoChildElement.getAttribute("userstylesheet"));
            if (jGoChildElement.getAttribute("parservalidating").length() > 0)
                hints.put(ImageTranscoder.KEY_XML_PARSER_VALIDATING, new Boolean(jGoChildElement.getAttribute("parservalidating")));
            if (jGoChildElement.getAttribute("defviewportheight").length() > 0) {
                int h = Integer.parseInt(jGoChildElement.getAttribute("defviewportheight"));
                int w = Integer.parseInt(jGoChildElement.getAttribute("defviewportwidth"));
                defaultViewport = new Dimension(w, h);
            }
            super.SVGReadObject(svgDoc, jGoDoc, svgElement, jGoChildElement.getNextSiblingJGoClassElement());
        }
        return svgElement.getNextSibling();
    }

    //==============================================================
    // Display
    //==============================================================

    /**
     * Draws this image on the given graphics context. If no
     * image has been loaded yet, nothing is drawn. If only part
     * of the image has been loaded, the part that is loaded is
     * drawn.
     *
     * @param g the graphics context on which to draw.
     * @param view the view we're drawing in
     */
    public void paint(Graphics2D g, JGoView view) {
      // If we don't have an image, create one
      if (getImage() == null) {
        if (getURL() != null)
          loadImage(getURL(), false);
        else if (getFilename() != null)
          loadImage(getFilename(), false);
        if ( (getFilename() != null) || (getURL() != null)) {
          double currentScale = view.getScale();
          int neww = (int) (getWidth() * currentScale);
          int newh = (int) (getHeight() * currentScale);
          hints.put(ImageTranscoder.KEY_WIDTH, new Float(neww));
          hints.put(ImageTranscoder.KEY_HEIGHT, new Float(newh));

//          System.out.println("Initial load");
          lookupAndLoadImage(newh, neww);
        }
      }
      else {
        // We have an image.  Is it the right size?
        // Maintain previous aspect ratio
        BufferedImage img = (BufferedImage)getImage();
        double currentScale = view.getScale();
        int oldw = img.getWidth();
        int oldh = img.getHeight();
        int neww = (int)(getWidth() * currentScale);
        int newh = (int)(getHeight() * currentScale);
        if (oldw <= 0) oldw = 1;
        float ratio = oldh/((float)oldw);
        if (neww <= 0) neww = 1;
        float newratio = newh/((float)neww);
        if (ratio != 0) {
            if (ratio < newratio) {
                neww = (int) Math.rint(newh / ratio);
            }
            else {
                newh = (int) Math.rint(neww * ratio);
            }
        }
        if ((oldw != neww) || (oldh != newh) || bUpdateImage) {
          // Image dimension or scale factor has changed - need to rasterize again
          hints.put(ImageTranscoder.KEY_WIDTH, new Float(neww));
          hints.put(ImageTranscoder.KEY_HEIGHT, new Float(newh));
//          System.out.println("Image lookup");
          lookupAndLoadImage(newh, neww);
        }
      }
      if (getImage() != null) {
        Rectangle rect = getBoundingRect();
        if (getTransparentColor() == null) {
          g.drawImage(getImage(), rect.x, rect.y,
                      rect.width, rect.height, this);
        }
        else {
          g.drawImage(getImage(), rect.x, rect.y,
                      rect.width, rect.height,
                      getTransparentColor(),
                      this);
        }
      }
    }

    private void lookupAndLoadImage(int h, int w) {
      String key = "";
      if (getURL() != null)
        key = getURL().toString();
      if (getFilename() != null)
        key = getFilename();
      key = key + "_" + Integer.toString(h) + "_" + Integer.toString(w);
      Image img = (Image)myImageMap.get(key);
      if (img == null) {
//        System.out.println("Rasterize");
        img = rasterizeImage();
        if (img != null)
          myImageMap.put(key, img);
      }
      loadImage(img, true);
      bUpdateImage = false;
    }

    private Dimension defaultViewport = new Dimension(400, 400);

    /**
     * The transcoder hints.
     */
    protected TranscodingHints hints = new TranscodingHints();

    /**
     * The transcoder input.
     */
    transient protected TranscoderInput input;

    /**
     * The image that represents the SVG document.
     */
    transient protected BufferedImage img;

    /**
     * The last rendered view scale. If this value is different from the current view scale, re-draw the image
     * Also the last rendered bounding rect. If the width or height is different, also re-draw the image.
     */
//    transient protected double dLastRenderedViewScale = 0.0;
//    transient protected Rectangle rectLastBounding = new Rectangle(0,0,0,0);

    /**
     * The last URL of the image
     */
    transient protected URL urlOld = null;

    /**
     * The last Filename of the image
     */
    transient protected String strOldFile = "";

    /**
     * Runtime Rasterizer, placed here to reduce excess garbage being created
     */
    transient protected Rasterizer r = new Rasterizer();

    /**
     * A flag which indicates something has changed about the image (geom, etc...), and requires it's image to be
     * redrawn
     */
    transient protected boolean bUpdateImage = true;

    public static final int HintChanged = JGoDocumentEvent.LAST + 10000;
    public static final int ViewportChanged = JGoDocumentEvent.LAST + 10001;
    private static WeakHashMap myImageMap = new WeakHashMap();

    //===============================================================
    // Inner classes
    //===============================================================

    /**
     * An image transcoder that stores the resulting image.
     */
    protected class Rasterizer
    extends ImageTranscoder {

        protected Rasterizer() {
            MyImageTranscoderUserAgent agent = new MyImageTranscoderUserAgent();
            agent.setDefaultViewport(SVGImage.this.defaultViewport);
            this.userAgent = agent;
        }

        public BufferedImage createImage(int w, int h) {

            return new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
        }

        public void writeImage(BufferedImage img, TranscoderOutput output) throws
        TranscoderException {
            SVGImage.this.img = img;
        }

        protected void transcode(Document document, String uri, TranscoderOutput output) throws TranscoderException {
            super.transcode(document, uri, output);
            input.setDocument(document);
        }

        // Use the following declaration for Batik V1.5
//      protected class MyImageTranscoderUserAgent extends org.apache.batik.transcoder.SVGAbstractTranscoder.SVGAbstractTranscoderUserAgent {
        // Use the following declaration for Batik V1.1.1
        protected class MyImageTranscoderUserAgent extends ImageTranscoder.ImageTranscoderUserAgent {

            public java.awt.geom.Dimension2D getViewportSize() {
                return m_defaultViewport;
            }
            public void setDefaultViewport(Dimension vp) {
                m_defaultViewport = vp;
            }
            // Set default viewport size to be the same as is created by the
            // default batik user agent (400 x 400).
            protected Dimension m_defaultViewport = new Dimension(400, 400);
        }

    }
}
