/*
 *  Copyright (c) Northwoods Software Corporation, 2000-2008. All Rights
 *  Reserved.
 *
 *  Restricted Rights: Use, duplication, or disclosure by the U.S.
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
 *  52.227-14 Alt. III, as applicable.
 *
 */

package com.nwoods.jgo.examples.updatedemo;

import java.applet.*;
import java.awt.*;
import java.awt.dnd.*;
import java.awt.event.*;
import javax.swing.*;
import com.nwoods.jgo.*;

/**
 * This is a Simple JGo applet/application that demonstrates the use
 * of BasicNodes, both rectangular and elliptical, with the label in the middle.
 * <p>
 * The user can do all the standard mouse manipulations (select, move, copy, link)
 * and can insert new ones with the INSERT key, delete with the DELETE key,
 * cut/copy/paste (^X/^C/^V) if not an applet, and undo/redo (^Z/^Y).
 */
public class UpdateDemo extends JApplet implements Runnable, JGoViewListener, JGoDocumentListener {
  public UpdateDemo() {
    JPanel logPanel = new JPanel();
    logPanel.setLayout(new BorderLayout());
    myTextArea = new JTextArea(50, 30);
    myScrollPane = new JScrollPane(myTextArea);
    logPanel.add(myScrollPane, BorderLayout.CENTER);
    JPanel logControlsPanel = new JPanel();
    JButton clearButton = new JButton();
    clearButton.setText("Clear");
    clearButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        UpdateDemo.App.ClearLog();
      }
    });
    logControlsPanel.add(clearButton);
    logPanel.add(logControlsPanel, BorderLayout.SOUTH);

    JSplitPane splitPane = new JSplitPane();
    myView = new UpdateView();
    splitPane.add(myView, JSplitPane.LEFT);
    splitPane.add(logPanel, JSplitPane.RIGHT);
    getContentPane().add(splitPane, BorderLayout.CENTER);
  }
  
  public void start() {  // Applet starting
    new Thread(this).start();  // enable drag-and-drop from separate thread (JDC bug 4225247)
  }

  public void run() {
    myView.initializeDragDropHandling();
  }

  public static void main(String args[]) {  // run applet as an application
    try {
      JFrame mainFrame = new JFrame();
      // close the application when the main window closes
      mainFrame.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent event) {
          System.exit(0);
        }
      });
      mainFrame.setTitle("UpdateDemo -- Double-click in background to add new nodes; ^Z undo; ^Y redo");
      mainFrame.setSize(800, 400);

      UpdateDemo app = new UpdateDemo();
      UpdateDemo.App = app;
      Container contentPane = mainFrame.getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(app, BorderLayout.CENTER);
      contentPane.validate();

      mainFrame.setVisible(true);

      app.init();
      app.start();
    } catch (Throwable t) {
      System.err.println(t);
      t.printStackTrace();
      System.exit(1);
    }
  }

  public static UpdateDemo App = null;

  public static void AddLogMessage(String s) {
    UpdateDemo.App.myTextArea.append("\n" + s);
  }

  public static void ClearLog() {
    UpdateDemo.App.myTextArea.setText("");
  }

  public void init() {  // App initialization--only here do we do anything JGo specific
    // the view has a black background and magenta selection handles
    myView.setBackground(JGoBrush.ColorBlack);
    myView.setPrimarySelectionColor(JGoBrush.ColorMagenta);
    myView.setSecondarySelectionColor(myView.getPrimarySelectionColor());
    myView.setIncludingNegativeCoords(true);
    myView.setHidingDisabledScrollbars(true);
    myView.setDragsRealtime(false);
    myView.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent evt) {
        handleKeyPressed(evt);
      }
    });
    // detect whenever a temporary link is added to the view so we can change its color
    myView.addViewListener(this);

    // initialize the document
    JGoDocument doc = myView.getDocument();
    // detect whenever a link is added to the document so we can modify its appearance
    doc.addDocumentListener(this);
    
    // create a couple of nodes and a link between them
    JGoBasicNode b1 = insertNode(new Point(50, 50), true);
    b1.setText("first");
    JGoBasicNode b2 = insertNode(new Point(130, 50), false);
    b2.setText("second");
    doc.addObjectAtTail(new JGoLink(b1.getPort(), b2.getPort()));
    
    // enable undo & redo
    doc.setUndoManager(new UndoMgr());
  }

  // cut/copy/paste only works when the clipboard is available--
  // i.e. not normally when running as an applet
  void handleKeyPressed(KeyEvent evt) {
    int code = evt.getKeyCode();
    if (code == KeyEvent.VK_DELETE) {  // delete the currently selected objects
      myView.deleteSelection();
    } else if (code == KeyEvent.VK_INSERT) {   // insert a new node in the middle of the window
      Rectangle rectangle = myView.getViewRect();
      insertNode(new Point(rectangle.x + rectangle.width / 2, rectangle.y + rectangle.height / 2),
                 (myNodeCounter%2 == 0));
    } else if (code == KeyEvent.VK_HOME) {  // scroll so that the top-left-most object is at the top-left corner
      Rectangle docbounds = myView.getDocument().computeBounds();
      myView.setViewPosition(docbounds.x, docbounds.y);
    } else if (code == KeyEvent.VK_END) {  // scroll so that the bottom-right-most object is at the bottom-right corner
      Rectangle docbounds = myView.getDocument().computeBounds();
      Dimension viewsize = myView.getExtentSize();
      myView.setViewPosition(Math.max(docbounds.x, docbounds.x+docbounds.width-viewsize.width),
                             Math.max(docbounds.y, docbounds.y+docbounds.height-viewsize.height));
    } else if (evt.isControlDown() && code == KeyEvent.VK_A) {  // select all
      myView.selectAll();
    } else if (evt.isControlDown() && code == KeyEvent.VK_X) {  // cut
      myView.cut();
    } else if (evt.isControlDown() && code == KeyEvent.VK_C) {  // copy
      myView.copy();
    } else if (evt.isControlDown() && code == KeyEvent.VK_V) {  // paste
      myView.paste();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Z) {  // undo
      myView.getDocument().undo();
    } else if (evt.isControlDown() && code == KeyEvent.VK_Y) {  // redo
      myView.getDocument().redo();
    }
  }

  // JGoViewListener:
  // Modify the temporary link created when drawing a new link
  // (the default black would not appear against the black background)
  public void viewChanged(JGoViewEvent evt) {
    if (evt.getHint() == JGoViewEvent.INSERTED) {
      if (evt.getJGoObject() instanceof JGoLink) {
        JGoLink link = (JGoLink)evt.getJGoObject();
        link.setPen(JGoPen.make(JGoPen.SOLID, 2, JGoBrush.ColorGray));
        link.setBrush(JGoBrush.gray);
        link.setArrowHeads(false, true);
      }
    } else if (evt.getHint() == JGoViewEvent.BACKGROUND_DOUBLE_CLICKED) {
      insertNode(evt.getPointDocCoords(), (myNodeCounter%2 == 0));
    }
  }

  // JGoDocumentListener:
  // If a link is added to the document, make sure it has a new color, is wider than
  // normal, and has an arrowhead at its "to" end
  public void documentChanged(JGoDocumentEvent evt) {
    if (evt.getHint() == JGoDocumentEvent.INSERTED && (evt.getJGoObject() instanceof JGoLink)) {
      JGoLink link = (JGoLink)evt.getJGoObject();
      link.setBrush(JGoBrush.makeStockBrush(getRandomColor(100)));  // brush fills the arrowhead
      link.setPen(JGoPen.make(JGoPen.SOLID, 2, link.getBrush().getColor()));  // pen controls the line and the outline of the arrowhead
      link.setArrowHeads(false, true);  // have an arrowhead at the "to" end
      link.setCubic(true);  // get a curved path for the link, controlled by Curviness property
      link.calculateStroke();  // replot the points of the stroke so that they are curved
    }
  }

  // Create a JGoBasicNode at a given point with a given shape.
  // Give it a randomly colored brush and pen, and a generated label string.
  // When changing a document, we do so within a transaction so that undo/redo works normally.
  JGoBasicNode insertNode(Point point, boolean rectangular) {
    JGoDocument doc = myView.getDocument();
    doc.startTransaction();
    JGoBasicNode bnode = new JGoBasicNode(Integer.toString(++myNodeCounter));
    bnode.setLabelSpot(JGoObject.Center);
    bnode.setLocation(point);
    if (rectangular)
      bnode.setDrawable(new JGoRectangle());
    bnode.setBrush(JGoBrush.makeStockBrush(getRandomColor(100)));
    bnode.setPen(JGoPen.make(JGoPen.SOLID, 2, getRandomColor(130)));
    bnode.getLabel().setEditable(true);
    doc.addObjectAtTail(bnode);
    doc.endTransaction("inserted node");
    return bnode;
  }

  Color getRandomColor(int base) {
    return new Color(base+(int)((250-base)*Math.random()),
                     base+(int)((250-base)*Math.random()),
                     base+(int)((250-base)*Math.random()));
  }

  private JGoView myView;
  private JTextArea myTextArea;
  private JScrollPane myScrollPane;
  private int myNodeCounter = 0;
}

class UpdateView extends JGoView {
  public UpdateView() {}

  // work-around for rubber-band selection when mouse leaves view;
  // requires JRE 1.4+
  public void onDragDropEnd(DragSourceDropEvent e) {
    if (!isDragEnabled()) return;
    if (isInternalDragDrop() && getState() == MouseStateDragBoxSelection) {
      Point comploc = getLocationOnScreen();
      Point viewpt = new Point(e.getX() - comploc.x, e.getY() - comploc.y);
      Point docpt = viewToDocCoords(viewpt);
      doMouseUp(InputEvent.BUTTON1_MASK, docpt, viewpt);
    } else {
      super.onDragDropEnd(e);
    }
  }
}
